package tv.athena.core.sly

import android.os.Handler
import android.os.Handler.Callback
import android.os.HandlerThread
import android.os.Looper
import android.os.Message
import android.util.Log
import tv.athena.annotation.MessageBinding
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.locks.ReentrantReadWriteLock

/**
 * @author huangfan(kael)
 * @time 2018/5/23 11:31
 */
object SlyBridge : Callback{

    private const val TAG = "SlyBridge"

    private val mIoThread = HandlerThread("SlyBridgeIOThread")
    private var mIoHandler: Handler
    private val mMainHandler: Handler = Handler(Looper.getMainLooper(), this)

    /**
     * 存储绑定的消息与IMessageHandler对应关系
     */
    private val mMessageCenter: ConcurrentHashMap<Class<*>, HashMap<IMessageHandler, IMessage>> = ConcurrentHashMap()

    /**
     * 存储需要进行消息绑定的对象与代理类IMessageHandler的对应关系
     */
    private val mSlyCenter: ConcurrentHashMap<Any, IMessageHandler> = ConcurrentHashMap()

    /**
     * 用公平锁，非公平怕注册和注销线程需要等待更多事件（除非用子线程去注册和注销）
     */
    private val mLock: ReentrantReadWriteLock = ReentrantReadWriteLock(true)

    init {
        mIoThread.start()
        mIoHandler = Handler(mIoThread.looper, this)
    }

    /**
     * 废弃，使用message的callback来进行回调
     * 这里仅仅是重载
     */
    override fun handleMessage(msg: Message): Boolean {
//        mLock.readLock().lock()
//        mMessageCenter[msg.obj.javaClass]?.forEach {
//            it.key.handlerMessage(msg)
//        }
//        mLock.readLock().unlock()
        return true
    }

    /**
     * 发送消息通知MessageBinding的方法进行回调
     */
    fun sendMessage(message: SlyMessage){
        mLock.readLock().lock()
        mMessageCenter[message.javaClass]?.forEach { map ->
//            Log.e("SlyBridge", "-----sendMessage--${message.javaClass.name}---${mMessageCenter[message.javaClass]?.size}")
            val sync: Boolean = map.value.sync
            val mainThread: Boolean = map.value.mainThread
            val delay: Long = map.value.delay
            val msg = Message()
            msg.obj = message
            when {
                sync -> map.key.handlerMessage(msg)
                else -> when {
                    mainThread -> {
                        val m = Message.obtain(mMainHandler, {map.key.handlerMessage(msg)})
                        if(delay > 0) mMainHandler.sendMessageDelayed(m, delay) else mMainHandler.sendMessage(m)
                    }
                    else -> {
                        val m = Message.obtain(mIoHandler, {map.key.handlerMessage(msg)})
                        if(delay > 0) mIoHandler.sendMessageDelayed(m, delay) else mIoHandler.sendMessage(m)
                    }
                }
            }
        }
        mLock.readLock().unlock()
    }

    /**
     * 注册消息订阅
     */
    private fun register(messageHandler: IMessageHandler){
        mLock.writeLock().lock()
        messageHandler.messages().forEach {msg->
            (mMessageCenter[msg.event] ?: hashMapOf()).let {
                it[messageHandler] = msg
                mMessageCenter[msg.event] = it
            }
        }
        mLock.writeLock().unlock()
    }

    /**
     * 取消消息订阅
     */
    private fun unregister(messageHandler: IMessageHandler){
        mLock.writeLock().lock()
        messageHandler.messages().forEach {
            mMessageCenter[it.event]?.remove(messageHandler)
        }
        mLock.writeLock().unlock()
    }

    /**
     * 是否有方法带有注解MessageBinding
     */
    private fun existMessageBinding(observer: Any) : Boolean{
        return observer.javaClass.declaredMethods.any { it.getAnnotation(MessageBinding::class.java) != null }
    }

    /**
     * observer对象订阅消息绑定
     */
    fun subscribe(observer: Any): Boolean{
        val slyBinder = mSlyCenter[observer]
        if(slyBinder != null || !existMessageBinding(observer)){
            return false
        }else{
            try {
                val clazzName = "${observer.javaClass.canonicalName}\$\$SlyBinder"
                val clazz = Class.forName(clazzName)
                val constructor = clazz.getDeclaredConstructor(observer::class.java, SlyBridge::class.java)
                constructor.isAccessible = true
                val instance = constructor.newInstance(observer, this)
                if(instance is IMessageHandler){
                    mSlyCenter[observer] = instance
                    register(instance)
                }else{
                    return false
                }
            }catch(e: Exception){
                Log.e(TAG, "$observer subscribe sly fail, the reason is ${e.message}")
                return false
            }
            return true
        }
    }

    /**
     * observer对象取消消息订阅
     */
    fun unSubscribe(observer: Any): Boolean{
        if(mSlyCenter[observer] != null){
            unregister(mSlyCenter.remove(observer)!!)
            return true
        }
        return false
    }

    /**
     * 存储@MessageBinding信息的实体类
     */
    class IMessage(var event: Class<*>, var mainThread: Boolean = true, var sync: Boolean = false,var delay: Long = 0)

    /**
     * 消息代理类需要实现的接口
     */
    interface IMessageHandler {
        /**
         * 存储@MessageBinding绑定的所有消息
         */
        fun messages(): ArrayList<IMessage>

        /**
         * 对消息的回调的统一处理
         */
        fun handlerMessage(message: Message)
    }
}