/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tuweni.units.bigints;

import com.google.common.base.Preconditions;
import java.math.BigInteger;
import org.apache.tuweni.bytes.Bytes;
import org.apache.tuweni.bytes.MutableBytes;
import org.apache.tuweni.units.bigints.UInt64Value;

public final class UInt64
implements UInt64Value<UInt64> {
    private static final int MAX_CONSTANT = 64;
    private static UInt64[] CONSTANTS = new UInt64[65];
    public static final UInt64 MIN_VALUE;
    public static final UInt64 MAX_VALUE;
    public static final UInt64 ZERO;
    public static final UInt64 ONE;
    private static final BigInteger P_2_64;
    private final long value;

    public static UInt64 valueOf(long value) {
        Preconditions.checkArgument((value >= 0L ? 1 : 0) != 0, (Object)"Argument must be positive");
        return UInt64.create(value);
    }

    public static UInt64 valueOf(BigInteger value) {
        Preconditions.checkArgument((value.signum() >= 0 ? 1 : 0) != 0, (Object)"Argument must be positive");
        Preconditions.checkArgument((value.bitLength() <= 64 ? 1 : 0) != 0, (Object)"Argument is too large to represent a UInt64");
        return UInt64.create(value.longValue());
    }

    public static UInt64 fromBytes(Bytes bytes) {
        Preconditions.checkArgument((bytes.size() <= 8 ? 1 : 0) != 0, (Object)"Argument is greater than 8 bytes");
        return UInt64.create(bytes.toLong());
    }

    public static UInt64 fromHexString(String str) {
        return UInt64.fromBytes(Bytes.fromHexStringLenient((CharSequence)str));
    }

    private static UInt64 create(long value) {
        if (value >= 0L && value <= 64L) {
            return CONSTANTS[(int)value];
        }
        return new UInt64(value);
    }

    private UInt64(long value) {
        this.value = value;
    }

    @Override
    public boolean isZero() {
        return this.value == 0L;
    }

    @Override
    public UInt64 add(UInt64 value) {
        if (value.value == 0L) {
            return this;
        }
        if (this.value == 0L) {
            return value;
        }
        return UInt64.create(this.value + value.value);
    }

    @Override
    public UInt64 add(long value) {
        if (value == 0L) {
            return this;
        }
        return UInt64.create(this.value + value);
    }

    @Override
    public UInt64 addMod(UInt64 value, UInt64 modulus) {
        if (modulus.isZero()) {
            throw new ArithmeticException("addMod with zero modulus");
        }
        return UInt64.create(this.toBigInteger().add(value.toBigInteger()).mod(modulus.toBigInteger()).longValue());
    }

    @Override
    public UInt64 addMod(long value, UInt64 modulus) {
        if (modulus.isZero()) {
            throw new ArithmeticException("addMod with zero modulus");
        }
        return UInt64.create(this.toBigInteger().add(BigInteger.valueOf(value)).mod(modulus.toBigInteger()).longValue());
    }

    @Override
    public UInt64 addMod(long value, long modulus) {
        if (modulus == 0L) {
            throw new ArithmeticException("addMod with zero modulus");
        }
        if (modulus < 0L) {
            throw new ArithmeticException("addMod unsigned with negative modulus");
        }
        return UInt64.create(this.toBigInteger().add(BigInteger.valueOf(value)).mod(BigInteger.valueOf(modulus)).longValue());
    }

    @Override
    public UInt64 subtract(UInt64 value) {
        if (value.isZero()) {
            return this;
        }
        return UInt64.create(this.value - value.value);
    }

    @Override
    public UInt64 subtract(long value) {
        return this.add(-value);
    }

    @Override
    public UInt64 multiply(UInt64 value) {
        if (this.value == 0L || value.value == 0L) {
            return ZERO;
        }
        if (value.value == 1L) {
            return this;
        }
        return UInt64.create(this.value * value.value);
    }

    @Override
    public UInt64 multiply(long value) {
        if (value < 0L) {
            throw new ArithmeticException("multiply unsigned by negative");
        }
        if (value == 0L || this.value == 0L) {
            return ZERO;
        }
        if (value == 1L) {
            return this;
        }
        return UInt64.create(this.value * value);
    }

    @Override
    public UInt64 multiplyMod(UInt64 value, UInt64 modulus) {
        if (modulus.isZero()) {
            throw new ArithmeticException("multiplyMod with zero modulus");
        }
        if (this.value == 0L || value.value == 0L) {
            return ZERO;
        }
        if (value.value == 1L) {
            return this.mod(modulus);
        }
        return UInt64.create(this.toBigInteger().multiply(value.toBigInteger()).mod(modulus.toBigInteger()).longValue());
    }

    @Override
    public UInt64 multiplyMod(long value, UInt64 modulus) {
        if (modulus.isZero()) {
            throw new ArithmeticException("multiplyMod with zero modulus");
        }
        if (value == 0L || this.value == 0L) {
            return ZERO;
        }
        if (value == 1L) {
            return this.mod(modulus);
        }
        if (value < 0L) {
            throw new ArithmeticException("multiplyMod unsigned by negative");
        }
        return UInt64.create(this.toBigInteger().multiply(BigInteger.valueOf(value)).mod(modulus.toBigInteger()).longValue());
    }

    @Override
    public UInt64 multiplyMod(long value, long modulus) {
        if (modulus == 0L) {
            throw new ArithmeticException("multiplyMod with zero modulus");
        }
        if (modulus < 0L) {
            throw new ArithmeticException("multiplyMod unsigned with negative modulus");
        }
        if (value == 0L || this.value == 0L) {
            return ZERO;
        }
        if (value == 1L) {
            return this.mod(modulus);
        }
        if (value < 0L) {
            throw new ArithmeticException("multiplyMod unsigned by negative");
        }
        return UInt64.create(this.toBigInteger().multiply(BigInteger.valueOf(value)).mod(BigInteger.valueOf(modulus)).longValue());
    }

    @Override
    public UInt64 divide(UInt64 value) {
        if (value.value == 0L) {
            throw new ArithmeticException("divide by zero");
        }
        if (value.value == 1L) {
            return this;
        }
        return UInt64.create(this.toBigInteger().divide(value.toBigInteger()).longValue());
    }

    @Override
    public UInt64 divide(long value) {
        if (value == 0L) {
            throw new ArithmeticException("divide by zero");
        }
        if (value < 0L) {
            throw new ArithmeticException("divide unsigned by negative");
        }
        if (value == 1L) {
            return this;
        }
        if (UInt64.isPowerOf2(value)) {
            return this.shiftRight(UInt64.log2(value));
        }
        return UInt64.create(this.toBigInteger().divide(BigInteger.valueOf(value)).longValue());
    }

    @Override
    public UInt64 pow(UInt64 exponent) {
        return UInt64.create(this.toBigInteger().modPow(exponent.toBigInteger(), P_2_64).longValue());
    }

    @Override
    public UInt64 pow(long exponent) {
        return UInt64.create(this.toBigInteger().modPow(BigInteger.valueOf(exponent), P_2_64).longValue());
    }

    @Override
    public UInt64 mod(UInt64 modulus) {
        if (modulus.isZero()) {
            throw new ArithmeticException("mod by zero");
        }
        return UInt64.create(this.toBigInteger().mod(modulus.toBigInteger()).longValue());
    }

    @Override
    public UInt64 mod(long modulus) {
        if (modulus == 0L) {
            throw new ArithmeticException("mod by zero");
        }
        if (modulus < 0L) {
            throw new ArithmeticException("mod by negative");
        }
        return UInt64.create(this.value % modulus);
    }

    public UInt64 and(UInt64 value) {
        if (this.value == 0L || value.value == 0L) {
            return ZERO;
        }
        return UInt64.create(this.value & value.value);
    }

    public UInt64 and(Bytes bytes) {
        Preconditions.checkArgument((bytes.size() <= 8 ? 1 : 0) != 0, (Object)"and with more than 8 bytes");
        if (this.value == 0L) {
            return ZERO;
        }
        long value = bytes.toLong();
        if (value == 0L) {
            return ZERO;
        }
        return UInt64.create(this.value & value);
    }

    public UInt64 or(UInt64 value) {
        return UInt64.create(this.value | value.value);
    }

    public UInt64 or(Bytes bytes) {
        Preconditions.checkArgument((bytes.size() <= 8 ? 1 : 0) != 0, (Object)"or with more than 8 bytes");
        return UInt64.create(this.value | bytes.toLong());
    }

    public UInt64 xor(UInt64 value) {
        return UInt64.create(this.value ^ value.value);
    }

    public UInt64 xor(Bytes bytes) {
        Preconditions.checkArgument((bytes.size() <= 8 ? 1 : 0) != 0, (Object)"xor with more than 8 bytes");
        return UInt64.create(this.value ^ bytes.toLong());
    }

    public UInt64 not() {
        return UInt64.create(this.value ^ 0xFFFFFFFFFFFFFFFFL);
    }

    public UInt64 shiftRight(int distance) {
        if (distance == 0) {
            return this;
        }
        if (distance >= 64) {
            return ZERO;
        }
        return UInt64.create(this.value >>> distance);
    }

    public UInt64 shiftLeft(int distance) {
        if (distance == 0) {
            return this;
        }
        if (distance >= 64) {
            return ZERO;
        }
        return UInt64.create(this.value << distance);
    }

    public boolean equals(Object object) {
        if (object == this) {
            return true;
        }
        if (!(object instanceof UInt64)) {
            return false;
        }
        UInt64 other = (UInt64)object;
        return this.value == other.value;
    }

    public int hashCode() {
        return Long.hashCode(this.value);
    }

    @Override
    public int compareTo(UInt64 other) {
        return Long.compareUnsigned(this.value, other.value);
    }

    @Override
    public boolean fitsInt() {
        return this.value >= 0L && this.value <= Integer.MAX_VALUE;
    }

    @Override
    public int intValue() {
        if (!this.fitsInt()) {
            throw new ArithmeticException("Value does not fit a 4 byte int");
        }
        return (int)this.value;
    }

    @Override
    public boolean fitsLong() {
        return this.value >= 0L;
    }

    @Override
    public long toLong() {
        if (!this.fitsLong()) {
            throw new ArithmeticException("Value does not fit a 8 byte long");
        }
        return this.value;
    }

    public String toString() {
        return this.toBigInteger().toString();
    }

    @Override
    public BigInteger toBigInteger() {
        byte[] mag = new byte[]{(byte)(this.value >>> 56 & 0xFFL), (byte)(this.value >>> 48 & 0xFFL), (byte)(this.value >>> 40 & 0xFFL), (byte)(this.value >>> 32 & 0xFFL), (byte)(this.value >>> 24 & 0xFFL), (byte)(this.value >>> 16 & 0xFFL), (byte)(this.value >>> 8 & 0xFFL), (byte)(this.value & 0xFFL)};
        return new BigInteger(1, mag);
    }

    @Override
    public UInt64 toUInt64() {
        return this;
    }

    @Override
    public Bytes toBytes() {
        MutableBytes bytes = MutableBytes.create((int)8);
        bytes.setLong(0, this.value);
        return bytes;
    }

    @Override
    public Bytes toMinimalBytes() {
        int requiredBytes = 8 - Long.numberOfLeadingZeros(this.value) / 8;
        MutableBytes bytes = MutableBytes.create((int)requiredBytes);
        int j = 0;
        switch (requiredBytes) {
            case 8: {
                bytes.set(j++, (byte)(this.value >>> 56));
            }
            case 7: {
                bytes.set(j++, (byte)(this.value >>> 48 & 0xFFL));
            }
            case 6: {
                bytes.set(j++, (byte)(this.value >>> 40 & 0xFFL));
            }
            case 5: {
                bytes.set(j++, (byte)(this.value >>> 32 & 0xFFL));
            }
            case 4: {
                bytes.set(j++, (byte)(this.value >>> 24 & 0xFFL));
            }
            case 3: {
                bytes.set(j++, (byte)(this.value >>> 16 & 0xFFL));
            }
            case 2: {
                bytes.set(j++, (byte)(this.value >>> 8 & 0xFFL));
            }
            case 1: {
                bytes.set(j, (byte)(this.value & 0xFFL));
            }
        }
        return bytes;
    }

    @Override
    public int numberOfLeadingZeros() {
        return Long.numberOfLeadingZeros(this.value);
    }

    @Override
    public int bitLength() {
        return 64 - Long.numberOfLeadingZeros(this.value);
    }

    private static boolean isPowerOf2(long n) {
        assert (n > 0L);
        return (n & n - 1L) == 0L;
    }

    private static int log2(long v) {
        assert (v > 0L);
        return 63 - Long.numberOfLeadingZeros(v);
    }

    static {
        UInt64.CONSTANTS[0] = new UInt64(0L);
        for (int i = 1; i <= 64; ++i) {
            UInt64.CONSTANTS[i] = new UInt64(i);
        }
        MIN_VALUE = UInt64.valueOf(0L);
        MAX_VALUE = new UInt64(-1L);
        ZERO = UInt64.valueOf(0L);
        ONE = UInt64.valueOf(1L);
        P_2_64 = BigInteger.valueOf(2L).pow(64);
    }
}

