/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tuweni.crypto.sodium;

import com.google.common.base.Preconditions;
import java.util.Arrays;
import java.util.Objects;
import javax.security.auth.Destroyable;
import jnr.ffi.Pointer;
import jnr.ffi.byref.LongLongByReference;
import org.apache.tuweni.bytes.Bytes;
import org.apache.tuweni.crypto.sodium.Allocated;
import org.apache.tuweni.crypto.sodium.Sodium;
import org.apache.tuweni.crypto.sodium.SodiumException;

public final class Signature {
    private Signature() {
    }

    public static Bytes signDetached(Bytes message, SecretKey secretKey) {
        return Bytes.wrap((byte[])Signature.signDetached(message.toArrayUnsafe(), secretKey));
    }

    public static Allocated signDetached(Allocated message, SecretKey secretKey) {
        Preconditions.checkArgument((!secretKey.value.isDestroyed() ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
        Allocated signature = Allocated.allocate(Sodium.crypto_sign_bytes());
        int rc = Sodium.crypto_sign_detached(signature.pointer(), new LongLongByReference(Sodium.crypto_sign_bytes()), message.pointer(), (long)message.length(), secretKey.value.pointer());
        if (rc != 0) {
            throw new SodiumException("crypto_sign_detached: failed with result " + rc);
        }
        return signature;
    }

    public static byte[] signDetached(byte[] message, SecretKey secretKey) {
        Preconditions.checkArgument((!secretKey.value.isDestroyed() ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
        byte[] signature = new byte[(int)Sodium.crypto_sign_bytes()];
        int rc = Sodium.crypto_sign_detached(signature, null, message, (long)message.length, secretKey.value.pointer());
        if (rc != 0) {
            throw new SodiumException("crypto_sign_detached: failed with result " + rc);
        }
        return signature;
    }

    public static boolean verifyDetached(Bytes message, Bytes signature, PublicKey publicKey) {
        return Signature.verifyDetached(message.toArrayUnsafe(), signature.toArrayUnsafe(), publicKey);
    }

    public static boolean verifyDetached(Allocated message, Allocated signature, PublicKey publicKey) {
        int rc = Sodium.crypto_sign_verify_detached(signature.pointer(), message.pointer(), (long)message.length(), publicKey.value.pointer());
        if (rc == -1) {
            return false;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_sign_verify_detached: failed with result " + rc);
        }
        return true;
    }

    public static boolean verifyDetached(byte[] message, byte[] signature, PublicKey publicKey) {
        int rc = Sodium.crypto_sign_verify_detached(signature, message, (long)message.length, publicKey.value.pointer());
        if (rc == -1) {
            return false;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_sign_verify_detached: failed with result " + rc);
        }
        return true;
    }

    public static Bytes sign(Bytes message, SecretKey secretKey) {
        return Bytes.wrap((byte[])Signature.sign(message.toArrayUnsafe(), secretKey));
    }

    public static byte[] sign(byte[] message, SecretKey secretKey) {
        Preconditions.checkArgument((!secretKey.value.isDestroyed() ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
        byte[] signature = new byte[(int)Sodium.crypto_sign_bytes() + message.length];
        int rc = Sodium.crypto_sign(signature, null, message, message.length, secretKey.value.pointer());
        if (rc != 0) {
            throw new SodiumException("crypto_sign: failed with result " + rc);
        }
        return signature;
    }

    public static Bytes verify(Bytes signed, PublicKey publicKey) {
        return Bytes.wrap((byte[])Signature.verify(signed.toArrayUnsafe(), publicKey));
    }

    public static byte[] verify(byte[] signed, PublicKey publicKey) {
        byte[] message = new byte[signed.length];
        LongLongByReference messageLongReference = new LongLongByReference();
        int rc = Sodium.crypto_sign_open(message, messageLongReference, signed, signed.length, publicKey.value.pointer());
        if (rc != 0) {
            throw new SodiumException("crypto_sign_open: failed with result " + rc);
        }
        return Arrays.copyOfRange(message, 0, messageLongReference.intValue());
    }

    public static final class KeyPair {
        private final PublicKey publicKey;
        private final SecretKey secretKey;

        public KeyPair(PublicKey publicKey, SecretKey secretKey) {
            this.publicKey = publicKey;
            this.secretKey = secretKey;
        }

        public static KeyPair forSecretKey(SecretKey secretKey) {
            Preconditions.checkArgument((!secretKey.value.isDestroyed() ? 1 : 0) != 0, (Object)"SecretKey has been destroyed");
            int publicKeyLength = PublicKey.length();
            Pointer publicKey = Sodium.malloc(publicKeyLength);
            try {
                int rc = Sodium.crypto_sign_ed25519_sk_to_pk(publicKey, secretKey.value.pointer());
                if (rc != 0) {
                    throw new SodiumException("crypto_sign_ed25519_sk_to_pk: failed with result " + rc);
                }
                PublicKey pk = new PublicKey(publicKey, publicKeyLength);
                publicKey = null;
                return new KeyPair(pk, secretKey);
            }
            catch (Throwable e) {
                if (publicKey != null) {
                    Sodium.sodium_free(publicKey);
                }
                throw e;
            }
        }

        public static KeyPair random() {
            int publicKeyLength = PublicKey.length();
            Pointer publicKey = Sodium.malloc(publicKeyLength);
            Pointer secretKey = null;
            try {
                int secretKeyLength = SecretKey.length();
                secretKey = Sodium.malloc(secretKeyLength);
                int rc = Sodium.crypto_sign_keypair(publicKey, secretKey);
                if (rc != 0) {
                    throw new SodiumException("crypto_sign_keypair: failed with result " + rc);
                }
                PublicKey pk = new PublicKey(publicKey, publicKeyLength);
                publicKey = null;
                SecretKey sk = new SecretKey(secretKey, secretKeyLength);
                secretKey = null;
                return new KeyPair(pk, sk);
            }
            catch (Throwable e) {
                if (publicKey != null) {
                    Sodium.sodium_free(publicKey);
                }
                if (secretKey != null) {
                    Sodium.sodium_free(secretKey);
                }
                throw e;
            }
        }

        public static KeyPair fromSeed(Seed seed) {
            int publicKeyLength = PublicKey.length();
            Pointer publicKey = Sodium.malloc(publicKeyLength);
            Pointer secretKey = null;
            try {
                int secretKeyLength = SecretKey.length();
                secretKey = Sodium.malloc(secretKeyLength);
                int rc = Sodium.crypto_sign_seed_keypair(publicKey, secretKey, seed.value.pointer());
                if (rc != 0) {
                    throw new SodiumException("crypto_sign_seed_keypair: failed with result " + rc);
                }
                PublicKey pk = new PublicKey(publicKey, publicKeyLength);
                publicKey = null;
                SecretKey sk = new SecretKey(secretKey, secretKeyLength);
                secretKey = null;
                return new KeyPair(pk, sk);
            }
            catch (Throwable e) {
                if (publicKey != null) {
                    Sodium.sodium_free(publicKey);
                }
                if (secretKey != null) {
                    Sodium.sodium_free(secretKey);
                }
                throw e;
            }
        }

        public PublicKey publicKey() {
            return this.publicKey;
        }

        public SecretKey secretKey() {
            return this.secretKey;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof KeyPair)) {
                return false;
            }
            KeyPair other = (KeyPair)obj;
            return this.publicKey.equals(other.publicKey) && this.secretKey.equals(other.secretKey);
        }

        public int hashCode() {
            return Objects.hash(this.publicKey, this.secretKey);
        }
    }

    public static final class Seed {
        private final Allocated value;

        private Seed(Pointer ptr, int length) {
            this.value = new Allocated(ptr, length);
        }

        public static Seed fromBytes(Bytes bytes) {
            return Seed.fromBytes(bytes.toArrayUnsafe());
        }

        public static Seed fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_sign_seedbytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_sign_seedbytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, Seed::new);
        }

        public static int length() {
            long seedbytes = Sodium.crypto_sign_seedbytes();
            if (seedbytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_sign_seedbytes: " + seedbytes + " is too large");
            }
            return (int)seedbytes;
        }

        public static Seed random() {
            return Sodium.randomBytes(Seed.length(), Seed::new);
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof Seed)) {
                return false;
            }
            Seed other = (Seed)obj;
            return other.value.equals(this.value);
        }

        public int hashCode() {
            return Objects.hashCode(this.value);
        }

        public Bytes bytes() {
            return this.value.bytes();
        }

        public byte[] bytesArray() {
            return this.value.bytesArray();
        }
    }

    public static final class SecretKey
    implements Destroyable {
        Allocated value;

        private SecretKey(Pointer ptr, int length) {
            this.value = new Allocated(ptr, length);
        }

        @Override
        public void destroy() {
            this.value.destroy();
        }

        @Override
        public boolean isDestroyed() {
            return this.value.isDestroyed();
        }

        public static SecretKey fromBytes(Bytes bytes) {
            return SecretKey.fromBytes(bytes.toArrayUnsafe());
        }

        public static SecretKey fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_sign_secretkeybytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_sign_secretkeybytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, SecretKey::new);
        }

        public static SecretKey fromSeed(Seed seed) {
            return Sodium.dup(seed.bytes().toArray(), SecretKey::new);
        }

        public static int length() {
            long keybytes = Sodium.crypto_sign_secretkeybytes();
            if (keybytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_sign_secretkeybytes: " + keybytes + " is too large");
            }
            return (int)keybytes;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof SecretKey)) {
                return false;
            }
            SecretKey other = (SecretKey)obj;
            return other.value.equals(this.value);
        }

        public int hashCode() {
            return Objects.hashCode(this.value);
        }

        public Bytes bytes() {
            return this.value.bytes();
        }

        public byte[] bytesArray() {
            return this.value.bytesArray();
        }
    }

    public static final class PublicKey
    implements Destroyable {
        final Allocated value;

        private PublicKey(Pointer ptr, int length) {
            this.value = new Allocated(ptr, length);
        }

        public static PublicKey fromBytes(Bytes bytes) {
            return PublicKey.fromBytes(bytes.toArrayUnsafe());
        }

        public static PublicKey fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_sign_publickeybytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_sign_publickeybytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, PublicKey::new);
        }

        public static int length() {
            long keybytes = Sodium.crypto_sign_publickeybytes();
            if (keybytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_sign_publickeybytes: " + keybytes + " is too large");
            }
            return (int)keybytes;
        }

        public boolean verify(Bytes message, Bytes signature) {
            return Signature.verifyDetached(message, signature, this);
        }

        public boolean verify(Allocated message, Allocated signature) {
            return Signature.verifyDetached(message, signature, this);
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof PublicKey)) {
                return false;
            }
            PublicKey other = (PublicKey)obj;
            return Objects.equals(this.value, other.value);
        }

        public int hashCode() {
            return this.value.hashCode();
        }

        public Bytes bytes() {
            return this.value.bytes();
        }

        public byte[] bytesArray() {
            return this.value.bytesArray();
        }

        @Override
        public void destroy() {
            this.value.destroy();
        }

        @Override
        public boolean isDestroyed() {
            return this.value.isDestroyed();
        }
    }
}

