/************************************************************
  *  * EaseMob CONFIDENTIAL 
  * __________________ 
  * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
  *  
  * NOTICE: All information contained herein is, and remains 
  * the property of EaseMob Technologies.
  * Dissemination of this information or reproduction of this material 
  * is strictly forbidden unless prior written permission is obtained
  * from EaseMob Technologies.
  */
package com.hyphenate.chat;

import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import com.hyphenate.chat.adapter.message.EMAFileMessageBody.EMADownloadStatus;
import com.hyphenate.chat.adapter.message.EMAImageMessageBody;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.util.EMFileHelper;

import java.io.File;

/**
 * \~chinese
 * 图片消息体类。
 * 
 * ```java
 *     EMImageMessageBody body = new EMImageMessageBody(imageFile);
 * ```
 *
 * \~english
 * The image message body class.
 *
 * ```java
 *     EMImageMessageBody body = new EMImageMessageBody(imageFile);
 * ```
 */
public class EMImageMessageBody extends EMFileMessageBody implements Parcelable {
	private static final String TAG = EMImageMessageBody.class.getSimpleName();
	
	private boolean sendOriginalImage = false;
    
     /**
     * \~chinese
     * 用图片文件创建一个图片消息体。
     * 参考：一共有五种图片消息体生成的方式：  
     * 
     *   要求传入的参数是一个图片文件：
     *  - EMImageMessageBody(File imageFile)
     *   要求直接传入一个图片文件的 URI 路径：
     *  - EMImageMessageBody(Uri imageUri)
     *   要求传入一个图片文件和一个缩略图文件：
     *  - EMImageMessageBody(File imageFile, File thumbnailFile)
     *   要求直接传入一个图片和一个缩略图的 URI 路径：
     *  - EMImageMessageBody(Uri imageLocalPath, Uri thumbnailPath)
     *   要求传入一个 `EMAImageMessageBody` 类型的实例：
     *  - EMImageMessageBody(EMAImageMessageBody body)
     *
     * @param imageFile 图片消息体。
     *
     * \~english
     * Creates an image message body with an image file. 
     * 
     * Reference: There are five ways to create an image message body as follows:
	 * 
     *  Creates an image message body with an image file:
     *  - EMImageMessageBody(File imageFile)
     *  Creates an image message body with an image URI:
     *  - EMImageMessageBody(Uri imageUri)
     *  Creates an image message body with an image file and its thumbnail file:
     *  - EMImageMessageBody(File imageFile, File thumbnailFile)
     *  Creates an image message body with an image URI and its thumbnail URI:
     *  - EMImageMessageBody(Uri imageLocalPath, Uri thumbnailPath)
     *  Creates an image message body with an EMAImageMessageBody body:
     *  - EMImageMessageBody(EMAImageMessageBody body)
     * 
     * @param imageFile The image message body.
     */
    public EMImageMessageBody(File imageFile){
    	super(Uri.fromFile(imageFile), EMAMessageBody.EMAMessageBodyType_IMAGE);
    }

    /**
     * \~chinese
     * 传入图片的 URI 创建一个图片消息体。
     * @param imageUri 图片的 URI。
     *
     * \~english
     * Creates an image message body with the image URI.
	 *
     * @param imageUri The image URI.
     */
    public EMImageMessageBody(Uri imageUri){
    	super(imageUri, EMAMessageBody.EMAMessageBodyType_IMAGE);
    	setFileName(EMFileHelper.getInstance().getFilename(imageUri));
    }

	/**
	 * \~chinese
	 * 传入图片附件和缩略图文件创建一个图片消息体。
	 *
	 * @param imageFile 		图片文件。
	 * @param thumbnailFile 	缩略图文件。
	 *
	 * \~english
	 * Creates an image message body with an image and its thumbnail.
	 *
	 * @param imageFile 		The original image file.
	 * @param thumbnailFile		The thumbnail image file.
	 */
    public EMImageMessageBody(File imageFile, File thumbnailFile) {
        super(Uri.fromFile(imageFile), EMAMessageBody.EMAMessageBodyType_IMAGE);
        ((EMAImageMessageBody)emaObject).setThumbnailLocalPath(thumbnailFile == null ?
				"" : Uri.fromFile(thumbnailFile).toString());
		((EMAImageMessageBody)emaObject).setThumbnailFileLength(thumbnailFile != null && thumbnailFile.exists() ? thumbnailFile.length() : 0);
    }

	/**
	 * \~chinese
	 * 传入图片附件和缩略图的 URI 创建一个图片消息体。
	 *
	 * @param imageLocalPath 	图片 URI，即图片的资源标识符。
	 * @param thumbnailPath 	缩略图 URI，即缩略图的资源标识符。
	 *
	 * \~english
	 * Creates an image message body with the URI of an image and its thumbnail.
	 *
	 * @param imageLocalPath 	The original image URI.
	 * @param thumbnailPath		The thumbnail image URI.
	 */
    public EMImageMessageBody(Uri imageLocalPath, Uri thumbnailPath) {
        super(imageLocalPath, EMAMessageBody.EMAMessageBodyType_IMAGE);
        ((EMAImageMessageBody)emaObject).setThumbnailLocalPath(thumbnailPath != null ?
				thumbnailPath.toString() : "");
        setFileName(EMFileHelper.getInstance().getFilename(imageLocalPath));
		((EMAImageMessageBody)emaObject).setThumbnailFileLength(EMFileHelper.getInstance().getFileLength(thumbnailPath));
    }
    
    /**
	 * \~chinese
	 * 基于 `EMAImageMessageBody` 类型的实例创建一个图片消息体。
	 *
	 * @param body	`EMAImageMessageBody` 类型的实例。
	 *
	 * \~english
	 * Creates an image message body with an `EMAImageMessageBody` instance.
	 *
	 * @param body	The `EMAImageMessageBody` instance.
	 */
    public EMImageMessageBody(EMAImageMessageBody body) {
    	super(body);
	}
    
    /**
     * \~chinese
	 * 基于收到的图片消息创建一个图片消息体，包含图片和缩略图。
     * 
     * @param fileName       文件名。
     * @param remoteUrl      图片存放路径。
     * @param thumbnailUrl   缩略图路径。
     *
	 * \~english
     * Creates the image message body from an incoming message, which contains the image file and its thumbnail.
	 *
     * @param fileName       The name of the image file.
     * @param remoteUrl      The image path.
     * @param thumbnailUrl   The thumbnail path.
     * 
     */
    EMImageMessageBody(String fileName, String remoteUrl, String thumbnailUrl) {
    	super((Uri) null, EMAMessageBody.EMAMessageBodyType_IMAGE);
    	emaObject = new EMAImageMessageBody((Uri) null, (Uri) null);
    	((EMAImageMessageBody)emaObject).setDisplayName(fileName);
    	((EMAImageMessageBody)emaObject).setRemotePath(remoteUrl);
    	((EMAImageMessageBody)emaObject).setThumbnailRemotePath(thumbnailUrl);
    }

    /**
     * \~chinese
     * 获取缩略图的服务器地址。
     * 
     * 如果设置了 {@link EMOptions#setAutoDownloadThumbnail(boolean)}，接收消息时 SDK 会自动根据 thumbnailRemotePath 下载缩略图，存储到本地。
     * 
     * @return 缩略图的服务器地址。
     * 
     * \~english
     * Gets the URL of the thumbnail on the server.
     * 
     * If {@link EMOptions#setAutoDownloadThumbnail(boolean)} is set, the thumbnail will be downloaded from the path specified with thumbnailRemotePath to a local path during message reception.
     * 
     * @return The URL of the thumbnail on the server.

     */
    public String getThumbnailUrl() {
		return ((EMAImageMessageBody) emaObject).thumbnailRemotePath();
    }
    
    /**
     * \~chinese
     * 设置缩略图的服务器路径。
     * @param thumbnailUrl	缩略图的服务器路径。
     * 
     * \~english
     * Sets the URL of the thumbnail on the server.
     * @param thumbnailUrl The URL of the thumbnail on the server.
     */
    public void setThumbnailUrl(String thumbnailUrl) {
    	((EMAImageMessageBody)emaObject).setThumbnailRemotePath(thumbnailUrl);
    }

    public String toString() {
        return "image: " + ((EMAImageMessageBody)emaObject).displayName() +
				", localurl: " + ((EMAImageMessageBody)emaObject).getLocalUrl() +
				", remoteurl: " + ((EMAImageMessageBody)emaObject).getRemoteUrl() +
				", thumbnail: " + ((EMAImageMessageBody)emaObject).thumbnailRemotePath();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        //members from FileMessageBody
        dest.writeString(((EMAImageMessageBody)emaObject).displayName());
        dest.writeString(((EMAImageMessageBody)emaObject).getLocalUrl());
        dest.writeString(((EMAImageMessageBody)emaObject).getRemoteUrl());
        dest.writeString(((EMAImageMessageBody)emaObject).thumbnailRemotePath());
        
        dest.writeInt(((EMAImageMessageBody)emaObject).width());
        dest.writeInt(((EMAImageMessageBody)emaObject).height());
        
    }
    
    public static final Parcelable.Creator<EMImageMessageBody> CREATOR
            = new Parcelable.Creator<EMImageMessageBody>() {
        public EMImageMessageBody createFromParcel(Parcel in) {
            return new EMImageMessageBody(in);
        }

        public EMImageMessageBody[] newArray(int size) {
            return new EMImageMessageBody[size];
        }
    };

    private EMImageMessageBody(Parcel in) {
    	super((Uri) null, EMAMessageBody.EMAMessageBodyType_IMAGE);
        //members from FileMessageBody
    	((EMAImageMessageBody)emaObject).setDisplayName(in.readString());
    	((EMAImageMessageBody)emaObject).setLocalPath(in.readString());
    	((EMAImageMessageBody)emaObject).setRemotePath(in.readString());
    	((EMAImageMessageBody)emaObject).setThumbnailRemotePath(in.readString());
		int width = in.readInt();
		int height = in.readInt();
		((EMAImageMessageBody) emaObject).setSize(width, height);
    }

	/**
	 * \~chinese
	 * 设置访问缩略图的密钥。下载缩略图时用户需要提供密钥进行校验。
	 * @param secret 缩略图的访问密钥。
	 *
	 * \~english
	 * Sets the secret to access the thumbnail. A secret is required for verification for thumbnail download.
	 * @param secret The secret to access the thumbnail.
	 */
	public void setThumbnailSecret(String secret) {
    	((EMAImageMessageBody)emaObject).setThumbnailSecretKey(secret);
    }

	/**
	 * \~chinese
	 * 获取缩略图的访问密钥。下载缩略图时用户需要提供密钥进行校验。
	 * @return  缩略图的访问密钥。
	 *
	 * \~english
	 * Gets the secret to access the thumbnail. A secret is required for verification for thumbnail download.
	 * @return  The secret to access the thumbnail.
	 */
    public String getThumbnailSecret() {
    	return ((EMAImageMessageBody)emaObject).thumbnailSecretKey();
    }
    
    /**
     * \~chinese
     * 设置发送图片时，是否发送原图。
     * @param sendOriginalImage	是否发送原图。
     * - （默认）`false`：表示发送缩略图，图片超过 100k 会被压缩。
     * - `true`表示发送原图。
     * 
     * \~english
     * Sets whether to send the original image when sending an image.
     * 
     * @param sendOriginalImage Whether to send the original image. 
     * - (Default)`false`: Send the thumbnail(image with size larger than 100 KB will be compressed);
     * - `true`: Send the original image.
     */
    public void setSendOriginalImage(boolean sendOriginalImage) {
    	this.sendOriginalImage = sendOriginalImage;
    }
    
    /**
     * \~chinese
     * 获取发送图片时，是否发送原图。
     * @return	是否发送原图。
     * - （默认）`false`：表示发送缩略图，图片超过 100 KB 会被压缩。
     * - `true`：表示发送原图。
     * 
     * \~english
     * Gets whether to send the original image.
     * 
     * @return Whether to send the original image. 
     * - (Default)`false`: Send the thumbnail(image with size larger than 100 KB will be compressed);
     * - `true`: Send the original image.
     */
    public boolean isSendOriginalImage(){
    	return sendOriginalImage;
    }
    
    /**
     * \~chinese
     * 获取图片的宽度。
     * 
     * @return 图片的宽度。
     * 
     * \~english
     * Gets the image width.
     * 
     * @return The image width.
     */
    public int getWidth() {
		return ((EMAImageMessageBody)emaObject).width();
	}

    /**
     * \~chinese
     * 获取图片的高度。
     * @return 图片的高度。
     * 
     * \~english
     * Gets the image height.
     * @return The image height.
     */
	public int getHeight() {
		return ((EMAImageMessageBody)emaObject).height();
	}
	
	/**
     * \~chinese
     * 获取图片文件名字。
     * @return 图片文件名字。
     * 
     * \~english
     * Gets the image file name.
     * @return  The image file name.
     */
	public String getFileName() {
		return ((EMAImageMessageBody)emaObject).displayName();
	}
	
	void setSize(int width, int height) {
		((EMAImageMessageBody)emaObject).setSize(width, height);
	}

	/**
	 * \~chinese
	 * 获取图片文件大小。
	 * @return 图片文件大小。
	 *
	 * \~english
	 * Gets the size of the image file.
	 * @return  The size of the image file.
	 */
	public long getFileSize() {
		return ((EMAImageMessageBody)emaObject).fileLength();
	}
	
	/**
     * \~chinese
     * 获取缩略图的本地路径或者字符串形式的资源标识符。
     * @return 缩略图本地路径或者字符串形式的资源标识符。
     *
     * \~english
     * Gets the local path or the URI of the thumbnail as a string.
     * @return  The local path or the URI of the thumbnail as a string.
     */
	public String thumbnailLocalPath() {
		return EMFileHelper.getInstance().formatOutLocalUrl(((EMAImageMessageBody)emaObject).thumbnailLocalPath());
	}

	/**
     * \~chinese
     * 获取缩略图本地资源标识符（路径）。
     * @return 缩略图本地资源标识符（路径）。
     *
     * \~english
     * Gets the local path of the thumbnail as a URI.
     * @return The local path of the thumbnail as a URI.
     */
	public Uri thumbnailLocalUri() {
		return EMFileHelper.getInstance().formatOutUri(((EMAImageMessageBody)emaObject).thumbnailLocalPath());
	}

	/**
     * \~chinese
     * 用字符串形式设置缩略图本地路径。
	 * @param localPath	用字符串形式设置缩略图的本地路径。
     *
     * \~english
     * Sets the local path of the thumbnail as a string.
	 * @param localPath The local path of the thumbnail as a string.
     */
    public void setThumbnailLocalPath(String localPath) {
		((EMAImageMessageBody)emaObject).setThumbnailLocalPath(EMFileHelper.getInstance().formatInUriToString(localPath));
		((EMAImageMessageBody)emaObject).setThumbnailFileLength(EMFileHelper.getInstance().getFileLength(localPath));
    }

	/**
     * \~chinese
     * 用 URI 的格式设置缩略图本地路径。
	 * @param localPath	用 URI 的格式设置缩略图的本地路径。
     *
     * \~english
     * Sets the local path of the thumbnail as a URI.
	 * @param localPath	 The local path of the thumbnail as a URI.
     */
    public void setThumbnailLocalPath(Uri localPath) {
        ((EMAImageMessageBody)emaObject).setThumbnailLocalPath(EMFileHelper.getInstance().formatInUriToString(localPath));
		((EMAImageMessageBody)emaObject).setThumbnailFileLength(EMFileHelper.getInstance().getFileLength(localPath));
    }

	/**
	 * \~chinese
	 * 设置缩略图的尺寸。
	 * @param width 	缩略图的宽度。
	 * @param height 	缩略图的高度。
	 *
	 * \~english
	 * Sets the width and height of the thumbnail image.
	 * @param width		The width of thumbnail.
	 * @param height	The height of thumbnail.
	 */
    public void setThumbnailSize(int width, int height) {
		((EMAImageMessageBody)emaObject).setThumbnailSize(width, height);
	}

	/**
	 * \~chinese
	 * 获取缩略图的下载状态。
	 * @return  缩略图的下载状态。
	 * 
	 * \~english
	 * Gets the download status of the thumbnail.
	 * @return The download status of the thumbnail.
	 */
	public EMDownloadStatus thumbnailDownloadStatus() {
	    EMADownloadStatus _status = ((EMAImageMessageBody)emaObject).thumbnailDownloadStatus();
	    switch (_status) {
	    case DOWNLOADING:
	        return EMDownloadStatus.DOWNLOADING;
	    case SUCCESSED:
	        return EMDownloadStatus.SUCCESSED;
	    case FAILED:
	        return EMDownloadStatus.FAILED;
	    case PENDING:
	        return EMDownloadStatus.PENDING;
	    }
	    return EMDownloadStatus.SUCCESSED;
	}

	/**
	 * \~chinese
	 * 设置本地数据库缩略图的下载状态。
	 * @param status 缩略图的下载状态。
	 *
	 * \~english
	 * Sets the download status of the thumbnail in the local database.
	 * @param status The download status of the thumbnail.
	 */
	public void setThumbnailDownloadStatus(EMDownloadStatus status) {
		((EMAImageMessageBody)emaObject).setThumbnailDownloadStatus(EMADownloadStatus.valueOf(status.name()));
    }

	void setOriginalLocalPath(String originalLocalPath) {
		((EMAImageMessageBody)emaObject).setOriginalLocalPath(originalLocalPath);
	}

	String getOriginalLocalPath() {
		return ((EMAImageMessageBody)emaObject).getOriginalLocalPath();
	}
}
