package com.hyphenate.chat;


import com.hyphenate.chat.adapter.EMAConversation;
import com.hyphenate.chat.adapter.EMAFetchMessageOption;
import com.hyphenate.util.EMLog;

import java.util.List;

/**
 * \~chinese
 * 从服务端查询历史消息的参数配置类。
 *
 * \~english
 * The parameter configuration class for pulling historical messages from the server.
 */
public class EMFetchMessageOption extends EMBase<EMAFetchMessageOption> {

    /**
     * \~chinese
     * 构造函数。
     *
     * \~english
     * The constructor.
     */
    public EMFetchMessageOption() {
        emaObject = new EMAFetchMessageOption();
    }

    /**
     * \~chinese
     * 设置获取的消息是否保存到数据库。
     *
     * @param save 是否保存到数据库：
     *             -`true`：保存到数据库；
     *             -（默认）`false`：不保存到数据库。
     *
     * \~english
     * Sets whether the retrieved messages are saved to the database.
     *
     * @param save Whether to save the retrieved messages to the database:
     *             - `true`: Yes.
     *             - (Default)`false`: No.
     */
    public void setIsSave(boolean save) {
        emaObject.setSave(save);
    }

    /**
     * \~chinese
     * 设置消息的搜索方向。
     *
     * @param direction 消息搜索方向。详见 {@link EMConversation.EMSearchDirection}。
     *                          - （默认）`UP`：按照消息中的时间戳的逆序查询；
     * 							- `DOWN`：按照消息中的时间戳的正序查询；
     *
     * \~english
     * Sets the message search direction.
     *
     * @param direction  The message search direction. See {@link EMConversation.EMSearchDirection}.
     *                   - (Default) `UP`: Messages are retrieved in the descending order of the Unix timestamp included in them.
     *                   - `DOWN`: Messages are retrieved in the ascending order of the Unix timestamp included in them.
     *
     */
    public void setDirection(EMConversation.EMSearchDirection direction) {
        EMAConversation.EMASearchDirection d = direction == EMConversation.EMSearchDirection.UP ? EMAConversation.EMASearchDirection.UP : EMAConversation.EMASearchDirection.DOWN;
        emaObject.setDirection(d);
    }

    /**
     * \~chinese
     * 设置群组会话中的消息发送方。
     *
     * @param from 消息的发送方的用户 ID。
     *
     * \~english
     * Sets the message sender in the group conversation.
     *
     * @param from The user ID of the message sender in the group conversation.
     */
    public void setFrom(String from) {
        emaObject.setFrom(from);
    }

    /**
     * \~chinese
     * 设置要查询的消息类型。
     *
     * @param msgTypes 要查询的消息类型列表， 详见 {@link EMStatisticsManager.EMSearchMessageType}。
     *                 你可以设置多个消息类型。默认值 `null`，表示返回所有类型的消息。
     *
     *  \~english
     *  Sets the types of message for query. 
     * 
     * @param msgTypes The list of message types for query. See {@link EMStatisticsManager.EMSearchMessageType}.
     *                 You can set several message types. The default value is `null`, indicating that all types of messages are retrieved.
     */
    public void setMsgTypes(List<EMMessage.Type> msgTypes) {
        if(msgTypes!=null) {
            for (Object type : msgTypes) {
                if (!(type instanceof EMMessage.Type)) {
                    EMLog.e(getClass().getSimpleName(),"setMsgTypes-->"+"Invalid argument type:" + type.getClass().getName());
                    throw new IllegalArgumentException("Invalid argument type: " + type.getClass().getName());
                }
            }
            emaObject.setMsgTypes(msgTypes);
        }
    }

    /**
     * \~chinese
     * 设置消息查询的起始时间。
     *
     * @param startTime 消息查询的起始时间，Unix 时间戳，单位为毫秒。默认为 `-1`，表示消息查询时会忽略该参数。
     *                  - 若起始时间设置为特定时间点，而结束时间采用默认值 `-1`，则查询起始时间至当前时间的消息。
     *                  - 若起始时间采用默认值 `-1`，而结束时间设置了特定时间，SDK 返回从会话中最早的消息到结束时间点的消息。
     *
     *  \~english
     * Sets the start time for message query. 
     *
     * @param startTime  The start time for message query. The time is a UNIX time stamp in milliseconds. The default value is `-1`, indicating that this parameter is ignored during message query. 
     *                   - If the start time is set to a specific time spot and the end time uses the default value `-1`, the SDK returns messages that are sent and received in the period that is from the start time to the current time.
     *                   - If the start time uses the default value `-1` and the end time is set to a specific time spot, the SDK returns messages that are sent and received in the period that is from the timestamp of the first message to the current time.
     */
    public void setStartTime(long startTime) {
        emaObject.setStartTime(startTime);
    }

    /**
     * \~chinese
     * 设置消息查询的结束时间。
     *
     * @param endTime 消息查询的结束时间，Unix 时间戳，单位为毫秒。若传 `-1`，消息查询时会忽略该参数。
     *                 - 若起始时间设置为特定时间点，而结束时间采用默认值 `-1`，则查询起始时间至当前时间的消息。
     *                 - 若起始时间采用默认值 `-1`，而结束时间设置了特定时间，SDK 返回从会话中最早的消息到结束时间点的消息。
     *  \~english
     * Sets the end time for message query. 
     *
     * @param endTime  The end time for message query. The time is a UNIX time stamp in milliseconds. The default value is `-1`, indicating that this parameter is ignored during message query. 
     *                  - If the start time is set to a specific time spot and the end time uses the default value `-1`, the SDK returns messages that are sent and received in the period that is from the start time to the current time.
     *                  - If the start time uses the default value `-1` and the end time is set to a specific time spot, the SDK returns messages that are sent and received in the period that is from the timestamp of the first message to the current time.
     */
    public void setEndTime(long endTime) {
        emaObject.setEndTime(endTime);
    }
}
