package com.yy.mobile.framework.revenuesdk.gift;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Handler;
import android.support.annotation.Nullable;
import android.util.SparseArray;

import com.yy.mobile.framework.revenuesdk.baseapi.log.RLog;
import com.yy.mobile.framework.revenuesdk.gift.bean.GiftInfo;
import com.yy.mobile.framework.revenuesdk.gift.util.GiftCacheInfoUtil;
import com.yy.mobile.framework.revenuesdk.gift.util.GiftListUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

/**
 * Created by lhf on 2018/7/26.
 * <p>
 * YY.Inc
 */
@SuppressWarnings("javadoc")
public class GiftCache {

    private static final String TAG = "Revenue.GiftCache";

    private static final String SHARED_PRE_TAG = "Revenue_GiftCache";
    private static final String KEY_GIFT_CACHE_INFO_SET = "key_gift_cache_info_set";

    private int mAppId;
    private int defaultUsedChannel;
    private Context mContext;
    private Handler mWorkHandler;
    private SharedPreferences sp;

    /**
     * 礼物缓存信息map
     */
    private Map<String, GiftCacheInfo> giftCacheInfoMap = new HashMap<>();

    /**
     * 缓存礼物列表(根据usedChannel和liveCategoryId区分)
     */
    private Map<String, List<GiftInfo>> cacheGiftMap = new HashMap<>();

    /**
     * 所有渠道的礼物列表
     */
    private SparseArray<GiftInfo> allGiftInfos = new SparseArray<>();

    /**
     * 不同渠道的礼物列表集合
     */
    private SparseArray<SparseArray<GiftInfo>> giftInfoMap = new SparseArray<>();

    GiftCache(int appId, Context context, Handler workHandler, int usedChannel) {
        mAppId = appId;
        mContext = context;
        mWorkHandler = workHandler;
        if (mContext != null) {
            sp = mContext.getSharedPreferences(SHARED_PRE_TAG + appId, Context.MODE_PRIVATE);
        }
        defaultUsedChannel = usedChannel;
    }

    /**
     * 根据channelId和直播品类id获取礼物
     *
     * @param channelId
     * @param liveCategoryId
     * @return
     */
    public synchronized List<GiftInfo> getGiftList(int channelId, int liveCategoryId) {
        String key = String.valueOf(channelId) + String.valueOf(liveCategoryId);
        return cacheGiftMap.get(key);
    }

    public synchronized void updateGiftList(int usedChannel,
                                            int liveCategoryId,
                                            List<GiftInfo> propsList,
                                            String urlPrefix,
                                            String compressData,
                                            String md5Hash) {
        RLog.debug(TAG, "updateGiftList: compressData = %s", compressData);
        GiftCacheInfo cacheInfo =
                new GiftCacheInfo(usedChannel, liveCategoryId, urlPrefix, md5Hash, compressData);
        if (GiftCacheInfoUtil.writeGiftCacheInfoData(mContext, cacheInfo)) {
            // 缓存成功
            String key = String.valueOf(usedChannel) + "_" + String.valueOf(liveCategoryId);
            giftCacheInfoMap.put(key, cacheInfo);
            if (sp != null) {
                Set<String> resource = sp.getStringSet(KEY_GIFT_CACHE_INFO_SET, null);
                if (resource == null) {
                    resource = new HashSet<>();
                }
                // SharedPreferences 的坑这里要浅拷贝一下
                Set<String> modifyData = new HashSet<>(resource);
                modifyData.add(key);
                sp.edit().putStringSet(KEY_GIFT_CACHE_INFO_SET, modifyData).apply();
            }
        }
        updateGiftList(usedChannel, liveCategoryId, propsList);
    }

    private synchronized void updateGiftList(int usedChannel, int liveCategoryId,
                                             List<GiftInfo> propsList) {
        RLog.info(TAG, "updateGiftList: usedChannel = %d, liveCategoryId = %d, propsList Size = %d",
                usedChannel, liveCategoryId, propsList != null ? propsList.size() : 0);
        String key = String.valueOf(usedChannel) + String.valueOf(liveCategoryId);
        cacheGiftMap.put(key, propsList);
        //updateAllGiftList(propsList);
        updateGiftListMap(propsList, usedChannel);
    }

    private synchronized void updateGiftListMap(List<GiftInfo> propsList, int usedChannel) {
        SparseArray<GiftInfo> tempGiftInfos = giftInfoMap.get(usedChannel);
        if (tempGiftInfos == null) {
            tempGiftInfos = new SparseArray<>();
        }
        if (propsList != null && propsList.size() > 0) {

            for (GiftInfo giftInfo : propsList) {
                tempGiftInfos.put(giftInfo.propsId, giftInfo);
            }
        }
        giftInfoMap.put(usedChannel, tempGiftInfos);
    }

    @Deprecated
    private synchronized void updateAllGiftList(@Nullable List<GiftInfo> propsList) {
        if (propsList != null) {
            for (GiftInfo giftInfo : propsList) {
                allGiftInfos.put(giftInfo.propsId, giftInfo);
            }
        }
    }

    /**
     * 获取所有礼物
     *
     * @return
     */
    @Deprecated
    public synchronized List<GiftInfo> getAllChannelGift() {
        List<GiftInfo> allChannelGifts = new ArrayList<>();
        for (int i = 0; i < allGiftInfos.size(); ++i) {
            allChannelGifts.add(allGiftInfos.valueAt(i));
        }
        return allChannelGifts;
    }

    /**
     * 查找礼物
     *
     * @param propsId
     * @return
     */
    public synchronized GiftInfo findGiftById(int propsId) {
        return findGiftById(propsId, defaultUsedChannel);
    }

    /**
     * 查找礼物
     *
     * @param propsId
     * @param usedChannel
     * @return
     */
    public synchronized GiftInfo findGiftById(int propsId, int usedChannel) {
        SparseArray<GiftInfo> giftInfos = giftInfoMap.get(usedChannel);
        if (giftInfos != null) {
            return giftInfos.get(propsId);
        }
        return null;
    }

    /**
     * 初始化缓存中的礼物列表
     */
//    public void initGiftFromFile() {
//        if (mWorkHandler != null) {
//            mWorkHandler.post(() -> {
//                if (sp != null) {
//                    Set<String> cacheInfoSet = sp.getStringSet(KEY_GIFT_CACHE_INFO_SET, null);
//                    if (cacheInfoSet == null) {
//                        RLog.error(TAG, "initGiftFromFile: getCacheInfoSet fail.");
//                        return;
//                    }
//                    Iterator<String> iterator = cacheInfoSet.iterator();
//                    while (iterator.hasNext()) {
//                        String key = iterator.next();
//                        String[] array = key.split("_");
//                        if (array.length >= 2) {
//                            try {
//                                int usedChannel = Integer.valueOf(array[0]);
//                                int liveCategoryId = Integer.valueOf(array[1]);
//                                GiftCacheInfo giftCacheInfo = GiftCacheInfoUtil
//                                .readGiftCacheInfoData(mContext, usedChannel, liveCategoryId);
//                                if (giftCacheInfo != null) {
//                                    giftCacheInfoMap.put(key, giftCacheInfo);
//                                    if (getGiftList(usedChannel, liveCategoryId) == null) {
//                                        List<GiftInfo> propsList =
//                                        GiftListUtil.parserGiftList(giftCacheInfo.compressData,
//                                                usedChannel,
//                                                giftCacheInfo.urlPreFix,
//                                                liveCategoryId);
//                                        updateGiftList(usedChannel, liveCategoryId, propsList);
//                                    }
//                                }
//                            } catch (NumberFormatException e) {
//                                RLog.error(TAG, "initGiftFromFile error.", e);
//                            }
//                        }
//                    }
//                }
//            });
//        }
//    }

    /**
     * 加载礼物失败的情况下，获取缓存的礼物列表
     *
     * @param usedChannel
     * @param liveCategoryId
     */
    public void loadGiftListFromCache(int usedChannel, int liveCategoryId) {
        GiftCacheInfo giftCacheInfo =
                GiftCacheInfoUtil.readGiftCacheInfoData(mContext, usedChannel, liveCategoryId);
        if (giftCacheInfo != null) {
            giftCacheInfoMap
                    .put(String.format(Locale.ENGLISH, "%d_%d", usedChannel, liveCategoryId),
                            giftCacheInfo);
            if (getGiftList(usedChannel, liveCategoryId) == null) {
                List<GiftInfo> propsList = GiftListUtil.parserGiftList(giftCacheInfo.compressData,
                        usedChannel,
                        giftCacheInfo.urlPreFix,
                        liveCategoryId);
                updateGiftList(usedChannel, liveCategoryId, propsList);
            }
        }
    }

    /**
     * 获取上一次查询礼物的MD5
     *
     * @return
     */
    public String getLastMd5Hash(int usedChannel, int liveCategoryId) {
        String key = String.valueOf(usedChannel) + "_" + String.valueOf(liveCategoryId);
        GiftCacheInfo cacheInfo = giftCacheInfoMap.get(key);
        if (cacheInfo != null) {
            return cacheInfo.md5Hash;
        } else {
            return "";
        }
    }

    /**
     * 清除所有礼物缓存信息
     */
    public void clearAllGiftCache() {
        giftCacheInfoMap.clear();
        cacheGiftMap.clear();
        allGiftInfos.clear();
        giftInfoMap.clear();
    }

    /**
     * 清除缓存礼物信息(根据usedChannel和liveCategoryId)
     *
     * @param usedChannel
     * @param liveCategoryId
     */
    public void clearGiftCacheByChannelAndCategoryId(int usedChannel,
                                                                   int liveCategoryId) {
            giftInfoMap.remove(usedChannel);
            String key_cacheInfo =
                    String.valueOf(usedChannel) + "_" + String.valueOf(liveCategoryId);
            giftCacheInfoMap.remove(key_cacheInfo);
            String key = String.valueOf(usedChannel) + String.valueOf(liveCategoryId);
            cacheGiftMap.remove(key);
    }
}
