package com.yy.mobile.framework.revenue.gppay;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.support.annotation.NonNull;
import androidx.annotation.Nullable;

import com.android.billingclient.api.AcknowledgePurchaseParams;
import com.android.billingclient.api.AcknowledgePurchaseResponseListener;
import com.android.billingclient.api.BillingClient;
import com.android.billingclient.api.BillingClientStateListener;
import com.android.billingclient.api.BillingFlowParams;
import com.android.billingclient.api.BillingResult;
import com.android.billingclient.api.ConsumeParams;
import com.android.billingclient.api.Purchase;
import com.android.billingclient.api.PurchaseHistoryRecord;
import com.android.billingclient.api.PurchasesUpdatedListener;
import com.android.billingclient.api.SkuDetails;
import com.android.billingclient.api.SkuDetailsParams;
import com.android.billingclient.api.SkuDetailsResponseListener;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GoogleApiAvailability;
import com.yy.mobile.framework.revenue.gppay.diy.IBillingClientConnectCallback;
import com.yy.mobile.framework.revenuesdk.baseapi.IResult;
import com.yy.mobile.framework.revenuesdk.baseapi.PayCallBackBean;
import com.yy.mobile.framework.revenuesdk.baseapi.PurchaseStatus;
import com.yy.mobile.framework.revenuesdk.baseapi.log.RLog;
import com.yy.mobile.framework.revenuesdk.baseapi.reporter.trace.TraceInfo;
import com.yy.mobile.framework.revenuesdk.baseapi.reporter.trace.TraceReport;
import com.yy.mobile.framework.revenuesdk.baseapi.utils.MD5Utils;
import com.yy.mobile.framework.revenuesdk.baseapi.utils.Singleton;
import com.yy.mobile.framework.revenuesdk.baseapi.utils.ThreadPool;
import com.yy.mobile.framework.revenuesdk.payapi.IAcknowledgeCallback;
import com.yy.mobile.framework.revenuesdk.payapi.IPayCallback;
import com.yy.mobile.framework.revenuesdk.payapi.PayType;
import com.yy.mobile.framework.revenuesdk.payapi.bean.ProductInfo;
import com.yy.mobile.framework.revenuesdk.payapi.bean.PurchaseInfo;
import com.yy.mobile.framework.revenuesdk.payapi.bean.SkuDetailInfo;
import com.yy.mobile.framework.revenuesdk.payapi.payservice.DefaultPayMethod;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * Created by ouyangshaocong on 2018/8/1.
 * https://developer.android.com/google/play/billing/billing_reference
 */
public class PayMethodImpl extends DefaultPayMethod implements PurchasesUpdatedListener {
    private final static int QUERY_RESULT_IS_EMPTY = 301;

    private static final String TAG = "PayMethodImpl";

    // 连接 gp 支付 服务超时时长
    private static final long CONNECT_TIME_OUT = 60 * 1000;
    // 拉起支付的时候切后台，gp服务会中断，并且没有回调，导致一直loading
    private static final long FOREGROUND_TIME_OUT = 15 * 1000;

    private Context application;
    private boolean isPayStatus = false;
    private IPayCallback<PurchaseInfo> payCallback;

    private long requestTime;  // 开始请求时间
    private long responeTime;   // 反应时间

    private Singleton<BillingClient> billingClient = new Singleton<BillingClient>() {
        @Override
        protected BillingClient create() {
            BillingClient billingClient = BillingClient.newBuilder(application)
                            .setListener(PayMethodImpl.this)
                            .enablePendingPurchases()
                            .build();
            return billingClient;
        }
    };

    @Override
    public void onPurchasesUpdated(@androidx.annotation.NonNull BillingResult billingResult,
                                   @androidx.annotation.Nullable List<Purchase> purchases) {
        if (payCallback == null) {
            return;
        }
        int responseCode = billingResult.getResponseCode();
        int result = -1;
        RLog.info(TAG, "onPurchasesUpdated responseCode = %d", responseCode);
        if (responseCode == BillingClient.BillingResponseCode.OK) {
            isPayStatus = false;
            if (purchases != null && purchases.size() > 0) {
                PurchaseInfo info = new PurchaseInfo(purchases.get(0).getOriginalJson(),
                        purchases.get(0).getSignature());

                // 填充缺失的 developerPayload
                try {
                    String payload = PayTokenCache.getPayLoad(application, PayTokenCache.getProductId(info.data));
                    String developerPayload = PayTokenCache.getPayLoad(payload);
                    JSONObject json = new JSONObject(info.data);
                    json.put("developerPayload", developerPayload);

                    // 填充 chOrderId
                    String chOrderId = PayTokenCache.getCHOrderId(payload);
                    if (chOrderId != null && !chOrderId.isEmpty()) {
                        json.put("chOrderId", chOrderId);
                    }

                    String purchaseData = json.toString();
                    info = new PurchaseInfo(purchaseData,
                            purchases.get(0).getSignature());
                } catch (JSONException e) {
                    e.printStackTrace();
                }
                PayTokenCache.savePayToken(application, purchases.get(0).getOrderId(),
                        info);

                RLog.info(TAG,
                        String.format(Locale.ENGLISH,
                                "onPurchasesUpdate -- PurchaseInfo：" +
                                        "data =%s , sign =%s"
                                , info.data, info.signature));
                payCallback.onSuccess(info, null);
                result = 0;
            } else {
                payCallback.onFail(GPInAppBillingStatus.ERROR_PURCHASES_INFO_EMPTY.getCode(),
                        GPInAppBillingStatus.ERROR_PURCHASES_INFO_EMPTY.getMessage(), null);
                result = 1;
            }
        } else {
            isPayStatus = false;

            //TODO 对应的错误处理
            GPInAppBillingStatus status = GPInAppBillingStatus.valueOf(responseCode);
            payCallback.onFail(status.getCode()/*responseCode*/, status.getMessage(), null);
            result = 1;
        }

        /**
         * 拉起支付页面结果
         */
        if (result != -1) {
            try {
                TraceInfo info =
                        TraceReport.getInstance().getCacheInfo(TraceReport.TRACE_FOR_PAY_PAGE);
                info.setCompletetime(System.currentTimeMillis() - info.getApptime());
                info.setResult(result == 0 ? "Success" : "Fail");
                info.setReturncode(String.valueOf(responseCode));
                TraceReport.getInstance().doReport(info);
            } catch (Exception e) {
                RLog.error(TAG, "TraceReport onPurchasesUpdated error.", e);
                e.printStackTrace();
            }
        }
    }

    /**
     * 业务app已经返回前台，通过这个方法回调通知sdk
     * （用于解决"支付页面关闭退出，但是谷歌sdk没有任何回调给业务"）
     */
    @Override
    public void appHasReturnToForegroud() {
        RLog.info(TAG, "appHasReturnToForegroud isPayStatus = " + isPayStatus);
        if (isPayStatus) {
            if (payCallback == null) {
                return;
            }
            Runnable timeoutTask = new Runnable() {
                @Override
                public void run() {
                    RLog.info(TAG, "appHasReturnToForegroud onFail = %d" +
                            GPInAppBillingStatus.PAY_DIALOG_EXIT_WITHOUT_CALLBACK.code);
                    payCallback.onFail(GPInAppBillingStatus.PAY_DIALOG_EXIT_WITHOUT_CALLBACK.code,
                            GPInAppBillingStatus.PAY_DIALOG_EXIT_WITHOUT_CALLBACK.message, null);
                }
            };
            ThreadPool.getDefault().mainThreadIO().postDelay(timeoutTask, FOREGROUND_TIME_OUT);
        }
    }

    @Override
    public boolean hasHangPayJobs(@org.jetbrains.annotations.Nullable Context act,
                                  @org.jetbrains.annotations.Nullable
                                          IResult<List<PurchaseInfo>> iResult) {
        RLog.info(TAG, "---hasHangPayJobs---");
        application = act;

        if (!isReady()) {
            startConnection(new IBillingClientConnectCallback() {
                @Override
                public void onSetupFinished(int responseCode) {
                    // 不能链接到Google Play，包括缺失，无网络，没登录
                    if (responseCode == GPInAppBillingStatus.BILLING_UNAVAILABLE.code) {
                        iResult.onFail(responseCode,
                                GPInAppBillingStatus.BILLING_UNAVAILABLE.message, null);
                    } else {
                        hasHangJobInternal(act, BillingClient.SkuType.INAPP, iResult);
                    }
                }

                @Override
                public void onDisconnected() {
                    iResult.onFail(GPInAppBillingStatus.SERVICE_DISCONNECTED.code,
                            GPInAppBillingStatus.SERVICE_DISCONNECTED.getMessage(), null);
                }

                @Override
                public void onTimeOut() {
                    iResult.onFail(GPInAppBillingStatus.SERVICE_TIMEOUT.code,
                            GPInAppBillingStatus.SERVICE_TIMEOUT.message, null);
                }
            });
        } else {
            hasHangJobInternal(act, BillingClient.SkuType.INAPP, iResult);
        }

        return true;
    }

    @Override
    public boolean hasHangSubscribeJobs
            (@org.jetbrains.annotations.Nullable Context act,
             @org.jetbrains.annotations.Nullable
                     IResult<List<PurchaseInfo>> iResult) {
        RLog.info(TAG, "---hasHangSubscribeJobs---");
        application = act;

        if (!isReady()) {
            startConnection(new IBillingClientConnectCallback() {
                @Override
                public void onSetupFinished(int responseCode) {
                    // 不能链接到Google Play，包括缺失，无网络，没登录
                    if (responseCode ==
                            GPInAppBillingStatus.BILLING_UNAVAILABLE.code) {
                        iResult.onFail(responseCode,
                                GPInAppBillingStatus.BILLING_UNAVAILABLE.message, null);
                    } else {
                        hasHangJobInternal(act, BillingClient.SkuType.SUBS,
                                iResult);
                    }
                }

                @Override
                public void onDisconnected() {
                    iResult.onFail(
                            GPInAppBillingStatus.SERVICE_DISCONNECTED.code,
                            GPInAppBillingStatus.SERVICE_DISCONNECTED
                                    .getMessage(), null);
                }

                @Override
                public void onTimeOut() {
                    iResult.onFail(GPInAppBillingStatus.SERVICE_TIMEOUT.code,
                            GPInAppBillingStatus.SERVICE_TIMEOUT.message, null);
                }
            });
        } else {
            hasHangJobInternal(act, BillingClient.SkuType.SUBS, iResult);
        }

        return true;
    }

    @Override
    public boolean queryHistoryPurchaseBySkuType(@NonNull Context act, String skuType,
                                                 IResult<List<PurchaseInfo>> iResult) {
        if (skuType.equals(BillingClient.SkuType.INAPP)) {
            hasHangPayJobs(act, iResult);
        } else if (skuType.equals(BillingClient.SkuType.SUBS)) {
            hasHangSubscribeJobs(act, iResult);
        } else {
            iResult.onFail(-1, "no support skutype", null);
        }
        return false;
    }

    @Override
    public boolean doHangJob(Context context, String orderId,
                             @NonNull PurchaseInfo
                             info, IResult<String> result) {

        PayTokenCache.removePayToken(context, orderId);

        if (!isReady()) {
            startConnection(new IBillingClientConnectCallback() {
                @Override
                public void onSetupFinished(int responseCode) {
                    // 不能链接到Google Play，包括缺失，无网络，没登录
                    if (responseCode ==
                            GPInAppBillingStatus.BILLING_UNAVAILABLE.code) {
                        result.onFail(responseCode,
                                GPInAppBillingStatus.BILLING_UNAVAILABLE.message, null);
                    } else {
                        doHangJobInternal(info, result);
                    }
                }

                @Override
                public void onDisconnected() {
                    result.onFail(
                            GPInAppBillingStatus.SERVICE_DISCONNECTED.code,
                            GPInAppBillingStatus.SERVICE_DISCONNECTED
                                    .getMessage(), null);
                }

                @Override
                public void onTimeOut() {
                    result.onFail(GPInAppBillingStatus.SERVICE_TIMEOUT.code,
                            GPInAppBillingStatus.SERVICE_TIMEOUT.message, null);
                }
            });
        } else {
            doHangJobInternal(info, result);
        }

        return true;
    }

    /**
     * 谷歌服务是否可用，是否支持谷歌服务
     *
     * @param context
     * @return
     */
    @Override
    public boolean isSupported(Context context) {
        GoogleApiAvailability googleAPI = GoogleApiAvailability.getInstance();
        int ret = googleAPI.isGooglePlayServicesAvailable(context);
        return ret == ConnectionResult.SUCCESS;
    }

    @Override
    public void requestPay(Activity act, long uid,
                           ProductInfo produce,
                           String payload,
                           boolean isSetAccountId,
                           IPayCallback<PurchaseInfo> callback) {
        requestPay(act, uid, produce.productId, payload, isSetAccountId,
                callback);
    }

    @Override
    public void requestPay(final Activity act, final long uid,
                           final String productId,
                           final String payload, boolean isSetAccountId,
                           final IPayCallback<PurchaseInfo> callback) {
        if (callback == null) {
            return;
        }
        requestTime = System.currentTimeMillis();
        this.payCallback = callback;
        this.application = act.getApplication();
        RLog.info(TAG,
                String.format(Locale.ENGLISH,
                        "---requestPay uid = %d,productId = %s, payload = %s ,requestTime = %s ---",
                        uid, productId,
                        payload,
                        requestTime));

//        try {
//            // 本地保存payload
//            final String SP_NAME_PAYLOAD = "SP_NAME_TMP";
//            final String SP_KEY_PAYLOAD = "SP_KEY_PAYLOAD";
//            SharedPreferences sp = act.getSharedPreferences(SP_NAME_PAYLOAD, Context.MODE_PRIVATE);
//            sp.edit().putString(SP_KEY_PAYLOAD, payload).commit();
//        } catch (Exception e) {
//            RLog.error(TAG,
//                    String.format(Locale.ENGLISH,
//                            "---requestPay save error = %s",
//                            e.toString()));
//            e.printStackTrace();
//        }

        if (!isReady()) {
            startConnection(new IBillingClientConnectCallback() {
                @Override
                public void onSetupFinished(int responseCode) {
                    // 不能链接到Google Play，包括缺失，无网络，没登录
                    if (responseCode ==
                            GPInAppBillingStatus.BILLING_UNAVAILABLE.code) {
                        PayCallBackBean payCallBackBean = new PayCallBackBean(null, productId,
                                null, requestTime, null, payload, null,
                                null, PurchaseStatus.PAY_FAIL);
                        callback.onFail(responseCode,
                                GPInAppBillingStatus.BILLING_UNAVAILABLE.message, payCallBackBean);
                        callback.onPayStatus(PurchaseStatus.PAY_FAIL, payCallBackBean);
                    } else {
                        requestPayInternal(act, productId, payload, uid,
                                isSetAccountId);
                    }
                }

                @Override
                public void onDisconnected() {
                    PayCallBackBean payCallBackBean = new PayCallBackBean(null, productId,
                            null, requestTime, null, payload, null,
                            null, PurchaseStatus.PAY_FAIL);
                    callback.onFail(
                            GPInAppBillingStatus.SERVICE_DISCONNECTED.code,
                            GPInAppBillingStatus.SERVICE_DISCONNECTED
                                    .getMessage(), payCallBackBean);
                    callback.onPayStatus(PurchaseStatus.PAY_FAIL, payCallBackBean);
                }

                @Override
                public void onTimeOut() {
                    PayCallBackBean payCallBackBean = new PayCallBackBean(null, productId,
                            null, requestTime, null, payload, null,
                            null, PurchaseStatus.PAY_FAIL);
                    callback.onFail(GPInAppBillingStatus.SERVICE_TIMEOUT.code,
                            GPInAppBillingStatus.SERVICE_TIMEOUT.message, payCallBackBean);
                    callback.onPayStatus(PurchaseStatus.PAY_FAIL, payCallBackBean);
                }
            });
        } else {
            requestPayInternal(act, productId, payload, uid, isSetAccountId);
        }
    }

    @Override
    public void requestSubscription(Activity act, long uid,
                                    String productId, String payload,
                                    boolean isSetAccountId,
                                    IPayCallback<PurchaseInfo> callback) {
        if (callback == null) {
            return;
        }
        this.payCallback = callback;
        this.application = act;
        RLog.info(TAG,
                String.format(Locale.ENGLISH,
                        "---requestSubscription uid = %d,productId = %s, payload = %s---",
                        uid,
                        productId,
                        payload));

        if (!isReady()) {
            startConnection(new IBillingClientConnectCallback() {
                @Override
                public void onSetupFinished(int responseCode) {
                    // 不能链接到Google Play，包括缺失，无网络，没登录
                    if (responseCode ==
                            GPInAppBillingStatus.BILLING_UNAVAILABLE.code) {
                        callback.onFail(responseCode,
                                GPInAppBillingStatus.BILLING_UNAVAILABLE.message, null);
                    } else {
                        requestPaySubscription(act, productId, payload,
                                uid, isSetAccountId);
                    }
                }

                @Override
                public void onDisconnected() {
                    callback.onFail(
                            GPInAppBillingStatus.SERVICE_DISCONNECTED.code,
                            GPInAppBillingStatus.SERVICE_DISCONNECTED
                                    .getMessage(), null);
                }

                @Override
                public void onTimeOut() {
                    callback.onFail(GPInAppBillingStatus.SERVICE_TIMEOUT.code,
                            GPInAppBillingStatus.SERVICE_TIMEOUT.message, null);
                }
            });
        } else {
            requestPaySubscription(act, productId, payload, uid,
                    isSetAccountId);
        }

    }

    @Override
    public void acknowledgePurchase(String purchaseToken, IAcknowledgeCallback callback) {

        AcknowledgePurchaseParams params =
                AcknowledgePurchaseParams.newBuilder()
                        .setPurchaseToken(purchaseToken)
                        .build();
        billingClient.get().acknowledgePurchase(params, new AcknowledgePurchaseResponseListener() {
            @Override
            public void onAcknowledgePurchaseResponse(@androidx.annotation.NonNull BillingResult billingResult) {
                callback.onAcknowledgeResult(billingResult.getResponseCode(), billingResult.getDebugMessage());
            }
        });
    }

    @Override
    public void updateSubscription(Activity act, long uid,
                                   String oldProductId,
                                   String newProductId,
                                   int prorationMode,
                                   String payload,
                                   boolean isSetAccountId,
                                   IPayCallback<PurchaseInfo> callback) {
        if (callback == null) {
            return;
        }
        this.payCallback = callback;
        this.application = act;
        RLog.info(TAG,
                String.format(Locale.ENGLISH,
                        "---updateSubscription uid = %d,oldProductId = %s," +
                                " newProductId = %s,prorationMode = %d,payload = %s---",
                        uid, oldProductId, newProductId, prorationMode,
                        payload));
        if (!isReady()) {
            startConnection(new IBillingClientConnectCallback() {

                @Override
                public void onSetupFinished(int responseCode) {
                    // 不能链接到Google Play，包括缺失，无网络，没登录
                    if (responseCode ==
                            GPInAppBillingStatus.BILLING_UNAVAILABLE.code) {
                        callback.onFail(responseCode,
                                GPInAppBillingStatus.BILLING_UNAVAILABLE.message, null);
                    } else {
                        requestUpdateSubscription(act,
                                newProductId,
                                oldProductId,
                                uid,
                                prorationMode,
                                payload,
                                isSetAccountId);
                    }
                }

                @Override
                public void onDisconnected() {
                    callback.onFail(
                            GPInAppBillingStatus.SERVICE_DISCONNECTED.code,
                            GPInAppBillingStatus.SERVICE_DISCONNECTED
                                    .getMessage(), null);
                }

                @Override
                public void onTimeOut() {
                    callback.onFail(GPInAppBillingStatus.SERVICE_TIMEOUT.code,
                            GPInAppBillingStatus.SERVICE_TIMEOUT.message, null);
                }
            });
        } else {
            requestUpdateSubscription(act, newProductId, oldProductId, uid,
                    prorationMode,
                    payload, isSetAccountId);
        }
    }

    /**
     * 返回用户对每个SKU的最近购买，即使该购买是过期，取消或消耗。
     *
     * @param product
     * @param iResult
     * @return
     */
    @Override
    public boolean queryHistoryPurchaseByProductId(String product,
                                                   final IResult<PurchaseInfo> iResult) {

        if (!isReady()) {
            startConnection(new IBillingClientConnectCallback() {

                @Override
                public void onSetupFinished(int responseCode) {
                    // 不能链接到Google Play，包括缺失，无网络，没登录
                    if (responseCode ==
                            GPInAppBillingStatus.BILLING_UNAVAILABLE.code) {
                        iResult.onFail(responseCode,
                                GPInAppBillingStatus.BILLING_UNAVAILABLE.message, null);
                    } else {
                        queryHistoryPurchaseByProductIdInternal(product, iResult);
                    }
                }

                @Override
                public void onDisconnected() {
                    iResult.onFail(
                            GPInAppBillingStatus.SERVICE_DISCONNECTED.code,
                            GPInAppBillingStatus.SERVICE_DISCONNECTED
                                    .getMessage(), null);
                }

                @Override
                public void onTimeOut() {
                    iResult.onFail(GPInAppBillingStatus.SERVICE_TIMEOUT.code,
                            GPInAppBillingStatus.SERVICE_TIMEOUT.message, null);
                }
            });
        } else {
            queryHistoryPurchaseByProductIdInternal(product, iResult);
        }

        return true;
    }

    private void queryHistoryPurchaseByProductIdInternal(String product,
                                                         final IResult<PurchaseInfo> iResult) {
        billingClient.get()
                .queryPurchaseHistoryAsync(BillingClient.SkuType.INAPP,
                        (response, purchasesList) -> {
                            //onHasHangPayJobInternal(responseCode,purchasesList,iResult);
                            if (response.getResponseCode() == BillingClient.BillingResponseCode.OK &&
                                    purchasesList != null) {
                                for (PurchaseHistoryRecord purchase : purchasesList) {
                                    if (product.equals(purchase.getSku())) {
                                        iResult.onSuccess(new PurchaseInfo(
                                                purchase.getOriginalJson(),
                                                purchase.getSignature()), null);
                                        return;
                                    }
                                }
                            }
                            iResult.onFail(response.getResponseCode(), "query fail!", null);
                        });
    }

    /**
     * 查新GP商品信息
     *
     * @param skusList
     * @param {SkuType.INAPP, SkuType.SUBS}
     * @param iResult
     * @return
     */
    @Override
    public boolean querySkuDetails(Context act, List<String> skusList, String skuType,
                                   IResult<List<SkuDetailInfo>> iResult) {
        if (iResult == null) {
            return false;
        }
        this.application = act;

        if (!isReady()) {
            startConnection(new IBillingClientConnectCallback() {

                @Override
                public void onSetupFinished(int responseCode) {
                    // 不能链接到Google Play，包括缺失，无网络，没登录
                    if (responseCode ==
                            GPInAppBillingStatus.BILLING_UNAVAILABLE.code) {
                        iResult.onFail(responseCode,
                                GPInAppBillingStatus.BILLING_UNAVAILABLE.message, null);
                    } else {
                        querySkuDetailsInternal(skusList, skuType, iResult);
                    }
                }

                @Override
                public void onDisconnected() {
                    iResult.onFail(
                            GPInAppBillingStatus.SERVICE_DISCONNECTED.code,
                            GPInAppBillingStatus.SERVICE_DISCONNECTED
                                    .getMessage(), null);
                }

                @Override
                public void onTimeOut() {
                    iResult.onFail(GPInAppBillingStatus.SERVICE_TIMEOUT.code,
                            GPInAppBillingStatus.SERVICE_TIMEOUT.message, null);
                }
            });
        } else {
            querySkuDetailsInternal(skusList, skuType, iResult);
        }
        return false;
    }

    private void querySkuDetailsInternal(List<String> skusList, String skuType,
                                         IResult<List<SkuDetailInfo>> iResult) {
        SkuDetailsParams skuDetailsParams = getSkuDetailsParams(skusList, skuType);
        List<SkuDetailInfo> skuInfoList = new ArrayList<SkuDetailInfo>();
        billingClient.get()
                .querySkuDetailsAsync(skuDetailsParams, (responseCode, skuDetailsList) -> {
                    if (responseCode.getResponseCode() == BillingClient.BillingResponseCode.OK &&
                            skuDetailsList != null) {
                        for (SkuDetails skuDetails : skuDetailsList) {
                            if (skuDetails != null) {
                                SkuDetailInfo skuDetailInfo = new SkuDetailInfo();
                                skuDetailInfo.description = skuDetails.getDescription();
                                skuDetailInfo.freeTrialPeriod = skuDetails.getFreeTrialPeriod();
                                skuDetailInfo.introductoryPrice = skuDetails.getIntroductoryPrice();
                                skuDetailInfo.introductoryPriceAmountMicros =
                                        skuDetails.getIntroductoryPriceAmountMicros();
                                skuDetailInfo.introductoryPriceCycles =
                                        skuDetails.getIntroductoryPriceCycles();
                                skuDetailInfo.price = skuDetails.getPrice();
                                skuDetailInfo.priceAmountMicros = skuDetails.getPriceAmountMicros();
                                skuDetailInfo.priceCurrencyCode = skuDetails.getPriceCurrencyCode();
                                skuDetailInfo.priceCurrencyCode = skuDetails.getPriceCurrencyCode();
                                skuDetailInfo.sku = skuDetails.getSku();
                                skuDetailInfo.subscriptionPeriod =
                                        skuDetails.getSubscriptionPeriod();
                                skuDetailInfo.title = skuDetails.getTitle();
                                skuDetailInfo.type = skuDetails.getType();
//                                skuDetailInfo.isRewarded = skuDetails.isRewarded();
                                skuInfoList.add(skuDetailInfo);
                            }
                        }
                        iResult.onSuccess(skuInfoList, null);
                    } else {
                        iResult.onFail(responseCode.getResponseCode(), "querySkuDetails fail!", null);
                    }
                });

    }

    private SkuDetailsParams getSkuDetailsParams(List<String> skusList, String skuType) {
        SkuDetailsParams skuDetailsParams = SkuDetailsParams.newBuilder()
                .setSkusList(skusList)
                .setType(skuType)
                .build();
        return skuDetailsParams;
    }

    @Override
    public boolean isPayingStatus() {
        return isPayStatus;
    }

    // private int requestPayInternal(Activity act, String productId, String payload) {
    //     BillingFlowParams flowParams = BillingFlowParams.newBuilder()
    //             .setSku(productId)
    //             .setType(BillingClient.SkuType.INAPP)
    //             .setPayload(payload)
    //             .build();
    //     return billingClient.get().launchBillingFlow(act, flowParams);
    // }

    /**
     * 请求支付
     *
     * @param act
     * @param productId
     * @param payload
     * @param uid            用户ID （业务唯一字段），建议使用UID
     * @param isSetAccountId 是否需要设置SetAccountId的开关（需要业务服务端增加一个这样的开关），以便随时关闭
     * @return
     */
    private int requestPayInternal(Activity act, String productId,
                                   String payload, long uid,
                                   boolean isSetAccountId) {
//        BillingFlowParams flowParams =
//                getBillingFlowParams(productId, payload, uid, isSetAccountId);
//        return billingClient.get().launchBillingFlow(act, flowParams).getResponseCode();

        List<String> skuList = new ArrayList<>();
        skuList.add(productId);

        SkuDetailsParams skuDetailsParams = SkuDetailsParams.newBuilder()
                .setSkusList(skuList)
                .setType(BillingClient.SkuType.INAPP)
                .build();
        billingClient.get().querySkuDetailsAsync(skuDetailsParams,
                new SkuDetailsResponseListener() {

                    @Override
                    public void onSkuDetailsResponse(@androidx.annotation.NonNull BillingResult billingResult, @Nullable List<SkuDetails> skuDetailsList) {
                        if (billingResult.getResponseCode() == BillingClient.BillingResponseCode.OK) {
                            if (skuDetailsList.size() == 0) {
                                RLog.error(TAG, String.format(Locale.ENGLISH,
                                        "onSkuDetailsResponse skuDetails is empty!!!"));
                                if (payCallback != null) {
                                    payCallback.onFail(GPInAppBillingStatus.ERROR_PURCHASES_INFO_EMPTY.getCode(),
                                            GPInAppBillingStatus.ERROR_PURCHASES_INFO_EMPTY.getMessage(), null);
                                }
                                return;
                            }
                            BillingFlowParams flowParams =
                                    getBillingFlowParams(productId, payload,
                                            uid, isSetAccountId,
                                            skuDetailsList.get(0));

                            int billingResponseCode = startPay(act, flowParams, payload, productId);
                            if (billingResponseCode == BillingClient.BillingResponseCode.OK) {
                                // do something you want
                            }
                        } else {

                        }
                    }
                });
        return 0;
    }

    private int startPay(Activity act, BillingFlowParams flowParams, String payload, String productId) {
        try {
            // 本地保存payload
            final String SP_NAME_PAYLOAD = "SP_NAME_TMP";
            final String SP_KEY_PAYLOAD = "SP_KEY_PL_" + productId;
            SharedPreferences sp = act.getSharedPreferences(SP_NAME_PAYLOAD, Context.MODE_PRIVATE);
            sp.edit().putString(SP_KEY_PAYLOAD, payload).commit();

            RLog.info(TAG, "savePay: (" + SP_KEY_PAYLOAD + ")" + payload);
        } catch (Exception e) {
            RLog.error(TAG,
                    String.format(Locale.ENGLISH,
                            "---startPay save error = %s",
                            e.toString()));
            e.printStackTrace();
        }

        int code = billingClient.get().launchBillingFlow(act, flowParams).getResponseCode();

        payCallback.onPayStart();
        isPayStatus = true;


        return code;
    }

    private BillingFlowParams getBillingFlowParams(String productId,
                                                   String payload,
                                                   long uid,
                                                   boolean isSetAccountId,
                                                   SkuDetails skuDetails) {
        // 重要（必须）：谷歌建议通过单向哈希加密方式上传uid，不能以明文方式（请看谷歌开发者文档）
        String MD5Uid = MD5Utils.md5(String.valueOf(uid));
        BillingFlowParams flowParams;
        if (isSetAccountId) {      //需要设置AccountId
            flowParams = BillingFlowParams.newBuilder()
                    .setSkuDetails(skuDetails)
                    .setObfuscatedAccountId(MD5Uid)
                    .build();
        } else {
            flowParams = BillingFlowParams.newBuilder()
                    .setSkuDetails(skuDetails)
                    .build();
        }
        return flowParams;
    }

    //发起订阅请求
    private int requestPaySubscription(Activity act,
                                       String productId,
                                       String payload,
                                       long uid,
                                       boolean isSetAccountId) {
        BillingResult r = billingClient.get()
                .isFeatureSupported(BillingClient.FeatureType.SUBSCRIPTIONS);
        int resCode = r.getResponseCode();
        if (resCode != BillingClient.BillingResponseCode.OK) {
            RLog.info(TAG, String.format(Locale.ENGLISH,
                    "requestPaySubscription %s",
                    r.getDebugMessage()));
            return resCode;
        }

//        BillingFlowParams flowParams =
//                getPaySubscriptionBillingFlowParams(productId, payload, uid,
//                        isSetAccountId);


        List<String> skuList = new ArrayList<>();
        skuList.add(productId);

        SkuDetailsParams skuDetailsParams = SkuDetailsParams.newBuilder()
                .setSkusList(skuList)
                .setType(BillingClient.SkuType.SUBS)
                .build();
        billingClient.get().querySkuDetailsAsync(skuDetailsParams,
                new SkuDetailsResponseListener() {
                    @Override
                    public void onSkuDetailsResponse(@androidx.annotation.NonNull BillingResult billingResult,
                                                     @Nullable List<SkuDetails> skuDetailsList) {
                        if (billingResult.getResponseCode() == BillingClient.BillingResponseCode.OK) {
                            BillingFlowParams flowParams =
                                    getPaySubscriptionBillingFlowParams(productId, payload, uid,
                                            isSetAccountId, skuDetailsList.get(0));

                            int billingResponseCode = startPay(act, flowParams, payload, productId);
                            if (billingResponseCode == BillingClient.BillingResponseCode.OK) {
                                // do something you want
                            }
                        } else {

                        }
                    }
                });

        return 0;
//        return billingClient.get().launchBillingFlow(act, flowParams).getResponseCode();
    }

    private BillingFlowParams getPaySubscriptionBillingFlowParams(String productId,
                                                                  String payload,
                                                                  long uid,
                                                                  boolean isSetAccountId,
                                                                  SkuDetails skuDetails) {
        // 重要（必须）：谷歌建议通过单向哈希加密方式上传uid，不能以明文方式（请看谷歌开发者文档）
        String MD5Uid = MD5Utils.md5(String.valueOf(uid));
        BillingFlowParams flowParams;
        if (isSetAccountId) {      //需要设置AccountId
            flowParams = BillingFlowParams.newBuilder()
                    .setSkuDetails(skuDetails)
                    .setObfuscatedAccountId(MD5Uid)
//                    .setSku(productId)
//                    .setType(BillingClient.SkuType.SUBS)
//                    .setPayload(payload)
//                    .setAccountId(MD5Uid)
                    .build();
        } else {
            flowParams = BillingFlowParams.newBuilder()
                    .setSkuDetails(skuDetails)
//                    .setSku(productId)
//                    .setType(BillingClient.SkuType.SUBS)
//                    .setPayload(payload)
                    .build();
        }
        return flowParams;
    }

    //发起订阅升级
    private int requestUpdateSubscription(Activity act,
                                          String productId,
                                          String oldProductId,
                                          long uid, int prorationMode,
                                          String payload,
                                          boolean isSetAccountId) {
        BillingResult r = billingClient.get()
                .isFeatureSupported(BillingClient.FeatureType.SUBSCRIPTIONS);
        int resCode = r.getResponseCode();
        if (resCode != BillingClient.BillingResponseCode.OK) {
            RLog.info(TAG, String.format(Locale.ENGLISH,
                    "requestUpdateSubscription %s",
                    r.getDebugMessage()));
            return resCode;
        }
        // BillingFlowParams flowParams = BillingFlowParams.newBuilder()
        //         .setSku(productId)
        //         .setOldSku(oldProductId)
        //         .setPayload(payload)
        //         .setReplaceSkusProrationMode(prorationMode)
        //         .build();


//        BillingFlowParams flowParams =
//                getUpdateSubscriptionBillingFlowParams(productId, oldProductId,
//                        payload, uid,
//                        prorationMode, isSetAccountId);
//        return billingClient.get().launchBillingFlow(act, flowParams).getResponseCode();


        List<String> skuList = new ArrayList<>();
        skuList.add(productId);

        SkuDetailsParams skuDetailsParams = SkuDetailsParams.newBuilder()
                .setSkusList(skuList)
                .setType(BillingClient.SkuType.SUBS)
                .build();
        billingClient.get().querySkuDetailsAsync(skuDetailsParams,
                (billingResult, skuDetailsList) -> {
                    if (billingResult.getResponseCode() == BillingClient.BillingResponseCode.OK) {
                        BillingFlowParams flowParams =
                                getUpdateSubscriptionBillingFlowParams(productId,
                                        oldProductId,
                                        payload, uid,
                                        prorationMode,
                                        isSetAccountId,
                                        skuDetailsList.get(0));

                        int billingResponseCode = startPay(act, flowParams, payload, productId);
                        if (billingResponseCode == BillingClient.BillingResponseCode.OK) {
                            // do something you want
                        }
                    } else {

                    }
                });
        return 0;
    }

    private BillingFlowParams getUpdateSubscriptionBillingFlowParams(String productId,
                                                                     String oldProductId,
                                                                     String payload,
                                                                     long uid,
                                                                     int prorationMode,
                                                                     boolean isSetAccountId,
                                                                     SkuDetails skuDetails) {
        // 重要（必须）：谷歌建议通过单向哈希加密方式上传uid，不能以明文方式（请看谷歌开发者文档）
        String MD5Uid = MD5Utils.md5(String.valueOf(uid));

        BillingFlowParams flowParams;
        if (isSetAccountId) {      //需要设置AccountId
            flowParams = BillingFlowParams.newBuilder()
                    .setSkuDetails(skuDetails)
                    .setOldSku(oldProductId, "") //TODO
                    .setReplaceSkusProrationMode(prorationMode)
                    .setObfuscatedAccountId(MD5Uid)
                    .build();
        } else {
            flowParams = BillingFlowParams.newBuilder()
                    .setSkuDetails(skuDetails)
                    .setOldSku(oldProductId, "")   //TODO  purchaseToken
                    .setReplaceSkusProrationMode(prorationMode)
                    .build();
        }
        return flowParams;
    }

    private void doHangJobInternal(PurchaseInfo info, IResult<String> result) {
        try {
            Purchase purchase = new Purchase(info.data, info.signature);

            ConsumeParams consumeParams = ConsumeParams
                    .newBuilder()
                    .setPurchaseToken(purchase.getPurchaseToken())
                    .build();

            billingClient.get()
                    .consumeAsync(consumeParams,
                            (responseCode, purchaseToken) -> {
                                RLog.info(TAG, String.format(Locale.ENGLISH,
                                        "onConsumeResponse responseCode = %d,purchaseToken=%s",
                                        responseCode.getResponseCode(), purchaseToken));
                                if (result != null) {
                                    if (responseCode.getResponseCode() == BillingClient.BillingResponseCode.OK) {
                                        RLog.info(TAG,
                                                String.format(Locale.ENGLISH,
                                                        "onConsumeResponse---onSuccess responseCode = %d, " +
                                                                "purchaseToken=%s",
                                                        responseCode.getResponseCode(),
                                                        purchaseToken));
                                        result.onSuccess(purchaseToken, null);
                                    } else {
                                        RLog.error(TAG,
                                                String.format(Locale.ENGLISH,
                                                        "onConsumeResponse---onFail responseCode = %d, " +
                                                                "purchaseToken=%s",
                                                        responseCode.getResponseCode(),
                                                        purchaseToken));
                                        result.onFail(responseCode.getResponseCode(),
                                                "consume fail", null);
                                    }
                                }
                            });
        } catch (Exception e) {
            RLog.error(TAG, "onPaySuccess errorMsg = " + e.getMessage());
            if (result != null) {
                result.onFail(-1, "consume fail! Exception:" + e.getMessage(), null);
            }
        }
    }

    private boolean hasHangJobInternal(Context context, @BillingClient.SkuType String skuType,
                                       final IResult<List<PurchaseInfo>> iResult) {
        //获取您在应用内购买的所有商品的购买详细信息。 此方法使用Google Play商店应用程序，无需启动网络请求。
        Purchase.PurchasesResult result =
                billingClient.get().queryPurchases(skuType);
        return onHasHangJobInternal(context, result, iResult);
    }

    private boolean onHasHangJobInternal(Context context, Purchase.PurchasesResult result,
                                         IResult<List<PurchaseInfo>> iResult) {
        if (result.getResponseCode() == BillingClient.BillingResponseCode.OK) {
            List<Purchase> list = result.getPurchasesList();
//            List<PurchaseInfo> cacheList = PayTokenCache.readPayToken(context);

            if (list != null && list.size() != 0) {
                List<PurchaseInfo> resultList = new ArrayList<>();
                for (Purchase purchase : list) {
                    String data = purchase.getOriginalJson();

//                    // 填充缺失的 developerPayload
//                    try {
//                        String payload = PayTokenCache.getPayLoad(application, PayTokenCache.getProductId(data));
//                        String developerPayload = PayTokenCache.getPayLoad(payload);
//                        JSONObject json = new JSONObject(data);
//                        json.put("developerPayload", developerPayload);
//                        data = json.toString();
//                    } catch (JSONException e) {
//                        e.printStackTrace();
//                    }

                    String sign = purchase.getSignature();
                    resultList.add(new PurchaseInfo(data, sign));
                }
                RLog.info(TAG, "resultList size: " + resultList.size());
                /**
                 * 本地缓存数据
                 */
//                for (PurchaseInfo purchaseInfo : cacheList) {
//                    resultList.add(purchaseInfo);
//                }
                iResult.onSuccess(resultList, null);

//            } else if (cacheList != null && cacheList.size() != 0) {
//                iResult.onSuccess(cacheList, null);

            } else {
                iResult.onFail(QUERY_RESULT_IS_EMPTY, " List size=0",
                        null);
            }
            return true;
        }
        iResult.onFail(result.getResponseCode(), "query fail!", null);
        return false;
    }

    @Override
    public void clearHangPayJob(Context act, int type, IResult<
            PurchaseInfo> iResult) {
        application = act;

        if (!isReady()) {
            startConnection(new IBillingClientConnectCallback() {
                @Override
                public void onSetupFinished(int responseCode) {

                    // 不能链接到Google Play，包括缺失，无网络，没登录
                    if (responseCode ==
                            GPInAppBillingStatus.BILLING_UNAVAILABLE.code) {
                        iResult.onFail(responseCode,
                                GPInAppBillingStatus.BILLING_UNAVAILABLE.message, null);
                    } else {
                        clearAllHangPayJobInternal(iResult);
                    }
                }

                @Override
                public void onDisconnected() {
                    iResult.onFail(
                            GPInAppBillingStatus.SERVICE_DISCONNECTED.code,
                            GPInAppBillingStatus.SERVICE_DISCONNECTED
                                    .getMessage(), null);
                }

                @Override
                public void onTimeOut() {
                    iResult.onFail(GPInAppBillingStatus.SERVICE_TIMEOUT.code,
                            GPInAppBillingStatus.SERVICE_TIMEOUT.message, null);
                }
            });
        } else {
            clearAllHangPayJobInternal(iResult);
        }
    }

    private void clearAllHangPayJobInternal(IResult<PurchaseInfo> iResult) {
        Purchase.PurchasesResult result =
                billingClient.get().queryPurchases(BillingClient.SkuType.INAPP);
        if (result.getResponseCode() != BillingClient.BillingResponseCode.OK) {
            iResult.onFail(result.getResponseCode(), "BillingResponse not ok", null);
            return;
        }
        List<Purchase> mPurchaseList = result.getPurchasesList();
        if (mPurchaseList == null || mPurchaseList.isEmpty()) {
            iResult.onFail(-1, "no unConsume pay", null);
            return;
        }
        try {
            for (Purchase info : mPurchaseList) {
                Purchase purchase = new Purchase(info.getOriginalJson(),
                        info.getSignature());

                ConsumeParams consumeParams = ConsumeParams
                        .newBuilder()
                        .setPurchaseToken(purchase.getPurchaseToken())
                        .build();

                billingClient.get().consumeAsync(consumeParams,
                        (responseCode, purchaseToken) -> {
                            RLog.info(TAG, String.format(Locale.ENGLISH,
                                    "onConsumeResponse responseCode = %d,purchaseToken=%s",
                                    responseCode.getResponseCode(), purchaseToken));
                            iResult.onSuccess(
                                    new PurchaseInfo(purchase.getOriginalJson(),
                                            purchase.getSignature()), null);
                        });
            }
        } catch (Exception e) {
            RLog.error(TAG, "onPaySuccess errorMsg = " + e.getMessage());
        }
    }

    private boolean isReady() {
        return billingClient.get().isReady();
    }

    private void startConnection(IBillingClientConnectCallback callback) {

        GpConnectTaskInfo gpConnectTaskInfo = new GpConnectTaskInfo(callback);

        billingClient.get()
                .startConnection(new BillingClientStateListener() {
                    @Override
                    public void onBillingSetupFinished(@androidx.annotation.NonNull BillingResult billingResult) {
                        gpConnectTaskInfo.release();
                        callback.onSetupFinished(billingResult.getResponseCode());
                    }

                    @Override
                    public void onBillingServiceDisconnected() {
                        gpConnectTaskInfo.release();
                        callback.onDisconnected();
                    }
                });
    }

    private static class GpConnectTaskInfo {

        IBillingClientConnectCallback callback;

        GpConnectTaskInfo(IBillingClientConnectCallback callback) {
            this.callback = callback;
            ThreadPool.getDefault().mainThreadIO().postDelay(timeoutTask, CONNECT_TIME_OUT);
        }

        Runnable timeoutTask = new Runnable() {
            @Override
            public void run() {
                callback.onTimeOut();
            }
        };

        void release() {
            ThreadPool.getDefault().mainThreadIO().removeCallback(timeoutTask);
        }

    }


    // private String getOrderId(String payload) {
    //     String chOrderId = "";
    //     try {
    //         JSONObject jsonObject = new JSONObject(payload);
    //         chOrderId = jsonObject.optString("chOrderId");
    //     } catch (JSONException e) {
    //         RLog.error(TAG, "getOrderId fail raw str: %s, err msg: %s", payload, e.getMessage());
    //     }
    //     return chOrderId;
    // }


//    private String toGpPayload(long uid, String payload) {
//        String ret = payload;
//        try {
//            JSONObject json = new JSONObject();
//            json.put("uid", uid);
//            json.put("payload", payload);
//            ret = json.toString();
//        } catch (JSONException e) {
//            RLog.error(TAG, "formatPayload fail errMsg:=" + e.getMessage());
//        }
//        return ret;
//    }
//
//    private String toRevenueServerPayload(String payload) {
//        String ret = payload;
//        try {
//            JSONObject json = new JSONObject(payload);
//            JSONObject devPayload = new JSONObject(json.optString("developerPayload", ""));
//            json.put("developerPayload", devPayload.optString("payload"));
//            ret = json.toString();
//        } catch (JSONException e) {
//            RLog.error(TAG, "formatPayload fail errMsg:=" + e.getMessage());
//        }
//        return ret;
//    }
}
