package com.yy.platform.baseservice.task;

import android.os.Bundle;
import android.os.Handler;

import com.yy.platform.baseservice.IRPCChannel;
import com.yy.platform.baseservice.marshal.ProtoPacket;

import java.nio.ByteBuffer;
import java.util.Map;

public class RPCTask extends AbstractTask<RPCTask.ResponseParam> {

    private RequestParam mSrvReq;

    public static final long URI = 10;

    public RPCTask(int reqId, RequestParam requestParam,
                   IRPCChannel.RPCCallback<ResponseParam> callback,
                   Bundle options, Handler scheduler) {
        super(URI, reqId, callback, options, scheduler);

        mSrvReq = requestParam;
    }

    public RPCTask(int reqId, RequestParam requestParam,
                   IRPCChannel.RPCCallbackWithTrace<ResponseParam> callback,
                   Bundle options, Handler scheduler) {
        super(URI, reqId, callback, options, scheduler);

        mSrvReq = requestParam;
    }

    @Override
    public byte[] marshall() {
        mBuffer = pushMarshallable(mSrvReq);
        return super.marshall();
    }

    @Override
    public void unmarshall(byte[] buf) {
        super.unmarshall(buf);
//        mBuffer = ByteBuffer.wrap(buf);
//        mBuffer.order(ByteOrder.LITTLE_ENDIAN);

        mSrvRes = (ResponseParam) popMarshallable(ResponseParam.class);
    }

    public static class RequestParam extends ProtoPacket {
        private String mBusinessContext = "";
        private String mServerName = "";
        private String mFuncName = "";
        private byte[] mRequestData = "".getBytes();
        private String mProtoType = "";
        private Map<String, String> mRouteArgs;
        private Map<String, String> mClientHeaders;
        private String mTraceId = "";

        /**
         * 构造一个上下行请求参数
         *
         * @param serverName  [必填]业务路由配置依据(大类)(合法字符[0-9a-zA-Z])
         * @param funcName    [可选]业务路由配置依据(小类)(合法字符[0-9a-zA-Z])
         * @param requestData [必填]业务包
         */
        public RequestParam(String serverName, String funcName, byte[] requestData) {
            mServerName = serverName == null ? "" : serverName;
            mFuncName = funcName == null ? "" : funcName;
            mRequestData = requestData;
        }

        /**
         * 构造一个上下行请求参数
         *
         * @param context       [可选]上下文字段，原样返回给客户端
         * @param serverName    [必填]业务路由配置依据(大类)(合法字符[0-9a-zA-Z])
         * @param funcName      [可选]业务路由配置依据(小类)(合法字符[0-9a-zA-Z])
         * @param requestData   [必填]业务包
         * @param protoType     [可选]协议类型
         * @param routeArgs     // [可选]路由参数(相当于http请求'?'后面的参数)
         *                      // key: 路由配置管理后台里用到的字段；(合法字符[_0-9a-zA-Z])
         *                      // value: 字段值；(合法字符 URL安全字符集)
         *                      // 例如业务希望能用频道id作为哈希路由的key，可以让客户端填 ”topsid”:”12345678”；
         *                      // 管理后台相应配置”topsid”
         * @param clientHeaders // [可选]客户端信息(相当于http请求的header)
         *                      // key: (合法字符 URL安全字符集)
         *                      // value: (合法字符 URL安全字符集)
         *                      // 接入点会修改
         *                      // 暂不支持相同key
         *                      //sdk---udb   sdk---channel 2个key留作SDK内部用，业务请勿使用!!!!
         * @param traceId       追踪监控用的ID，不填或填空则默认由SDK填入ReuestId
         */
        public RequestParam(String context, String serverName, String funcName,
                            byte[] requestData, String protoType,
                            Map<String, String> routeArgs, Map<String, String> clientHeaders, String traceId) {
            mBusinessContext = (context == null) ? "" : context;
            mServerName = serverName == null ? "" : serverName;
            mFuncName = funcName == null ? "" : funcName;
            mRequestData = requestData;
            mProtoType = protoType == null ? "" : protoType;
            mRouteArgs = routeArgs;
            mClientHeaders = clientHeaders;
            mTraceId = traceId == null ? "" : traceId;
        }

        /**
         * 设置追踪监控用的ID，不填或填空则默认由SDK填入ReuestId，[必填]
         *
         * @param traceId 监控用的标识ID
         */
        public void setTraceId(String traceId) {
            if (traceId == null || (traceId.isEmpty())) return;
            if (mTraceId == null || (mTraceId.isEmpty())) {
                mTraceId = traceId;
            }
        }

        @Override
        public void marshall(ByteBuffer byteBuffer) {
            super.marshall(byteBuffer);
            pushBytes(mBusinessContext.getBytes());
            pushBytes(mServerName.getBytes());
            pushBytes(mFuncName.getBytes());
            pushBytes32(mRequestData);
            pushBytes(mProtoType.getBytes());
            pushMap(mRouteArgs, String.class);
            pushMap(mClientHeaders, String.class);
            pushBytes(mTraceId.getBytes());
        }
    }

    public static class ResponseParam extends ProtoPacket {
        public String mBusinessContext;
        public int mResCode;
        public String mResMsg;
        public String mServerName;
        public String mFuncName;
        public String mProtoType;
        public byte[] mResponseData;
        public Map<String, String> mServerHeaders;
        public String mTraceId;

        @Override
        public void unmarshall(ByteBuffer buf) {
            super.unmarshall(buf);
            mBusinessContext = popString16("UTF-8");
            mResCode = popInt();
            mResMsg = popString16("UTF-8");
            mServerName = popString16("UTF-8");
            mFuncName = popString16("UTF-8");
            mProtoType = popString16("UTF-8");
            mResponseData = popBytes32();
            mServerHeaders = popMap(String.class, String.class);
            mTraceId = popString16("UTF-8");
        }
    }

    @Override
    public void onResponseSuccess(final int requestId, final int sdkResCode) {
        mScheduler.post(new Runnable() {
            @Override
            public void run() {
                if (mCallback != null) {
                    mCallback.onSuccess(requestId, mSrvRes);
                } else if (mCallbackWithTrace != null) {
                    mCallbackWithTrace.onSuccess(requestId, mSrvRes.mTraceId, mSrvRes);
                }
            }
        });
    }

    @Override
    public void onResponseFail(final int requestId, final int sdkResCode) {
        mScheduler.post(new Runnable() {
            @Override
            public void run() {
                if (mCallback != null) {
                    mCallback.onFail(requestId, sdkResCode, mSrvRes.mResCode,
                            new Exception(mSrvRes.mResMsg));
                } else if (mCallbackWithTrace != null) {
                    mCallbackWithTrace.onFail(requestId, mSrvRes.mTraceId, sdkResCode, mSrvRes.mResCode,
                            new Exception(mSrvRes.mResMsg));
                }
            }
        });
    }
}
