package com.yy.platform.baseservice;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;

import com.yy.gslbsdk.DnsResultInfo;
import com.yy.gslbsdk.HttpDnsService;
import com.yy.platform.baseservice.profile.DefaultServiceProfileFactory;
import com.yy.platform.baseservice.profile.LogProfile;
import com.yy.platform.baseservice.profile.ServiceProfileFactory;
import com.yy.platform.baseservice.statis.StatisReporter;
import com.yy.platform.baseservice.storage.ShareStore;
import com.yy.platform.baseservice.utils.LoadLibraryUtils;

import java.util.Locale;

public class YYServiceCore {
    public static String TAG = "YYSDK_S";

    private static void loadLibrary(Context context) {
        if (Build.VERSION.SDK_INT <= 19) {
            LoadLibraryUtils.loadLibrary(context, "gnustl_shared");
        }
//        YYAccessTransJni.nativeLogCallback(0,null);//暂时不依赖它的java代码//解决onload里面使用了YYAccessTransJni这个Class
        LoadLibraryUtils.loadLibrary(context, "YYAccessTransSDK");
        LoadLibraryUtils.loadLibrary(context, "yyservicesdk");
    }

    private static HttpDnsService mGSLB = null;
    private static DnsResultInfo mDnsResult = null;
    private static Channel mServiceChannel = null;
    private static Context mAppContext = null;
    private static ServiceProfileFactory gProfile = null;

    private static Handler mMainHandler = null;


    private static LogProfile.ILog mLogger = new DefaultLogger();

    private static YYServiceCore sInstance = new YYServiceCore();

    private BroadcastReceiverImpl mReceiver = null;

    private YYServiceCore() {
        mReceiver = new BroadcastReceiverImpl();
    }

    /**
     * App初始化SDK通道
     *
     * @param appContext     Application上下文
     * @param appid          接入Service服务申请的AppId
     * @param region         无特殊需求给null即可，特殊场景下需要指定所在区域，取值于{https://zh.wikipedia.org/wiki/ISO_3166-1}中的二位代码
     * @param profileFactory 日志配置和测试环境配置，
     *                       LogProfile不能给null，提供给App层搜集log
     *                       ChannelProfile给null则默认链接正式环境
     * @param listener       通道的连通状态监听器，可及时感知到连接的可用性，
     *                       可根据状态描述采取进一步措施，比如提醒用户检查网络、绑定帐号
     * @return Service通道实例
     */
    public synchronized static IChannel init(Context appContext,
                                             long appid,
                                             String region,
                                             ServiceProfileFactory profileFactory,
                                             IChannelListener.IChannelStatusNotify listener) {
        if (mServiceChannel == null) {
            mAppContext = appContext;
            mMainHandler = new Handler(Looper.getMainLooper());
            ShareStore.INSTANCE.init(mAppContext, appid);
            StatisReporter.instance();

            //获取App初始化的gslb单例
            mGSLB = HttpDnsService.getService();
            //没有就创建service sdk的
            if (mGSLB == null) {
                String locale = Locale.getDefault().getCountry();
                mGSLB = HttpDnsService.getService(mAppContext, "servicesdk", null, "", locale);
            }
            dnsAsync();

            if (profileFactory == null) {
                gProfile = new DefaultServiceProfileFactory(mAppContext);
            } else {
                gProfile = profileFactory;
            }
            int logPrinter = LogProfile.Printer.SdkToAppInstallDir;
            String logPath = "";
            String testIp = "";
            int testPort = 0;
            LogProfile.ILog logger = null;
            boolean isLogCat = false;
            if (gProfile.logProfile() != null) {
                if (gProfile.logProfile().getLog() != null) {
                    logPrinter = LogProfile.Printer.App;
                    logger = gProfile.logProfile().getLog();
                } else if (gProfile.logProfile().logPath() != null && (!gProfile.logProfile().logPath().isEmpty())) {
                    logPrinter = LogProfile.Printer.SdkToAppProvideDir;
                    logPath = gProfile.logProfile().logPath();
                }
                isLogCat = gProfile.logProfile().isLogCat();
            }
            if (logPrinter == LogProfile.Printer.SdkToAppInstallDir) {
                logPath = mAppContext.getFilesDir().getAbsolutePath() + DefaultServiceProfileFactory.sDefLogDir;
            }

            if (gProfile.channelProfile() != null
                    && (gProfile.channelProfile().ipAddress() != null)) {
                testIp = gProfile.channelProfile().ipAddress();
            }
            if (gProfile.channelProfile() != null) {
                testPort = gProfile.channelProfile().ipPort();
            }
            if (logger != null) {
                mLogger = logger;
            }

            //加载库
            loadLibrary(appContext);

            byte[] appName = getAppName(mAppContext).getBytes();
            byte[] appVer = getAppVersion(mAppContext).getBytes();

            byte[] sdkVer = version().getBytes();

            byte[] area = (region == null) ? "".getBytes() : region.getBytes();

            sInstance.mReceiver.checkNetwork(mAppContext);
            int status = sInstance.mReceiver.getNetwork();

            if (logger != null) {
                initLibraryLogger(sInstance, appName, appVer, sdkVer, appid, area, status, isLogCat);
            } else {
                initLibrary(sInstance, appName, appVer, sdkVer, appid, area, status, logPath.getBytes(), isLogCat);
            }

            mServiceChannel = createChannel(appid, status, mLogger);

            mServiceChannel.initReport(appid, new String(appName), new String(appVer), new String(sdkVer));

            mServiceChannel.startConnect(testIp, testPort, listener);

            IntentFilter mFilter = new IntentFilter();
            mFilter.addAction(ConnectivityManager.CONNECTIVITY_ACTION); // 添加接收网络连接状态改变的Action
            mAppContext.registerReceiver(sInstance.mReceiver, mFilter);
        }
        return mServiceChannel;
    }

    private static String getAppName(Context context) {
        try {
            ApplicationInfo info = context.getApplicationInfo();
            String name = context.getResources().getString(info.labelRes);
            return name == null ? "" : name;
        } catch (Throwable e) {
            return "";
        }
    }

    private static String getAppVersion(Context context) {
        try {
            PackageManager manager = context.getPackageManager();
            PackageInfo packageInfo = manager.getPackageInfo(context.getPackageName(), 0);

            return packageInfo.versionName == null ? "" : packageInfo.versionName;
        } catch (Throwable e) {
            return "";
        }
    }

    public static String version() {
        return BuildConfig.BUILD_VERSION;
    }

    //建连接时候就需要HdId 统计
    public byte[] hdid() {
        return ShareStore.INSTANCE.getHdId().getBytes();
    }

    private static void dnsAsync() {
        //启动时候异步解析一次为建立连接做准备
        mDnsResult = mGSLB.getIpsByHostAsync(new String("aplbs.service.yy.com"));
    }

    public String[] dnsResolve(byte[] host) {
        String[] ips = new String[]{};
        if (mDnsResult == null || (mDnsResult.mIps == null) || (mDnsResult.mIps.length == 0)) {
            DnsResultInfo info = mGSLB.getIpsByHost(new String(host), true);
            if (info != null && (info.mIps != null)) {
                ips = info.mIps;
            }

        } else {
            ips = mDnsResult.mIps;
        }
        //为下次建连接做准备
        mDnsResult = mGSLB.getIpsByHostAsync(new String(host));

        return ips;
    }

    public String[] dnsStoreLoad() {
        return ShareStore.INSTANCE.dnsStoreLoad();
    }

    public int dnsStoreFlush(String[] ips) {
        return ShareStore.INSTANCE.dnsStoreFlush(ips);
    }

    private static class DefaultLogger implements LogProfile.ILog {

        @Override
        public void outputLog(String log) {
            Log.i(TAG, log);
        }
    }

    public static void log(final String msg) {
        if (mMainHandler == null) {
            Log.i(TAG, msg);
            return;
        }
        mMainHandler.post(new Runnable() {
            @Override
            public void run() {
                try {
                    mLogger.outputLog(msg);
                } catch (Throwable e) {
                    Log.e(TAG, "ex:" + e.getLocalizedMessage());
                }
            }
        });
    }

    public void nativeLog(final byte[] msg) {
        log(new String(msg));
    }

    /**
     * 提供给其他SDK获取ServiceChannel 实例
     *
     * @return
     */
    public static synchronized IChannel getInstance() {
        return mServiceChannel;
    }

    private static Channel createChannel(long appId, int netStatus,
                                         LogProfile.ILog loger) {
        Channel channel = new Channel(mMainHandler, loger);
        newChannel(channel, netStatus);
        return channel;
    }

    private static int releaseChannel() {
        if (mServiceChannel == null) {
            return -1;
        }
        delChannel(mServiceChannel);
        mServiceChannel = null;
        return 0;
    }

    private static native int initLibrary(Object obj, byte[] appName, byte[] appVer, byte[] sdkVer,
                                          long appid, byte[] region, int status,
                                          byte[] logPath, boolean logconsole);

    private static native int initLibraryLogger(Object obj, byte[] appName, byte[] appVer, byte[] sdkVer,
                                                long appid, byte[] region, int status,
                                                boolean logconsole);

    protected static native long increaseId();

    private static native int newChannel(Channel channel, int status);

    private static native int delChannel(Channel channel);

    private static final class NetStatus {
        public static final int NetworkType_NONE = 0;
        public static final int NetworkType_WIFI = 1;
        public static final int NetworkType_4G = 2;
    }

    public static class BroadcastReceiverImpl extends BroadcastReceiver {
        private int mLastNetworkStatus = -100;

        private int getNetwork() {
            int status = NetStatus.NetworkType_NONE;
            if (mLastNetworkStatus == ConnectivityManager.TYPE_WIFI) {
                status = (NetStatus.NetworkType_WIFI);
            } else if (mLastNetworkStatus == ConnectivityManager.TYPE_MOBILE) {
                status = (NetStatus.NetworkType_4G);
            }
            return status;
        }

        private void checkNetwork(Context context) {
            try {
                ConnectivityManager connectivityManager =
                        (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
                if (connectivityManager != null) {
//NetworkInfo [] networkInfos = connectivityManager.getAllNetworkInfo();
//获取网络的连接情况，断网时会是一个null
                    NetworkInfo activeNetInfo = connectivityManager.getActiveNetworkInfo();
                    if (activeNetInfo != null && activeNetInfo.getState() != null &&
                            activeNetInfo.getState() == NetworkInfo.State.CONNECTED) {
                        YYServiceCore.log("Network value is " + activeNetInfo.getType());
                        if (mLastNetworkStatus == activeNetInfo.getType()) {
                            YYServiceCore.log("Network type not changed:" + mLastNetworkStatus);
                            return;
                        }
                        //判断WIFI网
                        if (activeNetInfo.getType() == ConnectivityManager.TYPE_WIFI) {
                            YYServiceCore.log("Network is TYPE_WIFI");
                            mLastNetworkStatus = ConnectivityManager.TYPE_WIFI;
                            return;
                        } else if (activeNetInfo.getType() == ConnectivityManager.TYPE_MOBILE) {
                            //判断3G网
                            YYServiceCore.log("Network is TYPE_MOBILE");
                            mLastNetworkStatus = ConnectivityManager.TYPE_MOBILE;
                            return;
                        }
                    }
                }
            } catch (Throwable ex) {
                YYServiceCore.log("Network ex:" + ex.getMessage());
            }

            mLastNetworkStatus = -1;
            YYServiceCore.log("Network is invalidate");
        }

        @Override
        public void onReceive(Context context, Intent intent) {
            int previousStatus = mLastNetworkStatus;
            checkNetwork(context);

            if (mServiceChannel != null) {
                if (previousStatus != mLastNetworkStatus) {
                    YYServiceCore.log("Network type changed,previous status:" + previousStatus);
                    if (mLastNetworkStatus == ConnectivityManager.TYPE_WIFI) {
                        mServiceChannel.onNetworkChange(NetStatus.NetworkType_WIFI);
                    } else if (mLastNetworkStatus == ConnectivityManager.TYPE_MOBILE) {
                        mServiceChannel.onNetworkChange(NetStatus.NetworkType_4G);
                    } else {
                        mServiceChannel.onNetworkChange(NetStatus.NetworkType_NONE);
                    }
                }
            }
        }
    }
}
