package com.yy.aomi.analysis.common.util.tree;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.annotation.JSONField;
import com.yy.aomi.common.constant.DateConstant;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

/**
 * @author <a href= "mailto:909074682@yy.com" style="color:##E0E;">zhangzhibin</a>
 * @version V1.0
 * @date 2017年5月18日上午11:48:42
 */
public class TreeNode<T extends TreeIF<T>> {
	
	private static Logger logger = LoggerFactory.getLogger(TreeNode.class);
    private String treeId;
    private Integer ownerId;
    private Integer parentId;
    @JSONField(name="@timestamp")
    private String timeStamp;
    private T msgNode;


    public TreeNode() {
        super();
    }


    public TreeNode(Integer ownerId, Integer parentId, T msgNode) {
        this.ownerId = ownerId;
        this.parentId = parentId;
        this.msgNode = msgNode;
    }


    /**
     * 重建多棵树
     *
     * @param list
     * @return
     * @throws Exception 
     */
    public List<T> restoreMultiTree(List<JSONObject> list, Class<T> targetClz) throws Exception{
        List<T> result = new ArrayList<>();

        List<TreeNode<T>> beanList = jsonToObject(list, targetClz);

        Map<String, List<TreeNode<T>>> map = new HashMap<>();
        for (TreeNode<T> node : beanList) {
            String myTreeId = node.getTreeId();
            List<TreeNode<T>> oneTree = map.get(myTreeId);
            if (oneTree == null) {
                oneTree = new ArrayList<>();
                map.put(myTreeId, oneTree);
            }
            oneTree.add(node);
        }


        for (String key : map.keySet()) {
            List<TreeNode<T>> oneTree = map.get(key);
            T item = buildTree(oneTree);
            if(item!= null){
                result.add(item);
            }
        }


        return result;
    }

    /**
     * 重建一棵树
     *
     * @param list
     * @return
     * @throws Exception 
     */
    public T restore(List<JSONObject> list, Class<T> targetClz) throws Exception {
        List<TreeNode<T>> beanList = jsonToObject(list, targetClz);
        return buildTree(beanList);
    }

    /**
     * json转对象
     *
     * @param list
     * @return
     */
    public List<TreeNode<T>> jsonToObject(List<JSONObject> list, Class<T> targetClz) {
        List<TreeNode<T>> result = new ArrayList<>();
        for (JSONObject json : list) {
            JSONObject newJsonData = (JSONObject)json.clone();
            T msgNode = newJsonData.getJSONObject("msgNode").toJavaObject(targetClz);
            newJsonData.remove("msgNode");
            @SuppressWarnings("unchecked")
            TreeNode<T> treeNode = JSON.parseObject(newJsonData.toString(), this.getClass());
            treeNode.setMsgNode(msgNode);
            result.add(treeNode);
        }
        return result;
    }


    /**
     * 分解一棵树，转成结点json串
     *
     * @param tree
     * @return
     */
    public String[] split(T tree) {
        String treeId = UUID.randomUUID().toString().replaceAll("-", "");
        this.treeId = treeId;
        return splitTreeToJson(tree, treeId);
    }
    

    /**分解一棵树，转成结点json串
     * @param tree
     * @param treeId	自定义树的id，要保证不重复
     * @return
     */
    public String[] split(T tree, String treeId) {
        this.treeId = treeId;
        return splitTreeToJson(tree, treeId);
    }


    /**
     * 分解一棵树，转成对象结点
     *
     * @param tree
     * @return
     */
    public List<TreeNode<T>> splitTree(T tree, String treeId) {
        List<TreeNode<T>> result = new ArrayList<>();
        int lastId = 0;
        int parent = 0;
        TreeNode<T> node = new TreeNode<>(lastId, -1, tree);
        result.add(node);
        List<T> children = tree.getChildren();
        if (children != null) {
            for (T item : children) {
                lastId = encodeItem(item, lastId, parent, 1, result);
            }
        }
        for(TreeNode<T> treeNode : result){
            treeNode.setTreeId(treeId);
        }
        return result;
    }

    /**
     * 分解树，并转成json串
     *
     * @param tree
     * @param treeId
     * @return
     */
    public String[] splitTreeToJson(T tree, String treeId) {
        List<TreeNode<T>> result = splitTree(tree,treeId);
        List<String> jsons = new ArrayList<>(result.size());
        String time = DateFormatUtils.formatUTC(tree.acquireOriginalCreateTime(), DateConstant.UTC_TIME_PATTERN);
        for (TreeNode<T> node : result) {
        	node.setTimeStamp(time);
            List<T> children = node.getMsgNode().getChildren();
            node.getMsgNode().setChildren(null);
            jsons.add(JSONObject.toJSONString(node));
            node.getMsgNode().setChildren(children);
        }
        String[] jsonArray = new String[jsons.size()];
        jsons.toArray(jsonArray);
        return jsonArray;
    }

    /**
     * 分解树，并转成json串
     *
     * @param tree
     * @param treeId
     * @return
     */
    public  List<TreeNode<T>> splitTreeToJava(T tree, String treeId) {
        this.treeId = treeId;
        List<TreeNode<T>> result = splitTree(tree,treeId);
        List<TreeNode<T>> treeNodes = new ArrayList<>(result.size());
        String time = DateFormatUtils.formatUTC(tree.acquireOriginalCreateTime(), DateConstant.UTC_TIME_PATTERN);
        for (TreeNode<T> node : result) {
            node.setTimeStamp(time);
            List<T> children = node.getMsgNode().getChildren();
            node.getMsgNode().setChildren(null);
            treeNodes.add(node);
            node.getMsgNode().setChildren(children);
        }
        return treeNodes;
    }


    /**
     * 对结点编码，让其有父结点ID，自己ID
     *
     * @param item
     * @param id
     * @param parent
     * @param level
     * @param result
     * @return
     */
    public int encodeItem(T item, int id, int parent, int level, List<TreeNode<T>> result) {
        int lastId = id + 1;

        TreeNode<T> node = new TreeNode<T>(lastId, parent, item);
        result.add(node);
        List<T> children = item.getChildren();
        if (children != null) {
            parent = lastId;
            for (T child : children) {
                lastId = encodeItem(child, lastId, parent, level + 1, result);
            }
        }

        return lastId;
    }

    /**
     * 重建树
     *
     * @param list
     * @return
     * @throws Exception 
     */
    public T buildTree(List<TreeNode<T>> list) throws Exception {
        Collections.sort(list, new Comparator<TreeNode<T>>() {

            @Override
            public int compare(TreeNode<T> o1, TreeNode<T> o2) {
                if (o1.getParentId() == o2.getParentId()) {
                    return o1.getOwnerId() - o2.getOwnerId();
                }
                return o1.getParentId() - o2.getParentId();
            }


        });
        Map<Integer, TreeNode<T>> map = new HashMap<>();
        for (TreeNode<T> node : list) {
            map.put(node.getOwnerId(), node);
        }

        T result = null;
        if (list.size() > 0) {
            for (TreeNode<T> node : list) {
                T msgNode = node.getMsgNode();
                Integer parentId = node.getParentId();
                msgNode.setChildren(new ArrayList<>());
                msgNode.setTreeNode(node);
                if (parentId != -1) {
                    TreeNode<T> node2 = map.get(parentId);
                    if(node2 == null){
                    	String errorMsg = "parentId not found parentId="+parentId+" ownId="+node.getOwnerId()+" treeId="+node.getTreeId();
                    	logger.error(errorMsg);
                    	return null;
//                    	throw new Exception(errorMsg);
                    }
                    List<T> children = node2.getMsgNode().getChildren();
                    children.add(msgNode);
                } else {
                    result = msgNode;
                }
            }
        }
        return result;
    }


    /**
     * 打印树
     *
     * @param item
     * @param level
     */
    public void printItem(T item, int level) {
        for (int i = 0; i < level; i++) {
            System.out.print("--");
        }
        System.out.println(item.toString());
        List<T> children = item.getChildren();
        if (children != null) {
            for (T child : children) {
                printItem(child, level + 1);
            }
        }

    }


    @Override
    public String toString() {
        return "TreeNode [treeId=" + treeId + ", ownerId=" + ownerId + ", parantId=" + parentId + ", msgNode=" + msgNode
                + "]";
    }


    public Integer getOwnerId() {
        return ownerId;
    }


    public void setOwnerId(Integer ownerId) {
        this.ownerId = ownerId;
    }


    public Integer getParentId() {
        return parentId;
    }


    public void setParentId(Integer parentId) {
        this.parentId = parentId;
    }


    public String getTreeId() {
        return treeId;
    }


    public void setTreeId(String treeId) {
        this.treeId = treeId;
    }


    public T getMsgNode() {
        return msgNode;
    }


    public void setMsgNode(T msgNode) {
        this.msgNode = msgNode;
    }


    public String getTimeStamp() {
		return timeStamp;
	}


	public void setTimeStamp(String timeStamp) {
		this.timeStamp = timeStamp;
	}



}
