package com.yy.aomi.analysis.common.service;

import com.alibaba.fastjson.JSON;
import com.yy.aomi.analysis.common.constant.AnalysisEsTables;
import com.yy.aomi.analysis.common.constant.ErrorDefinition;
import com.yy.aomi.analysis.common.model.alarm.AlarmMsg;
import com.yy.aomi.analysis.common.model.alarm.AlarmMsgNodeTree;
import com.yy.aomi.analysis.common.model.alarm.ErrorChainNode;
import com.yy.aomi.analysis.common.model.analysis.MonitorAppInfo;
import com.yy.aomi.analysis.common.util.AlarmTreeNodeBuilder;
import com.yy.aomi.analysis.common.util.tree.TreeNode;
import com.yy.aomi.common.model.EventMessage;
import com.yy.aomi.elastic.ESCondition;
import com.yy.aomi.elastic.ElasticSearchImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;


/**
 * 分析告警服务类，负责将组装告警消息并将消息发送到kafka
 * Created by chengaochang on 2017/4/14.
 */
@Service
public class AlarmService {
    private static final Logger logger = LoggerFactory.getLogger(AlarmService.class);

    public final static String PROCESS_ERROR_TAG = "processError";

    public final static String SOURCE_ERROR_TAG = "sourceError";

    @Autowired
    private ElasticSearchImpl elkSearchImpl;

    @Autowired
    private TinyUrlService tinyUrlService;

    @Autowired
    private KafkaTemplate kafkaTemplate;

    @Autowired
    private AlarmMsgFormatService formatService;

    private static final ExecutorService EXECUTOR =  Executors.newFixedThreadPool(5);


    /**
     * 异步执行单进程分析告警
     *
     * @param alarmMsg
     */
    public String asyncSingleProcessorAlarm(AlarmMsg alarmMsg) {
        MonitorAppInfo appInfo = alarmMsg.getApp();
        if (Objects.isNull(appInfo) || Objects.isNull(alarmMsg)) {
            return null;
        }
        AlarmMsgNodeTree tree = AlarmTreeNodeBuilder.buildTreeNodeByAlarm(alarmMsg);
        if (tree == null) {
            return null;
        }
        return alarm(tree, PROCESS_ERROR_TAG);
    }


    /**
     * 异步执行多进程告警
     *
     *  @param tree
     * @return
     */
    public String asyncMultiProcessorAlarm(AlarmMsgNodeTree tree) {
        return alarm(tree, SOURCE_ERROR_TAG);
    }


    /**
     * 消息警告
     * @param tree 告警消息树对象
     * @param analysisType     分析类型
     * @return
     */
    public String alarm(AlarmMsgNodeTree tree, String analysisType) {
        //非空判断
        if (Objects.isNull(tree)
                || Objects.isNull(tree.getRoot())
                || Objects.isNull(tree.getRoot().getAlarmMsg())) {
            return null;
        }

        MessageBuildResult result = buildMessage(tree,analysisType);

        //对于其它错误，暂停发短信告警，后面记得删除
        if(tree.getRoot().getAlarmMsg().getErrorType() != ErrorDefinition.RPC.OTHER_ERROR){
            sendToKafka(result.messages);
        }

        saveEventAnalysis(result.messages.get(0), result.treeNode);

        return result.treeNode.getTreeId();
    }

    /**
     * 构建告警消息及原始数据
     * @param tree
     * @param analysisType
     * @return
     */
    private MessageBuildResult buildMessage(AlarmMsgNodeTree tree,  String analysisType){

        //构建treeNode
        TreeNode<ErrorChainNode> treeNode = AlarmTreeNodeBuilder.buildErrorChainTreeNode(tree);
        //生成url
        String url = null;
        ErrorChainNode node = treeNode.getMsgNode();
        if(!(StringUtils.isEmpty(treeNode.getTreeId()) || StringUtils.isEmpty(node.getBusinessId()) || node.acquireOriginalCreateTime() <= 0)){
            url = tinyUrlService.getErrorChainTinyUrl(node.getBusinessId(),treeNode.getTreeId(),node.acquireOriginalCreateTime());
        }

        List<EventMessage> messageList = formatService.formatSmsMsg(tree,analysisType,url);

        return new MessageBuildResult(treeNode,messageList);
    }


    /**
     * 保存分析事件原始数据到数据库
     *
     * @param message
     * @param treeNode
     */
    private void saveEventAnalysis(EventMessage message, TreeNode<ErrorChainNode> treeNode) {
        try {
            treeNode.getMsgNode().setCharge(message.getCharge());

            String[] chainNodes = treeNode.split(treeNode.getMsgNode(), treeNode.getTreeId());

            for (String info : chainNodes) {
                logger.info("insert error msg={}", info);
            }
            Date reportDate = new Date(treeNode.getMsgNode().getEndTime());
            String index = AnalysisEsTables.ERROR_CHAIN.getTimeIndex(reportDate);
            String esTypeName = AnalysisEsTables.ERROR_CHAIN.getType();
            logger.info("alarm saveAlarmToEs list.size={}", chainNodes.length);
            EXECUTOR.submit(new EsSaveTaskItem(ESCondition.getInsertEntity(index, esTypeName, chainNodes)));

        } catch (Exception e) {
            logger.warn("", e);
        }
    }

    private void sendToKafka(List<EventMessage> messageList) {
        for(EventMessage message : messageList){
            String msg = JSON.toJSONString(message);
            logger.info("sendToKafka msg={} ", msg);
            try {
                // 按抑制key hash到kafka分区
                kafkaTemplate.sendDefault(message.getReduceBy(), msg);
            } catch (Throwable e) {
                logger.error("send kafka inhibitKey=" + message.getReduceBy() + ",error=" + e, e);
            }
        }
    }

    public static class MessageBuildResult {

        TreeNode<ErrorChainNode> treeNode;

        List<EventMessage> messages;

        public MessageBuildResult(TreeNode<ErrorChainNode> treeNode, List<EventMessage> messages) {
            this.treeNode = treeNode;
            this.messages = messages;
        }
    }

    class EsSaveTaskItem implements Callable<ESCondition> {

        private ESCondition params;
        /**
         * 构造方法
         *
         * @param params
         */
        public EsSaveTaskItem(ESCondition params) {
            this.params = params;
        }


        @Override
        public ESCondition call() throws Exception {
            elkSearchImpl.insert(params);
            return null;
        }
    }

}
