/**
 * Copyright (C), 2018-2018, XXX有限公司
 * FileName: AlarmMsgFormatService
 * Author:   chengaochang
 * Date:     18/10/18 上午10:25
 * Description: AlarmMsgFormatService
 * History:
 * <author>          <time>          <version>          <desc>
 * 作者姓名           修改时间           版本号              描述
 */
package com.yy.aomi.analysis.common.service;

import com.yy.aomi.analysis.common.constant.ErrorDefinition;
import com.yy.aomi.analysis.common.dao.mysql.IBusinessSummaryDao;
import com.yy.aomi.analysis.common.model.alarm.*;
import com.yy.aomi.analysis.common.model.analysis.MonitorAppInfo;
import com.yy.aomi.analysis.common.model.entity.mysql.BusinessSummary;
import com.yy.aomi.analysis.common.util.UriSortUtil;
import com.yy.aomi.common.constant.DateConstant;
import com.yy.aomi.common.constant.EventTypeEm;
import com.yy.aomi.common.constant.ModelType;
import com.yy.aomi.common.model.EventMessage;
import com.yy.aomi.common.util.UriFormatUtil;
import com.yy.aomi.sdk.AomiSdk;
import com.yy.aomi.sdk.process.ProcessInfoCollector;
import freemarker.template.Configuration;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.ui.freemarker.FreeMarkerTemplateUtils;

import java.util.*;

/**
 * 〈一句话功能简述〉<br>
 * 〈AlarmMsgFormatService〉
 *
 * @author cgc
 * @create 18/10/18
 * @since 1.0.0
 */
@Service
public class AlarmMsgFormatService {
    private static final Logger logger = LoggerFactory.getLogger(AlarmMsgFormatService.class);

    public static String summaryFileName = "summary.ftl";
//    private String titleFileName = "title.ftl";

    public final static String ALARM_APP = "aomi";

    @Autowired
    Configuration configuration;

    @Autowired
    private IBusinessSummaryDao businessSummaryDao;

    private BusinessSummary defaultBusiness;


    /**
     * 当业务线不存在时，用这个业务信息填充数据
     *
     * @return
     */
    public BusinessSummary getDefaultBusinessSummary() {
        if (defaultBusiness == null) {
            defaultBusiness = new BusinessSummary();
            defaultBusiness.setBusiness("未知业务线");
            defaultBusiness.setCharge("未知大神");
            defaultBusiness.setCategoryId("0");
        }
        return defaultBusiness;
    }

    /**
     * 根据告警消息树节点格式化告警消息
     * @param tree
     * @param analysisType
     * @param url
     * @return
     */
    public List<EventMessage> formatSmsMsg(AlarmMsgNodeTree tree, String analysisType, String url) {
        List<EventMessage> messageList = new ArrayList<>();
        AlarmMsgProcNode treeRootNode = tree.getRoot();
        EventMessage message = formatSmsMsg(treeRootNode, analysisType, url);
        if(message != null){
            String asAppName = getAnalysisProcessName();
            if(StringUtils.isNotEmpty(asAppName)){
                message.getServices().add(asAppName);
            }
            messageList.add(message);
        }
        for (AlarmMsgNodeChain chain : tree.getChildren()) {
            for (AlarmMsgProcNode node : chain) {
                message = formatSmsMsg(node, analysisType, url);
                if(message != null){
                    messageList.add(message);
                }
            }
        }
        return messageList;
    }


    /**
     * 格式化短信的格式
     *
     * @param node
     * @return
     * @throws Exception
     */
    public EventMessage formatSmsMsg(AlarmMsgProcNode node, String analysisType, String url) {
        AlarmMsg alarmMsg = node.alarmMsg;
        MonitorAppInfo appInfo = node.getApp();

        if (ErrorDefinition.isError(alarmMsg.getErrorType())) {
            String content = "";
            String businessName = "";
            String chargeName = "";
            String percent = "0%";

            BusinessSummary businessSummary = null;
            //当业务线id不为Null，也不为默认的0时，增加跳转链接
            if (StringUtils.isNotEmpty(appInfo.getBusinessid())) {
                try {
                    businessSummary = businessSummaryDao.getByBusinessId(appInfo.getBusinessid());
                } catch (Exception e) {
                    logger.error("", e);
                }
            }

            if (businessSummary == null) {
                businessSummary = getDefaultBusinessSummary();
            }
            businessName = businessSummary.getBusiness();
            chargeName = businessSummary.getCharge();

            if (alarmMsg.getTotalCount() > 0) {
                percent = String.format("%.2f", alarmMsg.getErrorCount() * 100 / (alarmMsg.getTotalCount() * 1.0)) + "%";
            }

            Map<String, Object> simpleMap = new HashMap<>();
            simpleMap.put("data", alarmMsg);
            simpleMap.put("procInfo", appInfo.defaultHost() + "(" + appInfo.getClientname() + ")");
            simpleMap.put("rcpProcInfo", getRpcProcInfo(alarmMsg));
            simpleMap.put("business", businessName);
            simpleMap.put("charge", chargeName);
            simpleMap.put("errorRate", percent);
            simpleMap.put("link",url);
            simpleMap.put("top1", topUriSimpleFormat(alarmMsg));
            // 错误发生时间,精确到秒 2018-08-01 09:32:58
            simpleMap.put("errorTime", DateFormatUtils.format(appInfo.getTime(), DateConstant.yyyyMMddHHmmss));

            try {
                content = FreeMarkerTemplateUtils.processTemplateIntoString(configuration.getTemplate(summaryFileName), simpleMap);
            } catch (Exception e) {
                logger.error("freeMarkerTemplate parse error!", e);
            }
            logger.info("aomi-sms-msg-content=\n {}", content);


            //组装发送应用列表
            List<String> serviceNames = new ArrayList<>();
            serviceNames.add(appInfo.getClientname());
            //组装发送事件对象
            String inhibitKey = buildReduceKey(node, analysisType);
            //组装并发送告警
            EventMessage message = new EventMessage();
            message.setAppId(ALARM_APP);
            message.setCreateTime(DateFormatUtils.format(new Date(), DateConstant.yyyyMMddHHmmss));
            message.setAlarmTime(DateFormatUtils.format(appInfo.getTime(), DateConstant.yyyyMMddHHmmss));
            message.setType(EventTypeEm.ALARM);
            message.setAlarmLevel(node.alarmMsg.getLevel());
            message.setReduceBy(inhibitKey);
            message.setServices(serviceNames);
            message.setUrl(url);
            //格式化消息
            message.setData(content);
            message.setCharge(chargeName);
            message.setIp(appInfo.defaultIp());
            return message;
        }
        return null;
    }

    public static String topUriSimpleFormat(AlarmMsg alarmMsg) {
        if (CollectionUtils.isEmpty(alarmMsg.getUriList())) {
            return "";
        }
        if (ModelType.isQueue(alarmMsg.getModelType())) {
            UriSortUtil.sortUriAlarmByThreadUseRate(alarmMsg.getUriList());
        } else {
            UriSortUtil.sortUriAlarmByPercent(alarmMsg.getUriList());
        }
        UriInvokeInfo top1 = alarmMsg.getUriList().get(0);
        StringBuffer topUri = new StringBuffer();
        topUri.append(UriFormatUtil.formatYypUri(top1.getUri(), false));
        logger.info("topUri={} \n  invokeList={}", topUri, alarmMsg.getUriList());
        if (alarmMsg instanceof RpcAlarm) {
            RpcAlarm alarm = (RpcAlarm) alarmMsg;
            topUri.append("(")
                    .append(top1.getRpcHost())
                    .append("(")
                    .append(alarm.getRpcSrvName())
                    .append(")")
                    .append(")");
        }

        return topUri.toString();
    }

    public static String getRpcProcInfo(AlarmMsg alarmMsg) {
        StringBuffer rpcHost = new StringBuffer();
        if (alarmMsg instanceof RpcAlarm) {
            RpcAlarm alarm = (RpcAlarm) alarmMsg;
            rpcHost.append(alarm.getRpcHost())
                    .append("(")
                    .append(alarm.getRpcSrvName())
                    .append(")");
        }
        return rpcHost.toString();
    }

    /**
     * 创建reduceKey
     *
     * @param msgTreeNode
     * @param analysisType
     * @return
     */
    private String buildReduceKey(AlarmMsgProcNode msgTreeNode, String analysisType) {
        String errorTag = analysisType + "-" + msgTreeNode.getAlarmMsg().getErrorType();
        String reduceKey = ALARM_APP + ":" + msgTreeNode.getApp().getClientname() + "-" + errorTag;
        logger.info("buildReduceKey={}", reduceKey);
        return reduceKey;
    }

    /**
     * 获取当前分析进程服务名
     *
     * @return
     */
    private String getAnalysisProcessName() {
        ProcessInfoCollector collector = AomiSdk.getProcessInfoCollector();
        String procName = "";
        if (collector != null) {
            procName = collector.getInfo().getProcessname();
        }
        return procName;
    }

}