package com.yy.aomi.analysis.common.quartz;

import org.quartz.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.quartz.SchedulerFactoryBeanCustomizer;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.quartz.SchedulerFactoryBean;

import javax.annotation.PostConstruct;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;


/**
 * 定时任务计划类
 */
@Configuration
@EnableScheduling
public class WorkScheduler implements SchedulerFactoryBeanCustomizer {
    private static Logger log = LoggerFactory.getLogger(WorkScheduler.class);
    private static ScheduledExecutorService scheduledThreadPool = Executors.newScheduledThreadPool(1);
    private static int jobKey = 0;
    public final static String JOB_PARAM = "jobParam";


    @Autowired
    private Scheduler sched;

    @PostConstruct
    private void init(){
        try {
            sched.clear();
        } catch (SchedulerException e) {
            log.error("", e);
        }
    }

    public  ScheduledExecutorService getScheduledExecutorService() {
        return scheduledThreadPool;
    }

    /**
     * 制定一个工作计划
     *
     * @param jobClass
     * @param cronExpresstion
     */
    public  void executeCronSchedule(Class<? extends Job> jobClass, String cronExpresstion) {
        executeCronSchedule(jobClass, cronExpresstion,null);
    }



    /**
     * 制定一个工作计划
     *
     * @param jobClass        处理job class实现 Job
     * @param cronExpresstion 业务参数数据
     * @param paramMap        其它的各种参数
     */
    public  void executeCronSchedule(Class<? extends Job> jobClass, String cronExpresstion, Map<String, Object> paramMap) {
        int _jobKey = jobKey++;
        JobDetail job;
        CronTrigger trigger;
        Date ft;
        try {
            job = JobBuilder.newJob(jobClass).withIdentity("job" + _jobKey, "group" + _jobKey).build();
            JobDataMap map = new JobDataMap();
            map.put("jobClass", jobClass);
            if (paramMap != null) {
                map.put(JOB_PARAM, paramMap);
            }

            trigger = TriggerBuilder.newTrigger()
                    .withIdentity("trigger" + _jobKey, "group" + _jobKey)
                    .withSchedule(CronScheduleBuilder.cronSchedule(cronExpresstion))
                    .usingJobData(map)
                    .build();

            ft = sched.scheduleJob(job, trigger);
            log.info(job.getKey() + " has been scheduled to run at: " + ft
                    + " and repeat based on expression: "
                    + trigger.getCronExpression());
//            sched.start();
        } catch (SchedulerException e) {
            log.error("WorkScheduler error.", e);
        }
    }


    @Override
    public void customize(SchedulerFactoryBean schedulerFactoryBean) {
        schedulerFactoryBean.setStartupDelay(2);
        schedulerFactoryBean.setAutoStartup(true);
        schedulerFactoryBean.setOverwriteExistingJobs(true);
    }
}
