package com.yy.yylivesdk4cloud;


import android.graphics.Bitmap;

import com.yy.mediaframework.facedetection.IFaceDetection;
import com.yy.mediaframework.gpuimage.custom.IGPUProcess;
import com.yy.mediaframework.gpuimage.custom.OrangeFilterWrapper;
import com.yy.mediaframework.utils.YMFLog;
import com.yy.yylivesdk4cloud.helper.ThunderLog;
import com.yy.yylivesdk4cloud.helper.ThunderNative;
import com.yy.yylivesdk4cloud.video.IVideoCaptureObserver;
import com.yy.yylivesdk4cloud.video.IVideoDecodeObserver;
import com.yy.yylivesdk4cloud.video.ThunderVideoLogCallback;
import com.yy.yylivesdk4cloud.video.VideoFrameYuvCapture;

import java.util.ArrayList;
import java.util.HashMap;


/**
 * Created by xiongxiong on 2017/10/30.
 */

public class ThunderPublisher {

	/**
	 * 外部推音频数据的接口集
	 * <br>开始采集时{@link ThunderAudioCapture#startCapture(IAudioPublisher)}会传出一个它的实现的实例。
	 * 外部通过该实例调用接口向内部推送音频数据
	 */
	public interface IAudioPublisher {
		/**
		 * 推送原始音频数据
		 * @param samples 原始音频数据
		 * @param cts 音频采集时间戳
		 */
		void pushAudioData(byte[] samples, long cts);
	}

	public interface IVideoPublisher {

		/**
		 * 推送原始视频数据
		 * @param data 		Byte Array 型的数据数据
		 * @param format 	像素格式：
		 *        RGBA:  public static final int THUNDER_PIXEL_FORMAT_RGBA = 0;
		 *        I420:  public static final int THUNDER_PIXEL_FORMAT_I420 = 1;
		 *        NV21   public static final int THUNDER_PIXEL_FORMAT_NV21 = 2;
		 * @param width		视频帧的宽度
		 * @param height 	视频帧的高度
		 * @param rotation	视频帧顺时针旋转的角度。如果设置了旋转角度，媒体引擎会对图像进行旋转。
		 *                  你可以根据需要将角度值设为 0 度、90 度、180 度和 270 度，如果设置为其他数字，系统会自动忽略
		 * @param timestamp	传入的视频帧的时间戳。开发者必须为每一个视频帧设置一个时间戳
		 */
		void pushVideoData(byte[] data, int format, int width, int height, int rotation, long timestamp);

		/**
		 * 推送编码后的视频数据
		 * @param encodedData 编码后的视频数据（一帧）
		 * @param type 编码类型
		 * @param dts 当前帧编码时的系统时间，单位ms
		 * @param pts 当前帧显示时的系统时间，单位ms
		 */
		void pushVideoData(byte[] encodedData, ThunderConstant.ThunderVideoEncodeType type, long dts, long pts);
	}

	private ThunderPublishAudioConfig mAudioConfig = null;
	private ThunderAudioCapture mAudioCapture = null;
	private ThunderDefaultMic mDefaultMic = new ThunderDefaultMic();

	private ThunderPublishVideoConfig mVideoConfig = null;
	private ThunderVideoCapture mVideoCapture = null;
	private ThunderDefaultCamera mDefaultCamera = new ThunderDefaultCamera();


	ThunderPublisher() {
		if (BuildConfig.__YY_VIDEO_SUPPORT__) {
			YMFLog.registerLogger(ThunderVideoLogCallback.sharedInstance());
		}
	}

	/**
	 * 开启预览
	 *
	 * @param view   预览的目标view，传入一个普通的 SurfaceView
	 * @return 0：成功；<0：error
	 */
	public int startPreviewWithView(Object view, int playType, int publishMode) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return 0;
		}
//		if (mVideoCapture == null) {
//			this.attachVideoCapture(mDefaultCamera);
//		}
		ThunderLog.release(ThunderLog.kLogTagCall, "startPreviewWithView view:%s", view.toString());
		return ThunderNative.startVideoPreview(view, playType, publishMode);
	}

	/**
	 * 停止预览
	 *
	 * @return 0：成功；<0：error
	 */
	public int stopPreview() {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return 0;
		}
		ThunderLog.release(ThunderLog.kLogTagCall, "stopVideoPreview");
		return ThunderNative.stopVideoPreview();
	}

	/**
	 * rtmp开播时需要先调用该接口，设置rtmp参数，rtmp流连接建立完成后，再调用正常的开播接口
	 * @param bOpen 	是否打开rtmp进行推流
	 * @param url		rtmp推流的url
	 * @param playPath	播放地址(默认不用设置)
	 */
	public void setLivePublishRtmpParam(boolean bOpen, String url, String playPath) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return;
		}
		ThunderLog.release(ThunderLog.kLogTagCall, "setLivePublishRtmpParam");
		ThunderNative.setLivePublishRtmpParam(bOpen, url, playPath);
	}

	/**
	 * camera yuv（NV21）数据回调给业务
	 * @param observer 注册的给用户回调给出数据的实例
	 */
	public void setVideoCaptureFrame(IVideoCaptureObserver observer) {
		VideoFrameYuvCapture.getInstance().enableVideoCapture(observer);
	}

	/**
	 * 设置客户端实现的美颜库到sdk,具体用法参照FaceBeautyDemo
	 * @param gpuProcess 用于获取yuv数据的回调PreviewFrameCallback实例，即人脸识别库识别代码实现实例
	 */
	public void setVideoCaptureTexture(IGPUProcess gpuProcess) {
		ThunderBridgeLib.getInstance().setGPUImageProcessFilter(gpuProcess);
	}

	/**
	 * 获取瘦脸接口,在startPreviewWithView之后调用
	 */
	public OrangeFilterWrapper getBeautyFilter(){
		return ThunderBridgeLib.getInstance().getBeautyFilter();
	}

	/**
	 * 设置第三方人脸到sdk
	 */
	public void setMobileFaceDetection(IFaceDetection faceDetection){
		ThunderBridgeLib.getInstance().setMobileFaceDetection(faceDetection);
	}

	/**
	 * 视频开播
	 * <p>
	 * <br> 开播前需要通过 {@link ThunderAPI#setUserRole(int)} 设置为{@link ThunderConstant.ThunderUserRole#THUNDERUSER_ROLE_ANCHOR}
	 * <br> 根据配置信息启动视频采集编码并推流
	 * <br> 同一个用户不能同时开播多条音频或视频流，相同uid的音频和视频配对进行同步
	 * <br> 可以把多条流发布到同一个群组，适用于多主播会话且单个主播开播时间较短的场景
	 * <br> 开播后可以通过对应的set/enable API 修改单项配置或updatePublishVideoConfig批量修改
	 * <br> 通过getCurPublishVideoConfig获取当前生效的配置
	 *
	 * @param streamName 流名称
	 * @param groupNames 此流发布到的目标群组，可以为空
	 * @param config     配置信息, null代表使用默认配置
	 * @return 0：成功；<0：error
	 * @see ThunderPublishVideoConfig
	 */
	public int startPublishVideo(String streamName, ArrayList<String> groupNames, ThunderPublishVideoConfig config) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return 0;
		}
		if (streamName == null || streamName.isEmpty()) {
			ThunderLog.error(ThunderLog.kLogTagCall, "startPublishVideo stream name must be set");
			return -1;
		}
		if (groupNames == null) {
			groupNames = new ArrayList<String>(0);
		}
		if (config == null) {
			config = new ThunderPublishVideoConfig();
		}

		mVideoConfig = config;

		ThunderLog.release(ThunderLog.kLogTagCall,
				"startPublishVideo streamName:%s, groupNames size:%d, config:%s",
				streamName, groupNames.size(), config.toString());

		return ThunderNative.startPublishVideo(streamName, groupNames, config);
	}

	/**
	 * 停播
	 *
	 * @return 0：成功；<0：error
	 */
	public int stopPublishVideo() {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return 0;
		}
		ThunderLog.release(ThunderLog.kLogTagCall, "stopPublishVideo");
		return ThunderNative.stopPublishVideo();
	}
	
		/**
	 * 获取开播支持的档位
	 * 对应标记参考 ThunderPublishVideoConfig.java
	 */
	public HashMap<Integer, String> getSupportModeInfoWithPlayType(int playType){
		int convertType = convertPlayType(playType);
		HashMap<Integer, String> map = ThunderNative.getSupportModeInfoWithPlayType(convertType);
		return map;
	}

	/**
	 * 使能硬编
	 * @param disable  true 则禁止硬编
	 */
	public void disableHardEncoder(Boolean disable){
		ThunderNative.disableHardEncoder(disable);
	}

	/**
	 * 获取远程服务器返回的配置
	 */
	public ThunderPublishVideoConfig getRemotePublishVideoConfig(int playType, int mode){
		int convertType = convertPlayType(playType);
		ThunderPublishVideoConfig remotePublishVideoConfig = ThunderNative.getRemoteVideoConfig(convertType, mode);
		return  remotePublishVideoConfig;
	}

	public int getDefaultPublishVideoMode(int playType){
		int convertType = convertPlayType(playType);
		return ThunderNative.getDefaultPublishVideoMode(convertType);
	}


	/**
	 * 获取当前生效的视频开播配置
	 */
	public ThunderPublishVideoConfig getCurPublishVideoConfig() {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return null;
		}
		return ThunderNative.getVideoConfig();
	}

	/**
	 * 批量更新视频开播配置
	 *
	 * @return 成功or失败，返回失败可能是设置的参数不合理
	 */
	public boolean updatePublishVideoConfig(ThunderPublishVideoConfig config) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return false;
		}
		mVideoConfig = config;
		return ThunderNative.updateVideoConfig(config);
	}

	/**
	 * 音频开播
	 * <br> 开播前需要通过 {@link ThunderAPI#setUserRole(int)} 设置为{@link ThunderConstant.ThunderUserRole#THUNDERUSER_ROLE_ANCHOR}
	 * <br> 根据配置信息启动音频采集编码并推流
	 * <br>同一个用户不能同时开播多条音频或视频流，相同uid的音频和视频配对进行同步
	 * <br>可以把多条流发布到同一个群组，适用多主播会话且单个主播开播时间较短的场景
	 * <br>通过getCurPublishAudioConfig获取当前生效的配置
	 *
	 * @param streamName 流名称
	 * @param groupNames 此流发布到的目标群组，可以为空
	 * @param config	 配置信息, null代表使用默认配置
	 * @return 0：成功；<0：error
	 */
	public int startPublishAudio(String streamName, ArrayList<String> groupNames, ThunderPublishAudioConfig config) {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return 0;
		}

		if (mAudioCapture == null) {
			ThunderNative.attachAudioCapture(mDefaultMic);
		}

		if (streamName == null || streamName.isEmpty()) {
			ThunderLog.error(ThunderLog.kLogTagCall, "startPublishAudio stream name must be set");
			return -1;
		}
		if (groupNames == null) {
			groupNames = new ArrayList<String>(0);
		}
		if (config == null) {
			config = new ThunderPublishAudioConfig(ThunderPublishAudioConfig
					.YYPUBLISH_AUDIO_MODE_NORMALDELAY_NORMALFLOW_MEDIUMQUALITY_COMMON);
		}
		mAudioConfig = config;
		ThunderLog.release(ThunderLog.kLogTagCall,
				"startPublishAudio streamName:%s, groupNames size:%d, config:%s",
				streamName, groupNames.size(), config.toString());
		return ThunderNative.startPublishAudio(streamName, groupNames, config);
	}

	/**
	 * 音频停播
	 *
	 * @return 0：成功；<0：error
	 */
	public int stopPublishAudio() {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return 0;
		}
		ThunderLog.release(ThunderLog.kLogTagCall, "stopPublishAudio");
		return ThunderNative.stopPublishAudio();
	}

	/**
	 * 获取当前生效的音频开播配置
	 */
	public ThunderPublishAudioConfig getCurPublishAudioConfig() {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return null;
		}
		return mAudioConfig;
	}

	/**
	 * 摄像头位置
	 *
	 * @param position 摄像头位置 {@link ThunderConstant.ThunderCameraPosition}
	 * @return 成功or失败
	 */
	public boolean setCameraPosition(int position) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return false;
		}
		ThunderLog.release(ThunderLog.kLogTagCall, "setCameraPosition %d", position);
		return ThunderNative.setCameraPosition(position);
	}

	/**
	 * 给开播视频打水印
	 *
	 * @param posX  水印在视频图像中的位置
	 * @param posY  水印在视频图像中的位置
	 * @param image 水印图片
	 * @return 0：成功；<0：error
	 */
	public int setWatermark(int posX, int posY, Bitmap image) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return 0;
		}

		if (image == null) {
			return -1;
		}
		ThunderLog.release(ThunderLog.kLogTagCall, "setWatermark in (%d,%d)", posX, posY);
		return ThunderNative.setWatermark(posX, posY, image);
	}

	/**
	 * 设置美颜级别
	 *
	 * @param level 取值范围[0, 1]
	 * @return 0：成功；<0：error
	 */
	public int setFaceBeautyLevel(float level) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return 0;
		}

		// float保证六位有效数字
		if (level >= 1.000001f) {
			level = 1.0f;
		} else if (level < 0.000001f) {
			level = 0.0f;
		}
		ThunderLog.release(ThunderLog.kLogTagCall, "setFaceBeautyLevel %f", level);
		return ThunderNative.setFaceBeautyLevel(level);
	}

	/**
	 * 设置开播音量
	 * <br>实际采集到的音量通过监听通知{@link ThunderNotification#kThunderAPINotification_AudioCaptureVolume} 获取
	 *
	 * @param volume 音量值，取值范围[0，+00], suggest < 150
	 * @return 成功or失败
	 */
	public boolean setMicVolume(int volume) {
		if (volume < 0) {
			volume  = 0;
		}
		ThunderLog.release(ThunderLog.kLogTagCall, "setMicVolume %d", volume);

		return ThunderNative.setMicVolume(volume);
	}

	/**
	 * 静音检测开关，默认是开启状态
	 *
	 * @param enable true - 开启; false - 关闭
	 */
	public void enableVad(boolean enable) {
		ThunderLog.release(ThunderLog.kLogTagCall, "enableVad %b", enable);
		ThunderNative.enableVad(enable);
	}

	public void EnableEqualizer(boolean enable)
	{
		ThunderLog.release(ThunderLog.kLogTagCall, "EnableEqualizer %b", enable);
		ThunderNative.EnableEqualizer(enable);
	}

	public void SetGqGains(final int [] gains)
	{
		ThunderNative.SetGqGains(gains);
	}

	public void EnableCompressor(boolean enable)
	{
		ThunderNative.EnableCompressor(enable);
	}

	public void EnableLimiter(boolean enable)
	{
		ThunderNative.EnableLimiter(enable);
	}

	public void SetCompressorParam(int threshold, int makeupGain, int ratio, int knee, int releaseTime, int attackTime)
	{
		ThunderNative.SetCompressorParam(threshold, makeupGain, ratio, knee, releaseTime, attackTime);
	}

	public void SetLimiterParam(final float fCeiling, final float fThreshold, final float fPreGain,
								 final float fRelease, final float fAttack, final float fLookahead,
								 final float fLookaheadRatio, final float fRMS, final float fStLink)
	{
		ThunderNative.setLimiterParameter(fCeiling, fThreshold, fPreGain, fRelease, fAttack, fLookahead, fLookaheadRatio, fRMS, fStLink);
	}

	public void EnableReverb(boolean enable)
	{
		ThunderNative.EnableReverb(enable);
	}

	public void setReverbExParameter(final float RoomSize, final float PreDelay, final float Reverberance,
									 final float HfDamping, final float ToneLow, final float ToneHigh,
									 final float WetGain,  final  float DryGain, final float StereoWidth)
	{
		ThunderNative.setReverbExParameter(RoomSize,PreDelay,Reverberance,HfDamping,ToneLow,ToneHigh,WetGain,DryGain,StereoWidth);
	}

	/**
	 * 获取本地麦克风，它是YYLiveAudioCapture的子类，实现了调用麦克风采集数据
	 *
	 * @return  本地麦克风对象
	 * @see ThunderDefaultMic
	 */
	public ThunderDefaultMic getDefaluteMic() {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return null;
		}
		return mDefaultMic;
	}

	/**
	 * 外部采集音频推流
	 * <br>通过此接口导入外部音频，设置为YYLiveDefaultMic对象或设置为null，SDK会调用设备的麦克风进行采集
	 * <br>此接口使用于单流的情况，如果有多流，使用混流接口{@link #setMixAudioCapture(ArrayList)}，
	 * 两个接口不能同时使用。
	 * @param capture 音频采集对象
	 * @see ThunderDefaultMic
	 */
	public void attachAudioCapture(ThunderAudioCapture capture) {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return;
		}
		if (capture == null) {
			mAudioCapture = mDefaultMic;
		} else {
			mAudioCapture = capture;
		}
		ThunderLog.release(ThunderLog.kLogTagCall, "attachAudioCapture %s", mAudioCapture.toString());
		ThunderNative.attachAudioCapture(mAudioCapture);

	}

	/**
	 * 本地音频混音后推流 [当前不支持]
	 * <br> 把多个音频源混音后推出
	 * @param mixCaptureList 需要混音的音频源
	 * @see ThunderAudioCapture
	 */
	public void setMixAudioCapture(ArrayList<ThunderAudioCapture> mixCaptureList) {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return;
		}
	}

	/**
	 * 获取本地麦克风，它是YYLiveVideoCapture的子类，实现了调用麦克风采集数据
	 *
	 * @return  本地摄像头对象
	 * @see ThunderDefaultCamera
	 */
	public ThunderDefaultCamera getDefaluteCamera() {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return null;
		}
		return mDefaultCamera;
	}

	/**
	 * 外部采集视频推流
	 * <br>通过此接口导入外部视频，设置为YYLiveDefaultCamera对象或设置为null，SDK会调用设备的摄像头进行采集
	 * <br>此接口使用于单流的情况，如果有多流，使用混流接口{@link #setMixVideoCapture(ArrayList)}，
	 * 两个接口不能同时使用。
	 * @param capture 视频采集对象
	 * @see ThunderDefaultCamera
	 */
	public void attachVideoCapture(ThunderVideoCapture capture) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return;
		}
		mVideoCapture = capture;
		ThunderLog.release(ThunderLog.kLogTagCall, "attachVideoCapture %s", mVideoCapture.toString());
		ThunderNative.attachVideoCapture(mVideoCapture);
	}

	/**
	 * 本地视频混流后推流 [当前不支持]
	 * <br> 把多个视频源混音后推出
	 * @param mixCaptureList 需要混音的视频源
	 * @see ThunderVideoCapture
	 */
	public void setMixVideoCapture(ArrayList<ThunderVideoCapture> mixCaptureList) {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return;
		}
	}

	/**
	 * 初始化录屏
	 * @param yyLiveScreenCapture 录屏对象
	 */
	public void initSreenLiveSession(ThunderScreenCapture yyLiveScreenCapture) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return;
		}
		if (yyLiveScreenCapture != null) {
			this.attachVideoCapture(yyLiveScreenCapture);
		}
	}

	/**
	 * 设置开播角度（横屏/竖屏）
	 * @param orientation
	 */
	public void setOrientation(int orientation){
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return;
		}
		ThunderNative.setOrientation(orientation);
	}


	public void setAudioSourceType(int mode)
	{
		ThunderNative.setAudioSourceType(mode);
	}

	public boolean enableInEarMonitor(boolean enable)
	{
		return ThunderNative.enableInEarMonitor(enable);
	}

	/**
	 *设置不同音效模式
	 *
	 *@param mode  = ThunderConstant.ThunderSoundEffectMode.
	 *             THUNDER_SOUND_EFFECT_MODE_NONE               = 关闭模式
	 *             THUNDER_SOUND_EFFECT_MODE_VALLEY             = VALLEY模式
	 *             THUNDER_SOUND_EFFECT_MODE_RANDB              = R&B模式
	 *             THUNDER_SOUND_EFFECT_MODE_KTV                = KTV模式
	 *             THUNDER_SOUND_EFFECT_MODE_CHARMING           = CHARMING模式
	 *             THUNDER_SOUND_EFFECT_MODE_POP                = 流行模式
	 *             THUNDER_SOUND_EFFECT_MODE_HIPHOP             = 嘻哈模式
	 *             THUNDER_SOUND_EFFECT_MODE_ROCK               = 摇滚模式
	 *             THUNDER_SOUND_EFFECT_MODE_CONCERT            = 演唱会模式
	 *             THUNDER_SOUND_EFFECT_MODE_STUDIO             = 录音棚模式
	 *
	 */
	public void setSoundEffect(int mode) {
		ThunderLog.release(ThunderLog.kLogTagCall, "ThunderPublisher::setSoundEffect %d", mode);
		ThunderNative.setSoundEffect(mode);
	}

	/**
	 *设置变声模式
	 *
	 *@param mode  =ThunderRtcConstant.VoiceChangerMode.
	 *                     THUNDER_VOICE_CHANGER_NONE                = 关闭模式
	 *                     THUNDER_VOICE_CHANGER_ETHEREAL            = 空灵
	 *                     THUNDER_VOICE_CHANGER_THRILLER            = 惊悚
	 *                     THUNDER_VOICE_CHANGER_LUBAN               = 鲁班
	 *                     THUNDER_VOICE_CHANGER_LORIE               = 萝莉
	 *                     THUNDER_VOICE_CHANGER_UNCLE               = 大叔
	 *                     THUNDER_VOICE_CHANGER_DIEFAT              = 死肥仔
	 *                     THUNDER_VOICE_CHANGER_BADBOY              = 熊孩子
	 *                     THUNDER_VOICE_CHANGER_WRACRAFT            = 魔兽农民
	 *                     THUNDER_VOICE_CHANGER_HEAVYMETAL          = 重金属
	 *                     THUNDER_VOICE_CHANGER_COLD                = 感冒
	 *                     THUNDER_VOICE_CHANGER_HEAVYMECHINERY      = 重机械
	 *                     THUNDER_VOICE_CHANGER_TRAPPEDBEAST        = 困兽
	 *                     THUNDER_VOICE_CHANGER_POWERCURRENT        = 强电流
	 *
	 */
	public void setVoiceChanger(int mode) {
		ThunderLog.release(ThunderLog.kLogTagCall, "ThunderPublisher::setVoiceChanger %d", mode);
		ThunderNative.setVoiceChanger(mode);
	}


	private int convertPlayType(int playType) {
		switch (playType) {
			case ThunderConstant.ThunderPublishPlayType.THUNDERPUBLISH_PLAY_SINGLE:
				playType = ThunderPublishVideoConfig.YYPUBLISH_PLAYTYPE_SINGLE;
				break;
			case ThunderConstant.ThunderPublishPlayType.THUNDERPUBLISH_PLAY_INTERACT:
				playType = ThunderPublishVideoConfig.YYPUBLISH_PLAYTYPE_LIANMAI;
				break;
			case ThunderConstant.ThunderPublishPlayType.THUNDERPUBLISH_PLAY_SCREENCAP:
				playType = ThunderPublishVideoConfig.YYPUBLISH_PLAYTYPE_SCREEN_CAPTURE;
				break;
			default:
				break;
		}
		return playType;
	}
}
