package com.yy.yylivesdk4cloud;

import com.yy.yylivesdk4cloud.helper.ThunderLog;
import com.yy.yylivesdk4cloud.helper.ThunderNative;

/**
 * 播放音频文件的接口，需要每个播放文件创建一个 ThunderAudioFilePlayer 的实例。
 * 如果作为伴奏使用，需要调用{@link #enablePublish(boolean)}
 */
public class ThunderAudioFilePlayer implements Comparable<ThunderAudioFilePlayer>{

    private IThunderAudioFilePlayerCallback mCallback = null;
    private long nativeCtx = 0;
    private boolean mIsDestroy = false;

    private final int PLAY_EVENT_END = 1;
    private final int FILE_PLAY_VOLUME = 2;

    public ThunderAudioFilePlayer(){
        nativeCtx = ThunderNative.createAudioFilePlayer(this);
    }

    /**
     * 播放事件的回调接口
     */
    public interface IThunderAudioFilePlayerCallback {
        /**
         * 播放结束
         */
        void onAudioFilePlayEnd();

        /**
         * 播放音量回调
         * @param volume 音量值[0-100]
         * @param currentMs 播放进度值（单位：毫秒）
         * @param totalMs 文件总长度（单位：毫秒）
         */
        void onAudioFileVolume(long volume, long currentMs, long totalMs);
    }

    /**
     * 设置播放回调接口
     * @param callback 回调接口
     */
    public synchronized void setPlayerNotify(IThunderAudioFilePlayerCallback callback) {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer setPlayerNotify" );

        if (mIsDestroy) {
            return;
        }
        mCallback = callback;
    }

    /**
     * 打开文件播放音量回调
     * @param interval 回调时间间隔（单位:毫秒），建议设置200ms以上;如果<=0,则重置为200ms；
     */
    public synchronized void enableVolumeIndication(boolean enable, int interval){
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer enableVolumeNotify enable = %b, interval=%d ", enable, interval);
        if (mIsDestroy) {
            return;
        }

        if (interval <= 0){
            interval = 200;
        }

        ThunderNative.enableAudioFileVolumeCallback(nativeCtx, enable, interval);
    }

    /**
     *  打开需要播放的文件，支持文件格式：mp3、aac、wav
     *  <br>
     *  @param path 文件路径
     *  @return 成功or失败
     */
    public boolean open(String path) {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer open path = %s", path);

        if (mIsDestroy) {
            return false;
        }

        int ret = (int) ThunderNative.audioFileOpen(nativeCtx, path);
        return  ret == 1;
    }

    /**
     * 关闭播放文件，可以重新调用{@link #open(String)}
     */
    public void close() {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer close" );

        if (mIsDestroy) {
            return;
        }
        ThunderNative.audioFileClose(nativeCtx);

    }

    /**
     * 开始播放
     */
    public void play() {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer play" );

        if (mIsDestroy) {
            return;
        }
        ThunderNative.audioFilePlay(nativeCtx);
    }

    /**
     * 停止播放，调用{@link #play()}重新开始播放
     */
    public void stop() {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer stop" );

        if (mIsDestroy) {
            return;
        }
        ThunderNative.audioFileStop(nativeCtx);
    }

    /**
     * 暂停播放，调用{@link #resume()}继续播放
     */
    public void pause() {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer pause" );

        if (mIsDestroy) {
            return;
        }
        ThunderNative.audioFilePause(nativeCtx);
    }

    /**
     * 继续播放
     */
    public void resume() {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer resume" );
        if (mIsDestroy) {
            return;
        }
        ThunderNative.audioFileResume(nativeCtx);
    }

    /**
     * 跳转到指定的播放时间
     * <br>
     * @param timeMS 需要跳转到的时间点（单位：毫秒），不应该大于{@link #getTotalPlayTimeMS()}获取的值
     */
    public void seek(long timeMS) {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer seek timems = %d ", timeMS );

        if (mIsDestroy) {
            return;
        }
        ThunderNative.audioFileSeek(nativeCtx, timeMS);
    }

    /**
     * 获取文件的总播放时长
     * <br>
     * @return 总播放时长，单位：毫秒
     */
    public long getTotalPlayTimeMS() {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer getTotalPlayTimeMS ");

        if (mIsDestroy) {
            return 0;
        }
        return ThunderNative.audioFileGetTotalTime(nativeCtx);
    }

    /**
     * 获取当前已经播放的时长
     * <br>
     * @return 已播放时长（单位：毫秒），不应该大于{@link #getTotalPlayTimeMS()}获取的值
     */
    public long getCurrentPlayTimeMS() {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer getCurrentPlayTimeMS ");

        if (mIsDestroy) {
            return 0;
        }
        return ThunderNative.audioFileGetCurrentPlayTime(nativeCtx);
    }

    /**
     * 设置当前文件的播放音量
     * <br>
     * @param volume 播放音量 [0-100]
     */
    public void setPlayVolume(int volume) {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer setPlayVolume volume = %d", volume);

        if (mIsDestroy) {
            return;
        }

        if (volume < 0) {
            volume = 0;
        } else if (volume > 100) {
            volume = 100;
        }
        ThunderNative.audioFileSetPlayVolume(nativeCtx, volume);
    }

    /**
     * 设置音频播放的声调
     * <br>
     * @param val -5, -4, -3, -2, -1, 0(normal), 1, 2, 3, 4, 5
     */
    public void setSemitone(int val) {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer setSemitone val = %d", val);

        if (mIsDestroy) {
            return;
        }

        float tone = val;
        ThunderNative.audioFileSetSemitone(nativeCtx, tone);
    }

    /**
     * 是否将当前播放的文件作为直播伴奏使用
     * <br>
     * @param enable 开启或关闭，默认关闭
     */
    public void enablePublish(boolean enable) {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer enablePublish: enable=%b", enable);

        if (mIsDestroy) {
            return;
        }
        ThunderNative.audioFileEnablePublish(nativeCtx, enable);
    }

    /**
     * 设置Karaoke混音延时
     */
    public void setKaraokeCompensateValue(int value) {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer setKaraokeCompensateValue value = %d ", value);

        ThunderNative.audioFileSetCompensateValue(nativeCtx, value);
    }

    /**
     * 向外通知播放事件
     */
    public void onPlayEvent(int event) {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer onPlayEvent event = %d", event);

        if (mIsDestroy) {
            return;
        }

        if (mCallback == null) {
            return;
        }

        switch (event) {
            case PLAY_EVENT_END:
                mCallback.onAudioFilePlayEnd();
            default:
                break;
        }
    }

    /*

     */
    public synchronized void onAudioFileVolume(long volume, long currentMs, long totalMs){
        if (mIsDestroy) {
            return;
        }

        if (mCallback == null) {
            return;
        }

        mCallback.onAudioFileVolume(volume, currentMs, totalMs);
    }

    /*
    销毁该类, 不再允许其他操作, 包内访问权限
     */
    void destroyAudioFilePlayer() {
        ThunderLog.release(ThunderLog.kLogTagCall,"ThunderAudioFilePlayer destroyAudioFilePlayer");

        setPlayerNotify(null);
        ThunderNative.destroyAudioFilePlayer(nativeCtx);
        mIsDestroy = true;
        nativeCtx  = 0;
    }

    protected void finalize(){
        destroyAudioFilePlayer();
    }

    @Override
    public int compareTo(ThunderAudioFilePlayer audioFilePlayer) {
        if (audioFilePlayer == null) {
            throw new NullPointerException();
        }

        if (this == audioFilePlayer) {
            return 0;
        }
        return 1;
    }
}
