package com.yy.yylivesdk4cloud.system;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.telephony.TelephonyManager;

import com.yy.yylivesdk4cloud.helper.YYLiveLog;
import com.yy.yylivesdk4cloud.helper.YYLiveNative;

/**
 * Created by xiaojun on 2018/1/12.
 * Copyright (c) 2017 YY Inc. All rights reserved.
 */

public class YYNetStateService {

	private static final String INTERNET_PERMISSION = "android.permission.INTERNET";
	private static final String NETWORK_PERMISSION = "android.permission.ACCESS_NETWORK_STATE";
	private Context mContext;

	private static final class NetState {
		private static final int SYSNET_WIFI		= 0;
		private static final int SYSNET_MOBILE		= 1;
		private static final int SYSNET_DISCONNECT	= 2;
		private static final int SYSNET_2G			= 3;
		private static final int SYSNET_3G			= 4;
		private static final int SYSNET_4G			= 5;
		private static final int SYSNET_UNKNOWN		= 127;
	};

	public YYNetStateService(Context mContext) {
		this.mContext = mContext;

	}

	public void init() {
		IntentFilter mFilter = new IntentFilter();
		mFilter.addAction(ConnectivityManager.CONNECTIVITY_ACTION);
		mContext.registerReceiver(mNetReceiver, mFilter);
	}

	public void fini() {
		mContext.unregisterReceiver(mNetReceiver);
	}

	private BroadcastReceiver mNetReceiver = new BroadcastReceiver() {
		@Override
		public void onReceive(Context context, Intent intent) {
			YYLiveLog.info(YYLiveLog.kLogTagSdk, "NetworkStateService onReceive pid %d",
					Thread.currentThread().getId());
			String action = intent.getAction();
			if (action != null && action.equals(ConnectivityManager.CONNECTIVITY_ACTION)) {
				YYLiveLog.info(YYLiveLog.kLogTagSdk, "current network connectivity action");
				updateNetInfo(context);
			}
		}
	};

	private static boolean checkHasPermission(Context context, String perm) {
		int result = context.checkCallingOrSelfPermission(perm);
		return result == PackageManager.PERMISSION_GRANTED;
	}

	private static int getMobileNetworkState(Context context) {
		TelephonyManager telephonyManager =
				(TelephonyManager)context.getSystemService(Context.TELEPHONY_SERVICE);
		if (telephonyManager == null) {
			YYLiveLog.warn(YYLiveLog.kLogTagSdk, "cannot get TelephonyManager");
			return NetState.SYSNET_MOBILE;
		}
		int netType = telephonyManager.getNetworkType();
		switch (netType) {
			case TelephonyManager.NETWORK_TYPE_GPRS:
			case TelephonyManager.NETWORK_TYPE_GSM:
			case TelephonyManager.NETWORK_TYPE_EDGE:
			case TelephonyManager.NETWORK_TYPE_CDMA:
			case TelephonyManager.NETWORK_TYPE_1xRTT:
			case TelephonyManager.NETWORK_TYPE_IDEN:
				return NetState.SYSNET_2G;
			case TelephonyManager.NETWORK_TYPE_UMTS:
			case TelephonyManager.NETWORK_TYPE_EVDO_0:
			case TelephonyManager.NETWORK_TYPE_EVDO_A:
			case TelephonyManager.NETWORK_TYPE_HSDPA:
			case TelephonyManager.NETWORK_TYPE_HSUPA:
			case TelephonyManager.NETWORK_TYPE_HSPA:
			case TelephonyManager.NETWORK_TYPE_EVDO_B:
			case TelephonyManager.NETWORK_TYPE_EHRPD:
			case TelephonyManager.NETWORK_TYPE_HSPAP:
			case TelephonyManager.NETWORK_TYPE_TD_SCDMA:
				return NetState.SYSNET_3G;
			case TelephonyManager.NETWORK_TYPE_LTE:
			case TelephonyManager.NETWORK_TYPE_IWLAN:
				return NetState.SYSNET_4G;
			default:
				YYLiveLog.warn(YYLiveLog.kLogTagSdk, "unknown mobile network type:" + netType);
				return NetState.SYSNET_MOBILE;
		}
	}

	private static void updateNetInfo(Context context)
	{
		if (context == null)
			return;

		try {
			if (!(checkHasPermission(context, INTERNET_PERMISSION) &&
					checkHasPermission(context, NETWORK_PERMISSION))) {
				YYLiveLog.info(YYLiveLog.kLogTagSdk, "cannot get permission INTERNET or ACCESS_NETWORK_STATE!!");
				return;
			}

			ConnectivityManager connectivityManager =
					(ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
			if (connectivityManager == null) {
				YYLiveLog.info(YYLiveLog.kLogTagSdk, "cannot get ConnectivityManager!!");
				return;
			}
			NetworkInfo info = connectivityManager.getActiveNetworkInfo();

			int netState;
			if (info != null && info.isAvailable()) {
				int nType = info.getType();

				if (nType == ConnectivityManager.TYPE_MOBILE) {
					netState = getMobileNetworkState(context);
					YYLiveLog.info(YYLiveLog.kLogTagSdk, "current network TYPE_MOBILE, netState: %d",
							netState);
				}
				else if (nType == ConnectivityManager.TYPE_WIFI) {
					netState = NetState.SYSNET_WIFI;
					YYLiveLog.info(YYLiveLog.kLogTagSdk, "current network wifi");
				} else {
					YYLiveLog.warn(YYLiveLog.kLogTagSdk, "current network %s is omitted",
							info.getTypeName());
					return;
				}
			} else {
				YYLiveLog.info(YYLiveLog.kLogTagSdk, "current network No usable network!!");
				netState = NetState.SYSNET_DISCONNECT;
			}

			YYLiveNative.notifyNetState(netState);
		}catch (Exception exc) {
			YYLiveLog.warn(YYLiveLog.kLogTagSdk, "update net info error:" + exc);
		}
	}
}
