package com.yy.yylivesdk4cloud;


import android.graphics.Bitmap;

import com.yy.mediaframework.gpuimage.custom.OrangeFilterWrapper;
import com.yy.mediaframework.utils.YMFLog;
import com.yy.yylivesdk4cloud.helper.YYLiveLog;
import com.yy.yylivesdk4cloud.helper.YYLiveNative;
import com.yy.yylivesdk4cloud.video.YYVideoLogCallback;

import java.util.ArrayList;
import java.util.HashMap;


/**
 * Created by xiongxiong on 2017/10/30.
 */

public class YYLivePublisher {

	/**
	 * 外部推音频数据的接口集
	 * <br>开始采集时{@link YYLiveAudioCapture#startCapture(IAudioPublisher)}会传出一个它的实现的实例。
	 * 外部通过该实例调用接口向内部推送音频数据
	 */
	public interface IAudioPublisher {
		/**
		 * 推送原始音频数据
		 * @param samples 原始音频数据
		 * @param cts 音频采集时间戳
		 */
		void pushAudioData(byte[] samples, long cts);
	}

	public interface IVideoPublisher {
		/**
		 * 推送原始视频数据
		 * TODO: 添加参数
		 */
		void pushVideoData();

		/**
		 * 推送编码后的视频数据
		 * @param encodedData 编码后的视频数据（一帧）
		 * @param type 编码类型
		 * @param dts 当前帧编码时的系统时间，单位ms
		 * @param pts 当前帧显示时的系统时间，单位ms
		 */
		void pushVideoData(byte[] encodedData, YYConstant.YYVideoEncodeType type, long dts, long pts);
	}

	private YYPublishAudioConfig mAudioConfig = null;
	private YYLiveAudioCapture mAudioCapture = null;
	private YYLiveDefaultMic mDefaultMic = new YYLiveDefaultMic();

	private YYPublishVideoConfig mVideoConfig = null;
	private YYLiveVideoCapture mVideoCapture = null;
	private YYLiveDefaultCamera mDefaultCamera = new YYLiveDefaultCamera();

	YYLivePublisher() {
		if (BuildConfig.__YY_VIDEO_SUPPORT__) {
			YMFLog.registerLogger(YYVideoLogCallback.sharedInstance());
		}
	}

	/**
	 * 开启预览
	 *
	 * @param view   预览的目标view，传入一个普通的 SurfaceView
	 * @return 0：成功；<0：error
	 */
	public int startPreviewWithView(Object view, int playType, int publishMode) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return 0;
		}
		if (mVideoCapture == null) {
			this.attachVideoCapture(mDefaultCamera);
		}
		YYLiveLog.release(YYLiveLog.kLogTagCall, "startPreviewWithView view:%s", view.toString());
		return YYLiveNative.startVideoPreview(view, playType, publishMode);
	}

	/**
	 * 停止预览
	 *
	 * @return 0：成功；<0：error
	 */
	public int stopPreview() {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return 0;
		}
		YYLiveLog.release(YYLiveLog.kLogTagCall, "stopVideoPreview");
		return YYLiveNative.stopVideoPreview();
	}

	/**
	 * rtmp开播时需要先调用该接口，设置rtmp参数，rtmp流连接建立完成后，再调用正常的开播接口
	 * @param bOpen 	是否打开rtmp进行推流
	 * @param url		rtmp推流的url
	 * @param playPath	播放地址(默认不用设置)
	 */
	public void setLivePublishRtmpParam(boolean bOpen, String url, String playPath) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return;
		}
		YYLiveLog.release(YYLiveLog.kLogTagCall, "setLivePublishRtmpParam");
		YYLiveNative.setLivePublishRtmpParam(bOpen, url, playPath);
	}

	/**
	 * 获取瘦脸接口,在startPreviewWithView之后调用
	 */
	public OrangeFilterWrapper getBeautyFilter(){
		return YYLiveBridgeLib.getInstance().getBeautyFilter();
	}

	/**
	 * 视频开播
	 * <p>
	 * <br> 开播前需要通过 {@link YYLiveAPI#setUserRole(int)} 设置为{@link YYConstant.YYUserRole#YYUSER_ROLE_ANCHOR}
	 * <br> 根据配置信息启动视频采集编码并推流
	 * <br> 同一个用户不能同时开播多条音频或视频流，相同uid的音频和视频配对进行同步
	 * <br> 可以把多条流发布到同一个群组，适用于多主播会话且单个主播开播时间较短的场景
	 * <br> 开播后可以通过对应的set/enable API 修改单项配置或updatePublishVideoConfig批量修改
	 * <br> 通过getCurPublishVideoConfig获取当前生效的配置
	 *
	 * @param streamName 流名称
	 * @param groupNames 此流发布到的目标群组，可以为空
	 * @param config     配置信息, null代表使用默认配置
	 * @return 0：成功；<0：error
	 * @see YYPublishVideoConfig
	 */
	public int startPublishVideo(String streamName, ArrayList<String> groupNames, YYPublishVideoConfig config) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return 0;
		}
		if (streamName == null || streamName.isEmpty()) {
			YYLiveLog.error(YYLiveLog.kLogTagCall, "startPublishVideo stream name must be set");
			return -1;
		}
		if (groupNames == null) {
			groupNames = new ArrayList<String>(0);
		}
		if (config == null) {
			config = new YYPublishVideoConfig();
		}

		mVideoConfig = config;

		YYLiveLog.release(YYLiveLog.kLogTagCall,
				"startPublishVideo streamName:%s, groupNames size:%d, config:%s",
				streamName, groupNames.size(), config.toString());

		return YYLiveNative.startPublishVideo(streamName, groupNames, config);
	}

	/**
	 * 停播
	 *
	 * @return 0：成功；<0：error
	 */
	public int stopPublishVideo() {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return 0;
		}
		YYLiveLog.release(YYLiveLog.kLogTagCall, "stopPublishVideo");
		return YYLiveNative.stopPublishVideo();
	}
	
		/**
	 * 获取开播支持的档位
	 * 对应标记参考 YYPublishVideoConfig.java
	 */
	public HashMap<Integer, String> getSupportModeInfoWithPlayType(int playType){
		int convertType = convertPlayType(playType);
		HashMap<Integer, String> map = YYLiveNative.getSupportModeInfoWithPlayType(convertType);
		return map;
	}

	/**
	 * 使能硬编
	 * @param disable  true 则禁止硬编
	 */
	public void disableHardEncoder(Boolean disable){
		YYLiveNative.disableHardEncoder(disable);
	}

	/**
	 * 获取远程服务器返回的配置
	 */
	public YYPublishVideoConfig getRemotePublishVideoConfig(int playType, int mode){
		int convertType = convertPlayType(playType);
		YYPublishVideoConfig remotePublishVideoConfig = YYLiveNative.getRemoteVideoConfig(convertType, mode);
		return  remotePublishVideoConfig;
	}

	public int getDefaultPublishVideoMode(int playType){
		int convertType = convertPlayType(playType);
		return YYLiveNative.getDefaultPublishVideoMode(convertType);
	}


	/**
	 * 获取当前生效的视频开播配置
	 */
	public YYPublishVideoConfig getCurPublishVideoConfig() {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return null;
		}
		return YYLiveNative.getVideoConfig();
	}

	/**
	 * 批量更新视频开播配置
	 *
	 * @return 成功or失败，返回失败可能是设置的参数不合理
	 */
	public boolean updatePublishVideoConfig(YYPublishVideoConfig config) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return false;
		}
		mVideoConfig = config;
		return YYLiveNative.updateVideoConfig(config);
	}

	/**
	 * 音频开播
	 * <br> 开播前需要通过 {@link YYLiveAPI#setUserRole(int)} 设置为{@link YYConstant.YYUserRole#YYUSER_ROLE_ANCHOR}
	 * <br> 根据配置信息启动音频采集编码并推流
	 * <br>同一个用户不能同时开播多条音频或视频流，相同uid的音频和视频配对进行同步
	 * <br>可以把多条流发布到同一个群组，适用多主播会话且单个主播开播时间较短的场景
	 * <br>通过getCurPublishAudioConfig获取当前生效的配置
	 *
	 * @param streamName 流名称
	 * @param groupNames 此流发布到的目标群组，可以为空
	 * @param config	 配置信息, null代表使用默认配置
	 * @return 0：成功；<0：error
	 */
	public int startPublishAudio(String streamName, ArrayList<String> groupNames, YYPublishAudioConfig config) {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return 0;
		}

		if (mAudioCapture == null) {
			YYLiveNative.attachAudioCapture(mDefaultMic);
		}

		if (streamName == null || streamName.isEmpty()) {
			YYLiveLog.error(YYLiveLog.kLogTagCall, "startPublishAudio stream name must be set");
			return -1;
		}
		if (groupNames == null) {
			groupNames = new ArrayList<String>(0);
		}
		if (config == null) {
			config = new YYPublishAudioConfig(YYPublishAudioConfig
					.YYPUBLISH_AUDIO_MODE_NORMALDELAY_NORMALFLOW_MEDIUMQUALITY_COMMON);
		}
		mAudioConfig = config;
		YYLiveLog.release(YYLiveLog.kLogTagCall,
				"startPublishAudio streamName:%s, groupNames size:%d, config:%s",
				streamName, groupNames.size(), config.toString());
		return YYLiveNative.startPublishAudio(streamName, groupNames, config);
	}

	/**
	 * 音频停播
	 *
	 * @return 0：成功；<0：error
	 */
	public int stopPublishAudio() {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return 0;
		}
		YYLiveLog.release(YYLiveLog.kLogTagCall, "stopPublishAudio");
		return YYLiveNative.stopPublishAudio();
	}

	/**
	 * 获取当前生效的音频开播配置
	 */
	public YYPublishAudioConfig getCurPublishAudioConfig() {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return null;
		}
		return mAudioConfig;
	}

	/**
	 * 摄像头位置
	 *
	 * @param position 摄像头位置 {@link YYConstant.YYCameraPosition}
	 * @return 成功or失败
	 */
	public boolean setCameraPosition(int position) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return false;
		}
		YYLiveLog.release(YYLiveLog.kLogTagCall, "setCameraPosition %d", position);
		return YYLiveNative.setCameraPosition(position);
	}

	/**
	 * 给开播视频打水印
	 *
	 * @param posX  水印在视频图像中的位置
	 * @param posY  水印在视频图像中的位置
	 * @param image 水印图片
	 * @return 0：成功；<0：error
	 */
	public int setWatermark(int posX, int posY, Bitmap image) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return 0;
		}

		if (image == null) {
			return -1;
		}
		YYLiveLog.release(YYLiveLog.kLogTagCall, "setWatermark in (%d,%d)", posX, posY);
		return YYLiveNative.setWatermark(posX, posY, image);
	}

	/**
	 * 设置美颜级别
	 *
	 * @param level 取值范围[0, 1]
	 * @return 0：成功；<0：error
	 */
	public int setFaceBeautyLevel(float level) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return 0;
		}

		// float保证六位有效数字
		if (level >= 1.000001f) {
			level = 1.0f;
		} else if (level < 0.000001f) {
			level = 0.0f;
		}
		YYLiveLog.release(YYLiveLog.kLogTagCall, "setFaceBeautyLevel %f", level);
		return YYLiveNative.setFaceBeautyLevel(level);
	}

	/**
	 * 设置开播音量
	 * <br>实际采集到的音量通过监听通知{@link YYLiveNotification#kYYLiveAPINotification_AudioCaptureVolume} 获取
	 *
	 * @param volume 音量值，取值范围[0，+00], suggest < 150
	 * @return 成功or失败
	 */
	public boolean setMicVolume(int volume) {
		if (volume < 0) {
			volume  = 0;
		}
		YYLiveLog.release(YYLiveLog.kLogTagCall, "setMicVolume %d", volume);

		return YYLiveNative.setMicVolume(volume);
	}

	/**
	 * 静音检测开关，默认是开启状态
	 *
	 * @param enable true - 开启; false - 关闭
	 */
	public void enableVad(boolean enable) {
		YYLiveLog.release(YYLiveLog.kLogTagCall, "enableVad %b", enable);
		YYLiveNative.enableVad(enable);
	}

	public void EnableEqualizer(boolean enable)
	{
		YYLiveLog.release(YYLiveLog.kLogTagCall, "EnableEqualizer %b", enable);
		YYLiveNative.EnableEqualizer(enable);
	}

	public void SetGqGains(final int [] gains)
	{
		YYLiveNative.SetGqGains(gains);
	}

	public void EnableCompressor(boolean enable)
	{
		YYLiveNative.EnableCompressor(enable);
	}

	public void EnableLimiter(boolean enable)
	{
		YYLiveNative.EnableLimiter(enable);
	}

	public void SetCompressorParam(int threshold, int makeupGain, int ratio, int knee, int releaseTime, int attackTime)
	{
		YYLiveNative.SetCompressorParam(threshold, makeupGain, ratio, knee, releaseTime, attackTime);
	}

	public void SetLimiterParam(final float fCeiling, final float fThreshold, final float fPreGain,
								 final float fRelease, final float fAttack, final float fLookahead,
								 final float fLookaheadRatio, final float fRMS, final float fStLink)
	{
		YYLiveNative.setLimiterParameter(fCeiling, fThreshold, fPreGain, fRelease, fAttack, fLookahead, fLookaheadRatio, fRMS, fStLink);
	}

	public void EnableReverb(boolean enable)
	{
		YYLiveNative.EnableReverb(enable);
	}

	public void setReverbExParameter(final float RoomSize, final float PreDelay, final float Reverberance,
									 final float HfDamping, final float ToneLow, final float ToneHigh,
									 final float WetGain,  final  float DryGain, final float StereoWidth)
	{
		YYLiveNative.setReverbExParameter(RoomSize,PreDelay,Reverberance,HfDamping,ToneLow,ToneHigh,WetGain,DryGain,StereoWidth);
	}

	/**
	 * 获取本地麦克风，它是YYLiveAudioCapture的子类，实现了调用麦克风采集数据
	 *
	 * @return  本地麦克风对象
	 * @see YYLiveDefaultMic
	 */
	public YYLiveDefaultMic getDefaluteMic() {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return null;
		}
		return mDefaultMic;
	}

	/**
	 * 外部采集音频推流
	 * <br>通过此接口导入外部音频，设置为YYLiveDefaultMic对象或设置为null，SDK会调用设备的麦克风进行采集
	 * <br>此接口使用于单流的情况，如果有多流，使用混流接口{@link #setMixAudioCapture(ArrayList)}，
	 * 两个接口不能同时使用。
	 * @param capture 音频采集对象
	 * @see YYLiveDefaultMic
	 */
	public void attachAudioCapture(YYLiveAudioCapture capture) {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return;
		}
		if (capture == null) {
			mAudioCapture = mDefaultMic;
		} else {
			mAudioCapture = capture;
		}
		YYLiveLog.release(YYLiveLog.kLogTagCall, "attachAudioCapture %s", mAudioCapture.toString());
		YYLiveNative.attachAudioCapture(mAudioCapture);

	}

	/**
	 * 本地音频混音后推流 [当前不支持]
	 * <br> 把多个音频源混音后推出
	 * @param mixCaptureList 需要混音的音频源
	 * @see YYLiveAudioCapture
	 */
	public void setMixAudioCapture(ArrayList<YYLiveAudioCapture> mixCaptureList) {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return;
		}
	}

	/**
	 * 获取本地麦克风，它是YYLiveVideoCapture的子类，实现了调用麦克风采集数据
	 *
	 * @return  本地摄像头对象
	 * @see YYLiveDefaultCamera
	 */
	public YYLiveDefaultCamera getDefaluteCamera() {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return null;
		}
		return mDefaultCamera;
	}

	/**
	 * 外部采集视频推流
	 * <br>通过此接口导入外部视频，设置为YYLiveDefaultCamera对象或设置为null，SDK会调用设备的摄像头进行采集
	 * <br>此接口使用于单流的情况，如果有多流，使用混流接口{@link #setMixVideoCapture(ArrayList)}，
	 * 两个接口不能同时使用。
	 * @param capture 视频采集对象
	 * @see YYLiveDefaultCamera
	 */
	public void attachVideoCapture(YYLiveVideoCapture capture) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return;
		}
		if (capture == null) {
			mVideoCapture = mDefaultCamera;
		} else {
			mVideoCapture = capture;
		}
		YYLiveLog.release(YYLiveLog.kLogTagCall, "attachVideoCapture %s", mVideoCapture.toString());
		YYLiveNative.attachVideoCapture(mVideoCapture);
	}

	/**
	 * 本地视频混流后推流 [当前不支持]
	 * <br> 把多个视频源混音后推出
	 * @param mixCaptureList 需要混音的视频源
	 * @see YYLiveVideoCapture
	 */
	public void setMixVideoCapture(ArrayList<YYLiveVideoCapture> mixCaptureList) {
		if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
			return;
		}
	}

	/**
	 * 初始化录屏
	 * @param yyLiveScreenCapture 录屏对象
	 */
	public void initSreenLiveSession(YYLiveScreenCapture yyLiveScreenCapture) {
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return;
		}
		if (yyLiveScreenCapture != null) {
			this.attachVideoCapture(yyLiveScreenCapture);
		}
	}

	/**
	 * 设置开播角度（横屏/竖屏）
	 * @param orientation
	 */
	public void setOrientation(int orientation){
		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
			return;
		}
		YYLiveNative.setOrientation(orientation);
	}


	public void setAudioSourceType(int mode)
	{
		YYLiveNative.setAudioSourceType(mode);
	}

	public boolean enableInEarMonitor(boolean enable)
	{
		return YYLiveNative.enableInEarMonitor(enable);
	}

	/**
	 *设置不同音效模式
	 *
	 *@param mode  = YYConstant.YYSoundEffectMode.
	 *                     YYSOUND_EFFECT_MODE_NONE               = 关闭所有模式
	 *                     YYSOUND_EFFECT_MODE_VALLEY              = VALLEY模式
	 *                     YYSOUND_EFFECT_MODE_RANDB               = R&B模式
	 *                     YYSOUND_EFFECT_MODE_KTV                 = KTV模式
	 *                     YYSOUND_EFFECT_MODE_CHARMING           = CHARMING模式
	 *
	 */
	public void setSoundEffect(int mode) {
		YYLiveLog.release(YYLiveLog.kLogTagCall, "YYLivePublisher::setSoundEffect %d", mode);
		YYLiveNative.setSoundEffect(mode);
	}

	private int convertPlayType(int playType) {
		switch (playType) {
			case YYConstant.YYPublishPlayType.YYPUBLISH_PLAY_SINGLE:
				playType = YYPublishVideoConfig.YYPUBLISH_PLAYTYPE_SINGLE;
				break;
			case YYConstant.YYPublishPlayType.YYPUBLISH_PLAY_INTERACT:
				playType = YYPublishVideoConfig.YYPUBLISH_PLAYTYPE_LIANMAI;
				break;
			case YYConstant.YYPublishPlayType.YYPUBLISH_PLAY_SCREENCAP:
				playType = YYPublishVideoConfig.YYPUBLISH_PLAYTYPE_SCREEN_CAPTURE;
				break;
			default:
				break;
		}
		return playType;
	}
}
