#ifndef _YYAUDIO_CODEC_H_
#define _YYAUDIO_CODEC_H_

#include "int_type.h"

#ifdef AUDIO_CODEC_BUILD
# if defined(WIN32)
#   define YY_AUDIO_CODEC_EXPORT __declspec(dllexport)
# elif defined(__GNUC__)
#  define YY_AUDIO_CODEC_EXPORT __attribute__ ((visibility ("default")))
# else
#  define YY_AUDIO_CODEC_EXPORT
# endif
#else
#  define YY_AUDIO_CODEC_EXPORT
#endif

namespace YYAudioCodecV0
{
	enum CodecStatusCode
	{
		CodecStatusOk = 0,
		CodecGetOptionFail = -1,
		CodecEncoderInputSizeError = -2,
		CodecEncoderNotInitError = -3,
		CodecEncoderOutputSizeError = -4,
	};

	enum AudioCodecId
	{
		AudioCodecIdAAC = 0,
		AudioCodecIdSilk = 1,
		AudioCodecIdPCM = 2,
		AudioCodecIdOpus = 3,
		AudioCodecIdTotal,
	};

	enum AudioFileCodecId
	{
		AudioFileCodecIdUnknown = -1,
		AudioFileCodecIdWav = 0,
		AudioFileCodecIdMp3 = 1,
		AudioFileCodecIdAAC = 2,
		AudioFileCodecIdM4a = 3,
		AudioFileCodecIdMediaCodec = 4,  // android 
		AudioFileCodecIdIosExt = 5,
		AudioFileCodecIdIosHttp = 6,
		AudioFileCodecIdSilk = 7,
		AudioFileCodecIdOgg = 8,
	};

	enum AacTransType
	{
		AAC_TRANS_RAW = 0,
		AAC_TRANS_ADIF = 1,
		AAC_TRANS_ADTS = 2,
		AAC_TRANS_TOTAL
	};

	enum OutputMuxerType
	{
		OutputMuxer_RAW = 0,
		OutputMuxer_2BLH = 1,  // 2byte length header
		OutputMuxer_ADTS = 2,
		OutputMuxer_YY2BLH = 3,
		OutputMuxer_ADTS_V1 = 4, // demux data include adts header, mux is same
		OutputMuxer_TOTAL,
	};

	enum AacProfile
	{
		AacProfile_LCAAC = 0,
		AacProfile_HEAAC_V1 = 1,
		AacProfile_HEAAC_V2 = 2,
		AacProfile_ELDAAC = 3,
		AacProfile_TOTAL,
	};

	class YY_AUDIO_CODEC_EXPORT MuxerDataInfo
	{
	public:

		MuxerDataInfo();

		int sampleRate;
		int channel;
		int profile;
		int frameLen;
	};

	class YY_AUDIO_CODEC_EXPORT DemuxDataInfo
	{
	public:

		DemuxDataInfo();
		~DemuxDataInfo();

		void AddNewFrame(const char* dataIn, int length);
		void Clear();

		int frameCount;
		int* frameLen;
		char** dataPtr;
	};

	struct AuxiliaryData
	{
		char* ancOffset[8]; /* dse max size: 8 */
		int ancSize[8];
		int eleTag[8];
		int number;
	};

	enum AudioSideDataType
	{
		SIDE_DATA_AUXILIARY = 0,
		SIDE_DATA_TOTAL,
	};

	struct AudioSideData
	{
		AudioSideDataType sideDataType;
		void* data;
		int size;
	};

	class YY_AUDIO_CODEC_EXPORT AudioCodecFrame
	{
	public:

         AudioCodecFrame(char* data, int size);
         AudioCodecFrame();
         ~AudioCodecFrame();

		AudioSideData* NewSideData(AudioSideDataType sideDataType, int size);

		static const int kBufferSize = 9600;  // 50ms 48k,stereo pcm size

		char pcmPtr[kBufferSize];
		int  pcmSize;

		AudioSideData** sideDataPtr;
		int sideDataCount;
	};

	class YY_AUDIO_CODEC_EXPORT AudioCodecPacket
	{
	public:

         AudioCodecPacket(char* data, int size);
         AudioCodecPacket();

		static const int kBufferSize = 3840;  // 48k, stereo, 20ms pcm output

		char  packetPtr[kBufferSize];
		int   packetSize;
	};

	enum CodecSettingConfig
	{
		CodecConfigSampleRate = 0,
		CodecConfigChannel = 1,
		CodecConfigBitRate = 2,
		CodecConfigOutputMuxer = 3,
		CodecConfigProfile,
		CodecConfigVbr,
		CodecConfigAfterBurner,
		CodecConfigEnableSbr,
		CodecConfigTransType,
		CodecConfigFrameMs,
		CodecConfigAppMode,
		CodecConfigConcealMode,
		CodecConfigLimiterEnable,
		CodecConfigSignalMode,
		CodecConfigAsfData,
		CodecConfigSbrRatio,
        CodecConfigBandWidth,
		CodecConfigTotal,
    };

    const unsigned long dwASSEMBLE_CODEC_TYPE_BASE = 6;
    const unsigned long dwCUSTOM_CODEC_TYPE_BASE = 20;

    enum CodecRate {
        CODEC_NONE = -1,
        VOICE_MID = 0,//speex
        MUSIC_MID = 1,//aacplus
        VOICE_HIGHER = 2,//silk
        VOICE_HIGH = 3,//amrwb
        VOICE_HIGHER_JITTER = 4,
        MUSIC_MID_JITTER = 5,
        VOICE_MID_AND_MUSIC_MID = dwASSEMBLE_CODEC_TYPE_BASE + 1,	           // 7 			
        VOICE_HIGH_AND_MUSIC_MID = dwASSEMBLE_CODEC_TYPE_BASE + 2,			   // 8
        VOICE_HIGHER_AND_MUSIC_MID_JITTER = dwASSEMBLE_CODEC_TYPE_BASE + 3,    // 9
        VOICE_HIGHER_AND_MUSIC_MID = dwASSEMBLE_CODEC_TYPE_BASE + 4,		   // 10	
        VOICE_HIGHER_QUALITY1_AND_MUSIC_MID = dwASSEMBLE_CODEC_TYPE_BASE + 5,  // 11
        VOICE_LOW_FOR_MOBILE = dwCUSTOM_CODEC_TYPE_BASE + 1,                   // 21
        VOICE_HIGHER_QUALITY1 = dwCUSTOM_CODEC_TYPE_BASE + 2,                  // 22
        VOICE_LOW_FOR_MOBILE_QUALITY2 = dwCUSTOM_CODEC_TYPE_BASE + 3,          // 23
        MUSIC_MID_1_32KBR = dwCUSTOM_CODEC_TYPE_BASE + 4,                      // 24
        MUSIC_MID_1_40KBR = dwCUSTOM_CODEC_TYPE_BASE + 5,                      // 25
        MUSIC_MID_1_48KBR = dwCUSTOM_CODEC_TYPE_BASE + 6,                      // 26
        MUSIC_MID_2_24KBR = dwCUSTOM_CODEC_TYPE_BASE + 7,                      // 27
        MUSIC_MID_2_32KBR = dwCUSTOM_CODEC_TYPE_BASE + 8,                      // 28
        MUSIC_MID_2_40KBR = dwCUSTOM_CODEC_TYPE_BASE + 9,                      // 29
        MUSIC_MID_2_48KBR = dwCUSTOM_CODEC_TYPE_BASE + 10,                     // 30
        MUSIC_MID_2_24KBR_VBR = dwCUSTOM_CODEC_TYPE_BASE + 11,                 // 31
        MUSIC_MID_2_32KBR_VBR = dwCUSTOM_CODEC_TYPE_BASE + 12,                 // 32
        MUSIC_MID_2_40KBR_VBR = dwCUSTOM_CODEC_TYPE_BASE + 13,                 // 33
        MUSIC_MID_2_48KBR_VBR = dwCUSTOM_CODEC_TYPE_BASE + 14,                 // 34
        MUSIC_AAC_44100HZ_STEREO_128KBPS = dwCUSTOM_CODEC_TYPE_BASE + 15,      // 35
        MUSIC_AAC_44100HZ_STEREO_160KBPS = dwCUSTOM_CODEC_TYPE_BASE + 16,      // 36
        MUSIC_AAC_44100HZ_STEREO_192KBPS = dwCUSTOM_CODEC_TYPE_BASE + 17,      // 37
        MUSIC_AACELD_44100HZ_STEREO = dwCUSTOM_CODEC_TYPE_BASE + 18,           // 38 
        MUSIC_AACELDSBR_44100HZ_STEREO = dwCUSTOM_CODEC_TYPE_BASE + 19,        // 39 
        MUSIC_AAC_48000HZ_STEREO = dwCUSTOM_CODEC_TYPE_BASE + 20,              // 40
        MUSIC_AAC_44100HZ_MONO = dwCUSTOM_CODEC_TYPE_BASE + 21,                // 41 
        MUSIC_AAC_48000HZ_MONO = dwCUSTOM_CODEC_TYPE_BASE + 22,                // 42
        MUSIC_AACELD_16000HZ_MONO = dwCUSTOM_CODEC_TYPE_BASE + 23,             // 43
        MUSIC_AACELD_32000HZ_MONO = dwCUSTOM_CODEC_TYPE_BASE + 24,             // 44 
        VOICE_HIGHER_24000HZ_V1 = dwCUSTOM_CODEC_TYPE_BASE + 25,               // 45 
        VOICE_OPUS_48000HZ_MONO = dwCUSTOM_CODEC_TYPE_BASE + 30,               // 55 
        MUSIC_OPUS = dwCUSTOM_CODEC_TYPE_BASE + 60,                            // 80
    };

	class IAudioDecoder
	{
	public:
		virtual ~IAudioDecoder() {}

        virtual void SetCodecConfig(CodecSettingConfig config, void* data, int size) = 0;
        virtual int  GetCodecConfig(CodecSettingConfig config, void* data, int* size) = 0;
		virtual bool Init() = 0;
        virtual bool Uninit() = 0;
        virtual bool Flush() = 0;
		virtual int Decode(const AudioCodecPacket& packet, AudioCodecFrame& frame) = 0;
		virtual int DecodeLoss(AudioCodecFrame& frame) = 0;
		virtual int GetDecodeDelay() = 0;
		virtual void Destroy() = 0;
	};

	class IAudioEncoder
	{
	public:
		virtual ~IAudioEncoder() {}

		virtual void SetCodecConfig(CodecSettingConfig config, void* value, int size) = 0;
		virtual int  GetCodecConfig(CodecSettingConfig config, void* data, int* size) = 0;
		virtual bool Init() = 0;
        virtual bool Uninit() = 0;
        virtual bool Flush() = 0;
		virtual void Destroy() = 0;
		virtual int Encode(const AudioCodecFrame& frame, AudioCodecPacket& packet) = 0;
		virtual int EncodeLoss(AudioCodecPacket& packet) = 0;
		virtual int PcmBytesPerFrame() = 0;
		virtual int EncoderDelay() = 0;
	};

	class YY_AUDIO_CODEC_EXPORT AudioDemuxer
	{
	public:
		AudioDemuxer(OutputMuxerType muxerType, int bufferSize);
		~AudioDemuxer();

		int Parse(const char* header, const int headerLen, DemuxDataInfo& info);
		int Parse(const AudioCodecPacket& packet, DemuxDataInfo& info);
		int BufferSize() { return m_bufferSize; }
		int LeftSize() { return (m_bufferSize - m_usedSize); }
		void Clear();

	private:
		OutputMuxerType m_muxerType;
		int m_bufferSize;
		char* m_buffer;
		int m_usedSize;
	};

	class YY_AUDIO_CODEC_EXPORT AudioMuxer
	{
	public:
		AudioMuxer(OutputMuxerType muxerType);
		~AudioMuxer();

		bool AddMuxerInfo(MuxerDataInfo info, AudioCodecPacket& packet);
		OutputMuxerType MuxerType();

	private:

		void MuxerHeader(MuxerDataInfo info, char*& headerOut, int* headerLen);

		OutputMuxerType m_muxerType;
		char m_muxerHeader[50];
	};

	class IAudioFileDecoder
	{
	public:

        YY_AUDIO_CODEC_EXPORT static IAudioFileDecoder* Create(AudioFileCodecId codecId);

		virtual ~IAudioFileDecoder() {}
		virtual void Destroy() = 0;
		virtual bool Open(char* pFileName) = 0;
		virtual int GetPCMData(char* pData, int dataLen) = 0;
		virtual uint32_t Seek(uint32_t timeMS) = 0;
		virtual uint32_t GetTotalPlayLengthMS() = 0;
		virtual int GetSampleRate() = 0;
		virtual int GetChannelCount() = 0;
        virtual bool HasNextLoop() = 0;
		virtual void EnableLoopPlay(bool enable, int cycle) = 0;
		virtual void Flush() = 0;
		virtual uint32_t GetCurrentPositionMS() = 0;
		virtual bool setOutputFormat(int sampleRate, int channel) = 0;
		virtual bool SelectAudioTrack(int audioTrack) = 0;
		virtual int GetAudioTrackCount() = 0;
	};

	enum FileCodecType
    {
        FileCodecWAV = 0,
        FileCodecLCAAC = 1,
        FileCodecMp3 = 2,
        FileCodecM4a = 3,
		FileCodecSilk = 4,
		FileCodecAACPLUS = 5,
		FileCodecELDAAC = 6,
    };

    enum FileCodecConfig
    {
        FileCodecSampleRate = 0,
        FileCodecChannel = 1,
        FileCodecBitRate = 2,
        FileCodecMp3Id3Tag = 3,
		FileCodecNeedEncBytes = 4,
		FileCodecTimePerEncodedFrame = 5,
		FileCodecSignalMode = 6,
		FileCodecProfile = 7,
		FileCodecTransType = 8,
		FileCodecVbr = 9,
		FileCodecAfterBurner = 10,
		FileCodecEnableSbr = 11,
		FileCodecSbrRatio = 12,
    };

    class IAudioFileEncoder
    {
    public:

        YY_AUDIO_CODEC_EXPORT static IAudioFileEncoder* Create(FileCodecType codecId);

        virtual ~IAudioFileEncoder() {}
		virtual int Init() = 0;
        virtual void Destroy() = 0;
		virtual void Flush() = 0;
        virtual int WriteData(const AudioCodecFrame &frame, AudioCodecPacket& packet) = 0;
        virtual void SetCodecConfig(FileCodecConfig config, void* data, int size) = 0;
        virtual int  GetCodecConfig(FileCodecConfig config, void* data, int* size) = 0;
    };

    class AudioCodecMgr
    {
    public:
        virtual ~AudioCodecMgr() {}

		YY_AUDIO_CODEC_EXPORT static int GetSampleRateFromCodecId(int codecId);
		YY_AUDIO_CODEC_EXPORT static int GetChannelFromCodecId(int codecId);
		YY_AUDIO_CODEC_EXPORT static int GetBitrateFromCodecId(int codecId);
		YY_AUDIO_CODEC_EXPORT static int GetFrameMsFromCodecId(int codecId);

        YY_AUDIO_CODEC_EXPORT static int GetFrameBytesFromCodecId(int codecId);


        YY_AUDIO_CODEC_EXPORT static bool IsFixedParameterCodecId(int codecId);
        YY_AUDIO_CODEC_EXPORT static bool IsSupportedCodecId(int codecId);
        YY_AUDIO_CODEC_EXPORT static int GetSupportedCodecIdNumber();
        YY_AUDIO_CODEC_EXPORT static int GetAllSupportedCodecId(int* codecIdArray, int arraySize);
        YY_AUDIO_CODEC_EXPORT static int GetProfileFromCodecId(int codecId);

		YY_AUDIO_CODEC_EXPORT static IAudioEncoder* CreateEncoder(int codecId);
		YY_AUDIO_CODEC_EXPORT static IAudioDecoder* CreateDecoder(int codecId);

		YY_AUDIO_CODEC_EXPORT static IAudioEncoder* CreateEncoder(AudioCodecId id);
		YY_AUDIO_CODEC_EXPORT static IAudioDecoder* CreateDecoder(AudioCodecId id);
        
    };

    YY_AUDIO_CODEC_EXPORT void SelectIOSDecoder(bool isIosDecoder);
    YY_AUDIO_CODEC_EXPORT void SelectIOSEncoder(bool isIosEncoder);
}

#endif
