package com.thunder.livesdk.helper;

import java.math.BigDecimal;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class UidMap {
    private final static String TAG = "UidMap";
    private static Map<Long, String> mUidToStringMap = new HashMap<Long, String>();

    /**
     * 添加uid32 -> string uid映射关系
     *
     * @param iUid   无符号32位uid
     * @param strUid 字符串uid
     */
    public static void addUid2String(long iUid, String strUid) {
        ThunderLog.release(TAG, String.format("addUid2String %d,%s", iUid, strUid));
        removeUid2String(strUid);
        removeUid2String(iUid);

        synchronized (UidMap.class) {
            mUidToStringMap.put(iUid, strUid);
        }
    }

    /**
     * 通过32位uid获取字符串uid。优先查询映射表，如果在映射表中查询不到，则将32位uid直接转成字符串
     *
     * @param iUid 无符号32位uid
     * @return 字符串uid
     */
    public static String getStringUid(long iUid) {
        String strUid;
        synchronized (UidMap.class) {
            strUid = mUidToStringMap.get(iUid);
        }

        if (strUid == null || strUid.isEmpty()) {
            strUid = Long.toString(iUid);
        }

        return strUid;
    }

    /**
     * 通过32位uid获取字符串uid。只查询映射表，如果查询不到则返回空
     *
     * @param iUid 无符号32位uid
     * @return 字符串uid，不存在则返回空
     */
    public static String getStringUidFromMap(long iUid) {
        String strUid;
        synchronized (UidMap.class) {
            strUid = mUidToStringMap.get(iUid);
        }
        return strUid;
    }

    /**
     * 通过字符串uid获取32位uid
     *
     * @param strUid 字符串uid
     * @return 无符号32位uid
     */
    public static long getLongUid(String strUid) {
        long iUid = 0;

        synchronized (UidMap.class) {
            if (strUid != null && strUid.length() != 0) {
                Iterator<Long> iter = mUidToStringMap.keySet().iterator();
                Long key;
                while (iter.hasNext()) {
                    key = iter.next();
                    if (strUid.equals(mUidToStringMap.get(key))) {
                        iUid = key;
                        break;
                    }
                }
            }
        }

        if (iUid == 0) {
            if (isNumeric(strUid)) {
                try {
                    iUid = Long.parseLong(strUid, 10);
                } catch (NumberFormatException exception) {
                    iUid = 0;
                }
            }
        }

        return iUid;
    }

    /**
     * 删除32位uid对应的映射关系
     *
     * @param iUid 无符号32位uid
     */
    public static void removeUid2String(long iUid) {
        synchronized (UidMap.class) {
            mUidToStringMap.remove(iUid);
        }
    }

    /**
     * 删除字符串uid对应的映射关系
     *
     * @param strUid 字符串uid
     */
    public static void removeUid2String(String strUid) {
        synchronized (UidMap.class) {
            if (strUid != null && strUid.length() != 0) {
                Iterator<Long> iter = mUidToStringMap.keySet().iterator();
                Long key;
                while (iter.hasNext()) {
                    key = iter.next();
                    if (strUid.equals(mUidToStringMap.get(key))) {
                        mUidToStringMap.remove(key);
                        break;
                    }
                }
            }
        }
    }

    /**
     * 删除所有uid映射关系
     */
    public static void removeAllUid2String() {
        synchronized (UidMap.class) {
            mUidToStringMap.clear();
        }
    }

    public static boolean isNumeric(String strNum) {
        Pattern pattern = Pattern.compile("^\\d+$");
        String bigStr;
        try {
            bigStr = new BigDecimal(strNum).toString();
        } catch (Exception e) {
            return false;
        }

        Matcher isNum = pattern.matcher(bigStr);
        if (!isNum.matches()) {
            return false;
        }
        return true;
    }
}