package com.thunder.livesdk;

import android.graphics.Bitmap;

import com.yy.mediaframework.gpuimage.custom.OrangeFilterWrapper;

import java.util.ArrayList;
import java.util.HashMap;

/**
 * Created by aoe on 2018/11/22.
 */

public interface ThunderPublishRoom {
    /**
     * 开启视频预览
     *
     * @param playType    玩法 {@link ThunderConstant.ThunderPublishPlayType}
     * @param publishMode 开播模式 {@link ThunderConstant.ThunderPublishVideoMode}
     * @return 0：成功；<0：error
     * @discussion 预览实际效果与开播上行视频源流效果一致
     * 如分辨率 摄像头前置/后置
     */
    int startVideoPreview(int playType, int publishMode);

    /**
     * 停止视频预览
     *
     * @return 0：成功；<0：error
     */
    int stopVideoPreview();

    /**
     * @param layout 连麦后主播在所有直播间的的布局和最终混画输出分辨率
     * @return YYCommonErrCode
     * @brief 开始直播间连麦
     * @discussion 调用此接口前需要先joinChannel否则无效
     * 跨频道连麦时候需先调用startSinglePublish开始推送音视频后再调用此接口连麦
     * 同频道连麦时候可选择先调用startSinglePublish开始推送音视频后再调用此接口连麦
     * 或者直接调用此接口 sdk内部会默认开启startSinglePublish逻辑上行音视频
     * <p>
     * layout outputSize为连麦后最终混画输出的分辨率
     * layouts (比如用户A和B连麦)
     * 跨频道连麦时主播A在自己频道A的布局和在B频道内的布局信息 同理主播B填在自己频道B的布局和在A频道内的布局信息 outputSize为混画后的分辨率
     * 同频道连麦时候主播A在自己频道A布局信息 观众B填在主播频道A的布局 outputSize为混画后的分辨率 观众可以不填outputSize
     * speakers为所有连麦的uid(uidA uidB)
     * <p>
     * 连麦结果通知kYYLiveAPINotification_PublishChannelStatus
     * 连麦主播源流通知kYYLiveAPINotification_PublishChannelStreamInfo
     * 主播可以遍历streams找到连麦主播的源流 通过startPlayStreams订阅对方主播音视频流
     */
    int startLiveInteract(ThunderLayout layout);

    /**
     * 停止直播间连麦
     *
     * @return 0：成功；<0：error
     * @discussion 跨频道连麦时候将恢复单人开播模式继续推音视频直播
     * 同频道连麦时候主播将恢复单人开播模式继续推音视频直播 观众则停止推音视频流
     */
    int stopLiveInteract();

    /**
     * 开始直播音频连麦
     *
     * @param speakers 所有连麦者的uid
     * @return 0：成功；<0：error
     * @discussion 调用此接口前需要先joinChannel否则无效
     * 连麦观众只上行音频 主播和观众只有音频混合视频仍为直播源视频
     * <p>
     * 连麦结果通知kYYLiveAPINotification_PublishChannelResult
     * 主播可以遍历streams找到连麦主播的源流 通过startPlayStreams订阅对方主播音视频流
     */
    int startAudioInteract(ArrayList<String> speakers);

    /**
     * 停止直播音频连麦
     *
     * @return 0：成功；<0：error
     * @discussion 同频道连麦时候主播将恢复单人开播模式继续推音视频直播 观众则停止推音频流
     */
    int stopAudioInteract();

    /**
     * 开始单人直播
     *
     * @param playType    玩法 {@link ThunderConstant.ThunderPublishPlayType}
     * @param publishMode 开播模式 {@link ThunderConstant.ThunderPublishVideoMode}
     * @return 0：成功；<0：error
     * @discussion 根据开播玩法和模式决定分辨率、码率、帧率等信息
     * 开播结果通知kYYLiveAPINotification_PublishChannelResult
     */
    int startSinglePublish(int playType, int publishMode);

    /**
     * 停止直播
     *
     * @return 0：成功；<0：error
     * @discussion 停止音视频推流 如果有连麦则自动停止连麦
     */
    int stopSinglePublish();

    /**
     * 设置预览视图
     *
     * @param view 预览View {@link ThunderPreviewView}
     * @return
     */
    int setLocalVideoView(ThunderPreviewView view);

    /**
     * 更新开播模式
     *
     * @param publishMode 开播模式 {@link ThunderConstant.ThunderPublishVideoMode}
     * @return 0：成功；<0：error
     * @discussion 根据开播模式变更动态改变音视频流上行分辨率、码率、帧率等而不需要重新开播
     */
    int updatePublishMode(int publishMode);

    /**
     * 获取玩法对应支持的开播模式
     *
     * @param playType 玩法 {@link ThunderConstant.ThunderPublishPlayType}
     * @return 开播模式数组 KEY {@link ThunderConstant.ThunderPublishVideoMode}
     */
    HashMap<Integer, String> getPublishMode(int playType);

    /**
     * 获取玩法对应的默认开播模式
     *
     * @param playType 玩法 {@link ThunderConstant.ThunderPublishPlayType}
     * @return 默认开播模式 {@link ThunderConstant.ThunderPublishVideoMode}
     */
    int getDefaultPublishMode(int playType);

    /**
     * 获取当前开播配置信息
     */
    ThunderPublishInfo getCurrentPublishInfo();

    /**
     * 摄像头位置
     *
     * @param position 摄像头位置 {@link ThunderConstant.ThunderCameraPosition}
     * @return 成功or失败
     */
    boolean setCameraPosition(int position);

    /**
     * 设置开播角度（横屏/竖屏）
     *
     * @param orientation {@link ThunderConstant.ThunderPublishOrientation}
     */
    void setPublishOrientation(int orientation);

    /**
     * 设置开播视频打水印
     *
     * @param posX  水印在视频图像中的位置
     * @param posY  水印在视频图像中的位置
     * @param image 水印图片
     * @return 0：成功；<0：error
     */
    int setWatermark(int posX, int posY, Bitmap image);

    /**
     * 设置美颜级别
     *
     * @param level 取值范围[0, 1]
     * @return 0：成功；<0：error
     */
    int setFaceBeautyLevel(float level);

    /**
     * 设置开播音量
     * <br>实际采集到的音量通过监听通知{@link ThunderNotification#kThunderAPINotification_AudioCaptureVolume} 获取
     *
     * @param volume 音量值，取值范围[0，100]
     * @return 成功or失败
     */
    boolean setMicVolume(int volume);

    /**
     * 获取瘦脸接口,在startPreview之后调用
     */
    OrangeFilterWrapper getBeautyFilter();

    /**
     * 外部采集音频推流 用录屏开播场景时设置
     * <br>通过此接口导入外部音频，设置为YYLiveDefaultMic对象或设置为null，SDK会调用设备的麦克风进行采集
     *
     * @param capture 音频采集对象
     * @see ThunderDefaultMic
     */
    void attachAudioCapture(ThunderAudioCapture capture);

    /**
     * 外部采集视频推流 用录屏开播场景时设置YYLiveScreenCapture
     * <br>通过此接口导入外部视频，设置为YYLiveDefaultCamera对象或设置为null，SDK会调用设备的摄像头进行采集
     *
     * @param capture 视频采集对象
     * @see ThunderDefaultCamera
     */
    void attachVideoCapture(ThunderVideoCapture capture);

    /**
     * channelId
     */
    void setChannelId(String channelId);

    /**
     * 同步用户直播间信息 内部传递
     */
    void syncUserInfo(long uid, int userRole);

    int startCapture();

    int stopCapture();

    /**
     * 开播/停播音频
     *
     * @param enableAudio true：开播 false：停播
     */
    void setAudioEnable(boolean enableAudio);

    /**
     * 开播/停播视频
     *
     * @param enableVideo true：开播 false：停播
     */
    void setVideoEnable(boolean enableVideo);

    /**
     * 更新玩法和开播模式
     *
     * @param playType    玩法
     * @param publishMode 开播模式
     * @return <0 失败， =0成功
     */
    int updatePlayTypeAndPublishMode(int playType, int publishMode);

    /**
     * 更新channelProfile和audioProfile信息
     *
     * @param channelProfile=ThunderRtcConstant.RoomConfig THUNDER_ROOMCONFIG_LIVE = 直播模式（流畅）
     *                                                     THUNDER_ROOMCONFIG_COMMUNICATION = 通话模式（延时低）
     *                                                     THUNDER_ROOMCONFIG_GAME = 游戏（省流量、延时低）
     * @param audioProfile                                 音质
     * @param commutMode                                   交互模式 ThunderRtcConstant.CommutMode
     * @param scenarioMode                                 ThunderRtcConstant.ScenarioMode
     */
    void updateProfile(int channelProfile, int audioProfile, int commutMode, int scenarioMode);
}
