package com.thunder.livesdk;

import com.thunder.livesdk.helper.ThunderLog;
import com.thunder.livesdk.helper.ThunderNative;
import com.thunder.livesdk.video.IVideoDecodeObserver;
import com.thunder.livesdk.video.ThunderVideoLogCallback;
import com.yy.videoplayer.utils.YMFLog;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * Created by xiongxiong on 2017/10/30.
 */

public class ThunderPlayer {

    public ArrayList<ThunderStream> subscribeStreams = null;
    private ConcurrentLinkedQueue<ThunderStream> highStreams = null;
    private ConcurrentLinkedQueue<ThunderStream> lowStreams = null;
    private IHighLowStreamChangeObserver streamChangeObserver = null;
    private final Object streamLock = new Object();
    private long lowStreamArriveTime = 0;
    // 大小流强制切换超时时间 8 秒
    private final long highLowStreamSwitchTimeout = 8000;

    ThunderPlayer() {
        if (BuildConfig.__YY_VIDEO_SUPPORT__) {
            YMFLog.registerLogger(ThunderVideoLogCallback.sharedInstance());
        }
        subscribeStreams = new ArrayList<ThunderStream>(0);
        highStreams = new ConcurrentLinkedQueue<ThunderStream>();
        lowStreams = new ConcurrentLinkedQueue<ThunderStream>();
    }

    /**
     * 播放流
     * <p>
     * <br>拉流播放前使用API {@link ThunderAPI#setUserRole(int)} 设置用户角色
     * <br>拉流播放状态通过监听通知{@link ThunderNotification#kThunderAPINotification_PlayStatus}获取
     * <br>播放实时信息通过监听通知{@link ThunderNotification#kThunderAPINotification_PlayRunTimeInfo}获取
     * <br>1. 填入跟自己角色匹配的流，例如：纯观众播放混画/转码的流，主播可以播放多条独立原流。
     * <br>2. 可以同时播放streamList和groupList里面的所有流
     * <br>3. 每个群组里的所有流都会被播放
     * <br>4. 相同speakerUid的音频和视频配对进行同步
     * </p>
     *
     * @param streamList 播放的流列表
     * @param groupList  播放的群组列表
     * @return 0：成功；<0：error
     */
    public int startPlayStreams(ArrayList<ThunderStream> streamList, ArrayList<ThunderGroup> groupList) {
        if (streamList == null) {
            streamList = new ArrayList<ThunderStream>(0);
        }
        if (groupList == null) {
            groupList = new ArrayList<ThunderGroup>(0);
        }

        if (streamList.size() == 0 && groupList.size() == 0) {
            ThunderLog.warn(ThunderLog.kLogTagCall,
                    "startPlayStreams both stream list and group list is empty");
            return -1;
        }

        for (ThunderStream oneStream : streamList) {
            if (oneStream.bVideo) {
                oneStream.startSubscribeTime = System.currentTimeMillis();
                synchronized (streamLock) {
                    //确保上次订阅信息已被清除，防止重新订阅时取得的订阅时间时上次的没有更新，强制更新订阅时间
                    Iterator<ThunderStream> iter = subscribeStreams.iterator();
                    while (iter.hasNext())
                    {
                        ThunderStream rmStream = iter.next();
                        if (rmStream.streamName.equals(oneStream.streamName) &&
                           rmStream.appId == oneStream.appId)
                        {
                            iter.remove();
                        }
                    }
                    subscribeStreams.add(oneStream);
                }

                ThunderLog.release(ThunderLog.kLogTagCall,
                        "startPlayStreams video streamName " + oneStream.streamName + "spkUid:" + oneStream.speakerUid +
                                "toVideo: " + oneStream.toView);
            }
        }


        ThunderLog.release(ThunderLog.kLogTagCall, "startPlayStreams list size: %d %d",
                streamList.size(), groupList.size());
        return ThunderNative.startPlayStreams(streamList, groupList);
    }

    /**
     * 停止播放流
     * streamList，groupList 里的所有流都会被停止，不需要跟YYUserRole匹配
     *
     * @param streamList 流列表
     * @param groupList  群组列表
     * @return 0：成功；<0：error
     */
    public int stopPlayStreams(ArrayList<ThunderStream> streamList, ArrayList<ThunderGroup> groupList) {
        if (streamList == null) {
            streamList = new ArrayList<ThunderStream>(0);
        }
        if (groupList == null) {
            groupList = new ArrayList<ThunderGroup>(0);
        }

        if (streamList.size() == 0 && groupList.size() == 0) {
            ThunderLog.warn(ThunderLog.kLogTagCall,
                    "stopPlayStreams both stream list and group list is empty");
            return -1;
        }

        for (ThunderStream oneStream : streamList) {
            if (oneStream.bVideo) {
                synchronized (streamLock) {

                    Iterator<ThunderStream> iter = subscribeStreams.iterator();
                    while (iter.hasNext())
                    {
                        ThunderStream rmStream = iter.next();
                        if (rmStream.streamName.equals(oneStream.streamName)
                                && rmStream.appId == oneStream.appId)
                        {
                            iter.remove();
                        }
                    }
                }
            }
        }

        ThunderLog.release(ThunderLog.kLogTagCall, "stopPlayStreams list size: %d %d",
                streamList.size(), groupList.size());
        return ThunderNative.stopPlayStreams(streamList, groupList);
    }

    /**
     * 保存大小流切换信息
     *
     * @param high 源流
     * @param low   目标流
     */
    public void setHighLowStream(ArrayList<ThunderStream> high, ArrayList<ThunderStream> low) {
        if (high == null || low == null) {
            return;
        }
        lowStreamArriveTime = 0;
        highStreams.clear();
        lowStreams.clear();
        highStreams.addAll(high);
        lowStreams.addAll(low);
        String highStreamName = "";
        for (ThunderStream stream : highStreams) {
            highStreamName = stream.streamName;
        }
        String lowStreamName = "";
        for (ThunderStream stream : lowStreams) {
            lowStreamName = stream.streamName;
        }
        ThunderLog.release(ThunderLog.kLogTagCall, "setHighLowStream high :" + highStreamName + " low " + lowStreamName);
    }

    public long getHighLowStreamUid() {
        for (ThunderStream stream : highStreams) {
            return stream.speakerUid;
        }
        return -1;
    }

    public void clearHighLowStream() {
        highStreams.clear();
        lowStreams.clear();
    }

    public boolean isHighStream(String streamKey) {
        for (ThunderStream stream : highStreams) {
            if (streamKey.equals(stream.packStreamKey())) {
                return true;
            }
        }
        return false;
    }

    public boolean isLowStream(String streamKey) {
        for (ThunderStream stream : lowStreams) {
            if (streamKey.equals(stream.packStreamKey())) {
                return true;
            }
        }
        return false;
    }

    public String getHighStreamName(String streamKey) {
        for (ThunderStream stream : highStreams) {
            if (streamKey.equals(stream.packStreamKey())) {
                return stream.streamName;
            }
        }
        return null;
    }

    public String getLowStreamName(String streamKey) {
        for (ThunderStream stream : lowStreams) {
            if (streamKey.equals(stream.packStreamKey())) {
                return stream.streamName;
            }
        }
        return null;
    }

    public void notifyLowStreamArrive() {
        lowStreamArriveTime = System.currentTimeMillis();
        ThunderLog.release(ThunderLog.kLogTagCall, "notifyLowStreamArrive time :" + lowStreamArriveTime);
    }

    public boolean forceSwitch() {
        long currentTime = System.currentTimeMillis();
        if (lowStreamArriveTime != 0 && currentTime - lowStreamArriveTime >= highLowStreamSwitchTimeout) {
            lowStreamArriveTime = 0;
            ThunderLog.release(ThunderLog.kLogTagCall, "high low stream forceSwitch true ." );
            return true;
        }
        return false;
    }

    /**
     * 判断当前是否存在大小流需要切换
     * @return true：存在大小流切换  false：不存在大小流切换
     */
    public boolean isNeedHighLowStreamChange() {
        return (highStreams.size() > 0 && lowStreams.size() > 0);
    }

    public void onHighLowStreamChangeFinish(String highStreamName, String lowStreamName) {
        if (streamChangeObserver != null) {
            streamChangeObserver.onHighLowStreamChangeFinish(highStreamName, lowStreamName);
        }
    }

    /**
     * 是否使用硬解，播放视频之前调用
     *
     * @param enable 开启or关闭，默认开启
     * @return 成功or失败
     */
    public boolean enableHardwareDecoder(boolean enable) {
        ThunderLog.release(ThunderLog.kLogTagCall, "enableHardwareDecoder:%b", enable);
        return ThunderNative.enableHardwareDecoder(enable,
                ThunderRtcConstant.ThunderVideoEncodeType.THUNDERVIDEO_ENCODE_TYPE_H264);
    }

//  更新渲染模式使用接口 YYLivePlayerView.setScaleMode
//	/**
//	 * 设置视频流显示模式, 一般在全屏切换等流尺寸可能变化的场合时调用
//	 * @param view      播放视频的view
//	 * @param scaleMode 显示模式
//	 * @param stream    视频流, 这个接口只取流名称、appId和主播uid
//	 * @return 成功or失败
//	 */
//	public boolean updatePlayVideoView(Object view, int scaleMode, ThunderStream stream) {
//		if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
//			return false;
//		}
//		if (stream == null || view == null) {
//			return false;
//		}
//
//		ThunderLog.release(ThunderLog.kLogTagCall, "updatePlayVideoView scale mode %d for stream %s",
//				scaleMode, stream.streamName);
//
//		return ThunderNative.updatePlayVideoView(view, scaleMode, stream);
//	}

    /**
     * 开启静音
     *
     * @param enable 开启or关闭，默认关闭
     * @param stream 音频流
     * @return 成功or失败
     */
    public boolean enableMute(boolean enable, ThunderStream stream) {
        if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
            return false;
        }

        if (stream == null) {
            return false;
        }

        ThunderLog.release(ThunderLog.kLogTagCall,
                "enableMute %b for stream %s", enable, stream.streamName);

        return ThunderNative.enableMuteStream(enable, stream);
    }

    /**
     * 开启静音
     *
     * @param enable 开启or关闭，默认关闭
     * @param uid    string型uid
     * @return 成功or失败
     */
    public boolean enableMute(boolean enable, String uid) {
        if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
            return false;
        }

        ThunderLog.release(ThunderLog.kLogTagCall,
                "enableMute %b for %s", enable, uid);

        return ThunderNative.enableMuteStringUid(enable, uid);
    }

    /**
     * 全部静音
     */
    public boolean enableAllMute(boolean enable) {
        if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
            return false;
        }

        ThunderLog.release(ThunderLog.kLogTagCall,
                "enableAllMute %b", enable);

        ThunderNative.enableAllMute(enable);

        return true;
    }

    /**
     * 设置播放音量， 实际播放的音量通过监听通知{@link ThunderNotification#kThunderAPINotification_AudioPlayVolume}获取
     *
     * @param volume 音量值，取值范围[0,100]
     * @param stream 音频流
     * @return 成功or失败
     */
    public boolean setPlayVolume(int volume, ThunderStream stream) {
        if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
            return false;
        }
        if (stream == null) {
            return false;
        }

        if (volume < 0) {
            volume = 0;
        } else if (volume > 100) {
            volume = 100;
        }

        ThunderLog.release(ThunderLog.kLogTagCall,
                "setPlayVolume %d for stream %s", volume, stream.streamName);

        return ThunderNative.setStreamPlayVolume(volume, stream);
    }

    /**
     * 设置远端用户声音的空间位置和音量
     *
     * @param azimuth 设置远端用户声音出现的位置，取值范围[-90,90]， 0：（默认）声音出现在正前方, -90：声音出现在左边, 90：声音出现在右边
     * @param gain    设置远端用户声音的音量，取值范围[0,100], 默认值为 100.0，表示该用户的原始音量。取值越小，则音量越低
     * @return 成功or失败
     */
    public boolean setRemoteUidVoicePosition(int azimuth, int gain, ThunderStream stream) {
        if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
            return false;
        }
        if (stream == null) {
            return false;
        }

        if (azimuth < -90 || azimuth > 90 || gain < 0 || gain > 100) {
            ThunderLog.error(ThunderLog.kLogTagCall, "setRemoteUidVoicePosition fail, parameters error!");
            return false;
        }

        ThunderLog.release(ThunderLog.kLogTagCall,
                "setRemoteStreamVoicePosition azimuth：%d, gain: %d, for stream %s", azimuth, gain, stream.streamName);

        return ThunderNative.setRemoteUidVoicePosition(azimuth, gain, stream);
    }

    /**
     * 设置扬声器音量
     *
     * @param volume
     * @return
     */
    public boolean setSpeakerVolume(int volume) {
        if (!BuildConfig.__YY_AUDIO_SUPPORT__) {
            return false;
        }

        if (volume < 0) {
            volume = 0;
        } else if (volume > 100) {
            volume = 100;
        }

        ThunderLog.release(ThunderLog.kLogTagCall, "setSpeakerVolume %d", volume);

        return ThunderNative.setSpeakerVolume(volume);
    }

    /**
     * 切换扬声器模式和听筒模式，SDK默认为扬声器模式，此接口不影响耳机的使用，插入耳机时，声音均通过耳机播放
     *
     * @param enable true - 扬声器模式; false - 听筒模式
     */
    public void enableLoudSpeaker(boolean enable) {
        ThunderLog.release(ThunderLog.kLogTagCall, "enableLoudSpeaker %b", enable);
        ThunderNative.enableLoudSpeaker(enable);
    }

    /**
     * 获取扬声器状态
     *
     * @return
     */
    public boolean getLoudSpeakerEnabled() {
        ThunderLog.release(ThunderLog.kLogTagCall, "getLoudSpeakerEnabled");
        return ThunderNative.getLoudSpeakerEnabled();
    }

    public boolean isSupportPlayH265() {
        return ThunderNative.getH265SupportMode() > 0;
    }

    public void setSubscribeStreamWHByUid(long uid, int _width, int _height) {
        synchronized (streamLock) {
            for (ThunderStream oneStream : subscribeStreams) {
                if (oneStream.bVideo && oneStream.speakerUid == uid) {
                    oneStream.width = _width;
                    oneStream.height = _height;
                }
            }
        }
    }

    public ThunderStream getSubscribeStreamByUid(long uid) {
        synchronized (streamLock) {
            for (ThunderStream oneStream : subscribeStreams) {
                if (oneStream.bVideo && oneStream.speakerUid == uid) {
                    return oneStream;
                }
            }
        }
        return null;
    }

    public void removeSubscribeStreamByUid(long uid) {
        ThunderStream findStream = null;
        synchronized (streamLock) {
            for (ThunderStream oneStream : subscribeStreams) {
                if (oneStream.bVideo && oneStream.speakerUid == uid) {
                    findStream = oneStream;
                }
            }
            if (findStream != null) {
                subscribeStreams.remove(findStream);
            }
        }
    }

    public void setVideoFrameObserver(String uid, IVideoDecodeObserver observer) {
        ThunderNative.setVideoFrameObserver(uid, observer);
    }

    public void setHighLowStreamChangeObserver(IHighLowStreamChangeObserver observer) {
        streamChangeObserver = observer;
    }

    public interface IHighLowStreamChangeObserver {
        void onHighLowStreamChangeFinish(String highStreamName, String lowStreamName);
    }
}
