package com.thunder.livesdk;

public interface ThunderEventHandler {

    /**
     * sdk错误信息回调(error定义待定)
     *
     * @param error
     */
    public void onError(int error);

    /**
     * sdk告警信息回调
     *
     * @param warningCode {@link ThunderRtcConstant.ThunderWarningCode}
     */
    public void onWarning(int warningCode);

    /**
     * 加入频道回调
     *
     * @param room
     * @param uid
     * @param elapsed 参数功能未实现
     */
    public void onJoinRoomSuccess(String room, String uid, int elapsed);

    /**
     * 离开频道回调
     *
     * @param status 功能暂未实现
     */
    public void onLeaveRoom(RoomStats status);

    /**
     * 业务鉴权结果
     */
    public void onBizAuthResult(boolean bPublish, int result);

    /**
     * sdk鉴权结果
     *
     * @param result (0-鉴权成功 10000-服务器内部错误，可以重试 10001-没有带token，需要调用[YYLiveAPI updateToken:]
     *               10002-token校验失败（数字签名不对），可能使用的appSecret不对
     *               10003-token中appid跟鉴权时带的appid不一致 10004-token中uid跟鉴权时带的uid不一致 10005-token已过期
     *               10006-app不存在，没有在管理后台注册 10007-token即将过期 10008-用户被封禁
     *               {@link ThunderRtcConstant.AuthResult}
     */
    public void onSdkAuthResult(int result);

    /**
     * 用户被封禁回调
     *
     * @param status 封禁状态 ture=封禁 false=解禁
     */
    public void onUserBanned(boolean status);

    /**
     * 远端用户加入当前频道回调
     *
     * @param uid     远端用户/主播 ID
     * @param elapsed 从本地用户调用 joinRoom 到触发该回调的延迟（毫秒）
     */
    public void onUserJoined(String uid, int elapsed);

    /**
     * 远端用户离开当前频道回调
     *
     * @param uid    远端用户/主播 ID
     * @param reason (1-用户主动离开 2-因过长时间收不到对方数据包，超时掉线 3-用户身份从主播切换为观众（直播模式下))
     *               {@link ThunderRtcConstant.UserOfflineReason}
     */
    public void onUserOffline(String uid, int reason);

    /**
     * Token 服务即将过期回调
     *
     * @param token 即将服务失效的Token
     */
    public void onTokenWillExpire(byte[] token);

    /**
     * Token 已过期回调
     */
    public void onTokenRequested();

    /**
     * 网络上下行质量报告回调, 返回的是ID对应的用户的网络质量
     *
     * @param uid       表示该回调报告的是持有该 ID 的用户的网络质量, 当 uid 为 0 时，返回的是本地用户的网络质量
     * @param txQuality 网络上行质量(0-质量未知 1-质量极好 2-用户主观感觉和极好差不多，但码率可能略低于极好
     *                  3-用户主观感受有瑕疵但不影响沟通 4-勉强能沟通但不顺畅 5-网络质量非常差，基本不能沟通
     *                  6-网络连接已断开，完全无法沟通)
     *                  {@link ThunderRtcConstant.NetworkQuality}
     * @param rxQuality 网络下行质量(0-质量未知 1-质量极好 2-用户主观感觉和极好差不多，但码率可能略低于极好
     *                  3-用户主观感受有瑕疵但不影响沟通 4-勉强能沟通但不顺畅 5-网络质量非常差，基本不能沟通
     *                  6-网络连接已断开，完全无法沟通)
     *                  {@link ThunderRtcConstant.NetworkQuality}
     */
    public void onNetworkQuality(String uid, int txQuality, int rxQuality);

    /**
     * 当前用户上下行流量通知
     *
     * @param stats 流量通知详细属性{@link ThunderNotification.RoomStats}
     */
    public void onRoomStats(ThunderNotification.RoomStats stats);

    /**
     * 说话声音音量提示回调
     *
     * @param speakers    用户音量和采集时间戳
     * @param totalVolume (混音后的)总音量
     *                    默认关闭，开关：setAudioVolumeIndication
     */
    public void onPlayVolumeIndication(ThunderEventHandler.AudioVolumeInfo[] speakers, int totalVolume);

    /**
     * 采集音量回调
     *
     * @param totalVolume 上行音量能量值[0-100]
     * @param cpt         采集时间戳
     * @param micVolume   仅麦克风采集的音量能量值[0-100]
     *                    默认关闭，开关：enableCaptureVolumeIndication
     */
    public void onCaptureVolumeIndication(int totalVolume, int cpt, int micVolume);

    /**
     * 功能暂未实现
     *
     * @param uid
     * @param quality
     * @param delay
     * @param lost
     */
    public void onAudioQuality(String uid, int quality, short delay, short lost);

    /**
     * 功能暂未实现
     */
    public void onConnectionLost();

    /**
     * 功能暂未实现
     */
    public void onConnectionInterrupted();

    /**
     * 功能暂未实现
     *
     * @param routing
     */
    public void onAudioRouteChanged(int routing);

    /**
     * 音频播放数据回调
     *
     * @param data
     * @param data     解码前数据
     * @param cpt      采集时间戳
     * @param pts      播放时间戳
     * @param uid      用户id
     * @param duration 时长
     *                 默认关闭，开关：enableAudioDataIndication
     */
    public void onAudioPlayData(byte[] data, long cpt, long pts, String uid, long duration);

    /**
     * 音频播放频谱数据回调
     *
     * @param data 数值范围[0-100]
     *             默认关闭，开关：enableAudioPlaySpectrum
     */
    public void onAudioPlaySpectrumData(byte[] data);

    /**
     * 音频采集数据回调
     *
     * @param data       采集的pcm数据
     * @param dataSize   数据大小
     * @param sampleRate 数据的采样率
     * @param channel    数据的声道数
     *                   默认关闭，开关：enableCapturePcmDataCallBack
     */
    public void onAudioCapturePcmData(byte[] data, int dataSize, int sampleRate, int channel);

    /**
     * 音频渲染数据回调
     *
     * @param data       采集的pcm数据
     * @param dataSize   数据大小
     * @param sampleRate 数据的采样率
     * @param channel    数据的声道数
     * @param duration   数据时长 单位：ms
     *                   默认关闭，开关：enableRenderPcmDataCallBack
     */
    public void onAudioRenderPcmData(byte[] data, int dataSize, long duration, int sampleRate, int channel);

    /**
     * 接收到的透传协议消息回调
     *
     * @param data 透传消息
     * @param uid  发该消息的uid
     */
    public void onRecvUserAppMsgData(byte[] data, String uid);

    /**
     * 透传协议发送失败状态回调
     *
     * @param status 失败状态(1-频率太高 2-发送数据太大 3-未成功开播)
     *               目前规定透传频率2次/s,发送数据大小限制在<=200Byte
     */
    public void onSendAppMsgDataFailedStatus(int status);

    /**
     * 远端用户音频流开启/停止通知
     *
     * @param uid  用户uid
     * @param stop ture：停止 false：开启
     */
    public void onRemoteAudioStopped(String uid, boolean stop);

    /**
     * 远端用户视频流开启/停止通知
     *
     * @param uid  用户uid
     * @param stop ture：停止 false：开启
     */
    public void onRemoteVideoStopped(String uid, boolean stop);

    /**
     * 已显示远端视频首帧回调
     *
     * @param uid     用户id
     * @param width   视频尺寸 - 宽
     * @param height  视频尺寸 - 高
     * @param elapsed 从调用 [joinRoom](#joinRoom) API 到回调该事件的耗时（ms）
     */
    public void onRemoteVideoPlay(String uid, int width, int height, int elapsed);

    /**
     * 本地或远端视频分辨率改变回调
     *
     * @param uid      用户id
     * @param width
     * @param height
     * @param rotation
     */
    public void onVideoSizeChanged(String uid, int width, int height, int rotation);

    /**
     * 已发送本地音频首帧的回调
     *
     * @param elapsed 从本地用户调用 joinRoom 方法直至该回调被触发的延迟（毫秒）
     */
    void onFirstLocalAudioFrameSent(int elapsed);

    /**
     * 已发送本地视频首帧的回调
     *
     * @param elapsed 从本地用户调用 joinRoom 方法直至该回调被触发的延迟（毫秒）
     */
    void onFirstLocalVideoFrameSent(int elapsed);

    /**
     * cdn推流结果回调
     *
     * @param url       推流的目标url
     * @param errorCode 推流错误码  {@link ThunderRtcConstant.ThunderPublishCDNErrorCode}
     */
    void onPublishStreamToCDNStatus(String url, int errorCode);

    /**
     * 网络连接类型变化回调
     *
     * @param type 网络连接类型 {@link ThunderRtcConstant.ThunderNetworkType}
     */
    void onNetworkTypeChanged(int type);

    /**
     * 与服务器网络连接状态回调(service和avp)
     * thunder模式下，只看跟avp的连接状态
     * thunderbolt模式下且没有开播或订阅（不会连接avp）只看service连接状态
     * thunderbolt模式下且有开播或订阅，同时看service和avp的连接状态
     *
     * @param status {@link ThunderRtcConstant.ThunderConnectionStatus}
     */
    void onConnectionStatus(int status);

    /**
     *通话中远端视频流信息回调
     *
     * @param uid 远端用户/主播id
     * @param stats 远端流信息，参见{@link RemoteVideoStats}
     */
    public void onRemoteVideoStatsOfUid(String uid, RemoteVideoStats stats);

    /**
     * 通话中远端音频流传输信息回调
     *
     * @param uid 远端用户id，指定是哪个用户/主播的音频包传输信息
     * @param delay 音频包从发送端到接收端的延时（毫秒）
     * @param lost 音频包从发送端到接收端的丢包率（%）
     * @param rxKBitRate 远端音频包的接收码率（kbps)
     */
    public void onRemoteAudioTransStatsOfUid(String uid, int delay, int lost, int rxKBitRate);

    /**
     * 通话中远端视频流传输信息回调
     *
     * @param uid 远端用户id，指定是哪个用户/主播的视频包传输信息
     * @param delay 视频包从发送端到接收端的延时（毫秒）
     * @param lost 视频包从发送端到接收端的丢包率（%）
     * @param rxKBitRate 远端视频包的接收码率（kbps)
     */
    public void onRemoteVideoTransStatsOfUid(String uid, int delay, int lost, int rxKBitRate);

    public static class AudioVolumeInfo {
        public String uid;
        public int volume;
        public int pts;

        public AudioVolumeInfo() {
        }
    }

    public static class RoomStats {
        public int duration; // 通话时长（秒）
        public int txBytes; // 发送字节数（bytes）
        public int rxBytes; // 接收字节数（bytes）
        public int txKBitRate; // 发送码率（kbps）
        public int rxKBitRate; // 接收码率（kbps）
        public int txAudioKBitRate; // 音频发送码率 (kbps)
        public int rxAudioKBitRate; // 音频接收码率 (kbps)
        public int txVideoKBitRate; // 视频发送码率 (kbps)
        public int rxVideoKBitRate; // 视频接收码率 (kbps)
        public float cpuAppUsage; // 当前应用程序的 CPU 使用率 (%)
        public float cpuTotalUsage; // 当前系统的 CPU 使用率 (%)

        public RoomStats() {
        }
    }

    public static class RemoteVideoStats {
        public int delay; // 远端视频推流到播放的延时
        public int width; // 远端视频流宽度
        public int height; // 远端视频流高度
        public int receivedBitrate; // 接收码率，单位为Kbps
        public int decoderOutputFrameRate; // 远端视频解码器的输出帧率，单位为fps
        public int rendererOutputFrameRate; // 远端视频渲染器的输出帧率，单位为fps
        public int packetLossRate; // 远端视频在网络对抗之后的丢包率(%)
        public int rxStreamType; // 视频流类型，大流或小流
        public int totalFrozenTime; // 远端用户在加入频道后发生视频卡顿的累计时长(ms)
        public int frozenRate; // 远端用户在加入频道后发生视频卡顿的累计时长占视频总有效时长的百分比(%)
    }

}
