package com.thunder.livesdk;

/**
 * Created by xiongxiong on 2017/10/30.
 */

import android.content.Context;
import android.os.Build;
import android.os.Handler;

import com.yy.argo.Argo;
import com.yy.gslbsdk.HttpDnsService;
import com.yy.mediaframework.VideoLibAPI;
import com.yy.platform.baseservice.YYServiceCore;
import com.yy.videoplayer.decoder.YYVideoLibMgr;
import com.thunder.livesdk.audio.IAudioFrameObserver;
import com.thunder.livesdk.helper.ThunderLog;
import com.thunder.livesdk.helper.ThunderNative;
import com.thunder.livesdk.system.ThunderForeBackgroundListener;
import com.thunder.livesdk.system.ThunderNetStateService;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;

/**
 * YY直播sdk API入口
 * 所有接口，int型返回值，未做特殊说明，0表示成功，<0 error
 */

public class ThunderAPI {

    private ThunderPlayer mPlayer = null;
    private ThunderPublisher mPublisher = null;
    private ThunderPublishRoom mPubRoom = null;
    private ThunderNetStateService mNetStateService = null;
    private ThunderForeBackgroundListener mForeBackgroundListener = null;
    private String mVersion = null;
    private int mVersionInt = 0;
    private ThunderNative.NotificationDispatcher mNotificationDispatcher = null;
    private boolean mIsInited = false;
    private boolean m32BitUid = false;

    private HashSet<String> mSrcStreamUrlSet = new HashSet<>();
    private HashMap<String, HashSet<String>> mTranscodingStreamUrlMap = new HashMap<>();
    private HashMap<String, LiveTranscodingParams> mTranscodingParamsMap = new HashMap<>();
    private HashMap<String, HashSet<String>> mTranscodingStreamNotify = new HashMap<>();

    private HashMap<String, String> mSubscribeUid2RoomId = new HashMap<>();
    private String mRoomId = null;

    private ThunderAPI() {
        if (BuildConfig.__YY_VIDEO_SUPPORT__) {
            // Android 4.2.2 系统通过依赖加载JNI有问题，所以这里主动加载
            try {
                System.loadLibrary("yyvideoplayer");
            } catch (Throwable e) {
                e.printStackTrace();
                //ThunderLog.error(ThunderLog.kLogTagCall, "load yyvideoplayer failed!");
            }
        }

        try {
            System.loadLibrary("thunder");
        } catch (Throwable e) {
            e.printStackTrace();
            //load lib失败调用日志模块会崩溃
            //ThunderLog.error(ThunderLog.kLogTagCall, "load thunder failed!");
            return;
        }

        mPlayer = new ThunderPlayer();
        mPublisher = new ThunderPublisher();
        mPubRoom = new ThunderPublishRoomImp();
    }

    //内部类实现单例，由JVM保证线程安全
    private static class SingletonHolder {
        private static ThunderAPI INSTANCE = new ThunderAPI();
    }

    /**
     * sdk实例，使用单例，需要调用 initWithAppId 初始化
     *
     * @return sdk实例对象
     */
    public static ThunderAPI sharedInstance() {
        if (SingletonHolder.INSTANCE == null) {
            SingletonHolder.INSTANCE = new ThunderAPI();
        }
        return SingletonHolder.INSTANCE;
    }

    /**
     * 是否打开日志
     * <br>
     * <br>默认开启日志，日志输出到logcat
     * <br>若开启了日志且 callback != null 日志通过代理回调，sdk不写文件，否则写入${logPath}/thunder.txt
     *
     * @param enable   是否开启日志， 默认true
     * @param level    允许输出的最低日志级别 {@link ThunderConstant.ThunderLogLevel}
     * @param logPath  保存日志文件的目录路径
     * @param callback 日志回调
     * @see ThunderConstant.ThunderLogLevel
     * @see IThunderLogCallback
     */
    public static void enableLog(boolean enable, int level, String logPath, IThunderLogCallback callback) {
        sharedInstance(); //cause load native library
        ThunderNative.enableLog(enable, callback, level, logPath);
        ThunderLog.release(ThunderLog.kLogTagCall, "enableLog enable:%b level:%d",
                enable, level);
    }

    /**
     * sdk版本号
     *
     * @return sdk版本号，非空
     */
    public String getVersion() {
        return mVersion;
    }

    /**
     * sdk版本号
     *
     * @return sdk版本号，整形
     */
    public int getVersionInt() {
        return mVersionInt;
    }

    /**
     * 初始化sdk，所有非静态方法都要求初始化后才能调用
     * <br>
     * <br>在yylivesdk.yy.com上创建app后，系统会分配唯一appId，在app管理页面可以自己配置应用内sceneId
     * <br>可以通过 <code>ThunderAPI.sharedInstance().setSceneId(100)</code> 修改sceneId, 初始化后appId不能修改
     *
     * @param appId      接入app的唯一标识，无符号32位数
     * @param sceneId    用来标识app内不同应用场景，无符号32位数
     * @param appContext Andoird APP 的上下文
     * @return 成功或失败
     * @see ThunderAPI#setSceneId(long)
     */
    public boolean initWithAppId(long appId, long sceneId, Context appContext, Handler handler) {
        if (mIsInited) {
            return false;
        }

        ThunderLog.release(ThunderLog.kLogTagCall, "init appId:%d sceneId:%d appContext:%s",
                appId, sceneId, appContext.toString());

        if (mNetStateService != null) {
            mNetStateService.fini();
        }
        if (mForeBackgroundListener != null) {
            mForeBackgroundListener.fini();
        }

        mNotificationDispatcher = new ThunderNative.NotificationDispatcher() {
            private final HashSet<Handler> mNotificationHandlers = new HashSet<>();

            @Override
            public void registerNotificationHandler(Handler handler) {
                synchronized (mNotificationHandlers) {
                    mNotificationHandlers.add(handler);
                }
            }

            @Override
            public void unregisterNotificationHandler(Handler handler) {
                synchronized (mNotificationHandlers) {
                    mNotificationHandlers.remove(handler);
                }
            }

            @Override
            public Object[] collectNotificationHandlers() {
                Object[] handlers = null;
                synchronized (mNotificationHandlers) {
                    if (mNotificationHandlers.size() > 0) {
                        handlers = mNotificationHandlers.toArray();
                    }
                }
                return handlers;
            }
        };

        //必须在Service初始化前register notification
        registerNotificationHandler(handler);

        Argo.init(appContext);
        HttpDnsService.getService(appContext, "thundersdk", null,
                "", Locale.getDefault().getCountry());
        YYServiceCore.loadLib(appContext);

        ThunderDeviceInfo info = new ThunderDeviceInfo(appContext);

        if (BuildConfig.__YY_VIDEO_SUPPORT__) {
            //播放库初始化，必须比YYLiveNative先
            YYVideoLibMgr.instance().init(appContext, "1.0", "YYLiveDemo",
                    Build.VERSION.RELEASE, Build.MODEL, null);
        }

        int ret = ThunderNative.init(appId, sceneId, appContext, info, mNotificationDispatcher);

        if (ret < 0) {
            ThunderLog.error(ThunderLog.kLogTagCall, "init failed!");
            return false;
        }

        mNetStateService = new ThunderNetStateService(appContext);
        mForeBackgroundListener = new ThunderForeBackgroundListener(appContext);

        mNetStateService.init();
        mForeBackgroundListener.init();

        if (BuildConfig.__YY_VIDEO_SUPPORT__) {
            //开播库初始化
            VideoLibAPI.instance().initVideoLib(appId, sceneId, appContext);
            //播放库初始化
            YYVideoLibMgr.instance().init(appContext, "1.0", "YYLiveDemo",
                    Build.VERSION.RELEASE, Build.MODEL, null);

        }

        mVersion = ThunderNative.getVersion();
        mVersionInt = ThunderNative.getVersionInt();

        //需要在YYLiveNative初始化之后
        mIsInited = true;

        ThunderLog.release(ThunderLog.kLogTagCall, "init succeeded, version %s", mVersion,
                " versionInt %u", mVersionInt);
        return true;
    }

    /**
     * 停止sdk的接口函数， app在退出的时候， 需要调用sdk的停止函数， 不然可能造成不可预知的崩溃等.
     */
    public void deInit() {
        if (!mIsInited) {
            return;
        }

        mIsInited = false;
        if (mNetStateService != null) {
            mNetStateService.fini();
        }
        if (mForeBackgroundListener != null) {
            mForeBackgroundListener.fini();
        }

        ThunderNative.fini();
        mNetStateService = null;
        mForeBackgroundListener = null;
        mVersion = null;
        mVersionInt = 0;
        SingletonHolder.INSTANCE = null;
        ThunderLog.release(ThunderLog.kLogTagCall, "[call] ThunderAPI.deInit success");

        return;
    }

    /**
     * 注册通知处理器
     *
     * @param handler 消息处理器
     * @see ThunderNotification
     */
    public void registerNotificationHandler(Handler handler) {
        ThunderLog.release(ThunderLog.kLogTagCall, "registerNotificationHandler %s", handler.toString());
        mNotificationDispatcher.registerNotificationHandler(handler);
    }

    /**
     * 注册通知处理器
     *
     * @param handler 消息处理器
     */
    public void unregisterNotificationHandler(Handler handler) {
        ThunderLog.release(ThunderLog.kLogTagCall, "unregisterNotificationHandler %s", handler.toString());
        mNotificationDispatcher.unregisterNotificationHandler(handler);
    }

    /**
     * 设置用户国家区域
     * <br>
     * <br>初始化后设置国家区域用于连接Ap服务器
     * <br>(最好不要中途切换会有重连服务器延迟!) 默认连国内服务器
     *
     * @param areaType 角色 {@link ThunderConstant.ThunderAreaType}
     * @see ThunderConstant.ThunderAreaType
     */
    public void setAreaType(int areaType) {
        ThunderLog.release(ThunderLog.kLogTagCall, "setAreaType %d", areaType);
        ThunderNative.setAreaType(areaType);
    }

    /**
     * 设置vad模式
     *
     * @param enable 是否使能vad
     * @return
     */
    public void enableVad(boolean enable) {
        ThunderLog.release(ThunderLog.kLogTagCall, "enableVad %b", enable);
        ThunderNative.enableVad(enable);
    }

    /**
     * 设置sceneId，同一app使用不同sceneId标识不同业务场景，方便分业务出质量统计报表
     *
     * @param sceneId 场景id，无符号32位数
     */
    public void setSceneId(long sceneId) {
        ThunderLog.release(ThunderLog.kLogTagCall, "setSceneId %d", sceneId);
        ThunderNative.setSceneId(sceneId);
    }

    /**
     * 更新token，用于appid和业务功能鉴权
     * <br> token 格式
     * <br> -------------------------------------------------------------------------------------------------
     * <br> | TokenLen(uint16)    | AppId(uint32)        | uid (uint64)          | Timestamp(uint64)(ms)    |
     * <br> -------------------------------------------------------------------------------------------------
     * <br> | ValidTime(uint32)(s)| BizExtInfoLen(uint16)| BizExtInfoData(nBytes)| DigitalSignature(20Bytes)|
     * <br> -------------------------------------------------------------------------------------------------
     * <br> 1. 多字节整数使用网络字节序
     * <br> 2. TokenLen：整个token长度，包括自身2字节和摘要
     * <br> 3. 过期时间=Timestamp+ValidTime*1000, UTC时间距离1970的毫秒数
     * <br> 4. BizExtInfoData: 业务鉴权可能需要的扩展信息，透传
     * <br> 5. DigitalSignature：采用hmac-sha1算法对DigitalSignature前的所有数据运算得出[TokenLen,BizExtInfoData], 秘钥使用申请appId时分配的appSecret
     * <br> 6. 整个token 进行 url安全的 base64编码
     *
     * @param token 业务服务器生成的token，需要url base64编码
     *              <p>
     *              通过监听通知 kThunderAPINotification_BizAuthRes, kYYLiveAPINotification_SdkAuthRes获取鉴权结果
     *              {@link ThunderNotification#kThunderAPINotification_BizAuthRes}
     *              {@link ThunderNotification#kThunderAPINotification_SdkAuthRes}
     */
    public void updateToken(byte[] token) {
        if (token == null) {
            return;
        }
        ThunderLog.release(ThunderLog.kLogTagCall, "updateToken sToken %d", token.length);
        ThunderNative.updateToken(token, null);
    }

    /**
     * 开启音视频服务
     *
     * @param uid 用户id，无符号32位数
     */
    public void joinMedia(long uid) {
        ThunderLog.release(ThunderLog.kLogTagCall, "joinMedia %d", uid);
        ThunderNative.joinMedia(uid, 0, 0);
    }

    /**
     * 开启音视频服务，有需要指定频道的场景使用此接口
     *
     * @param uid    用户id，无符号32位数
     * @param sid    频道id，无符号32位数
     * @param subSid 子频道id，无符号32位数
     */
    public void joinMedia(long uid, long sid, long subSid) {
        ThunderLog.release(ThunderLog.kLogTagCall, "joinMedia %d %d %d", uid, sid, subSid);
        ThunderNative.joinMedia(uid, sid, subSid);
    }

    /**
     * 开启音视频服务，有需要指定频道的场景使用此接口
     *
     * @param uid    用户id，string型
     * @param sid    频道id，无符号32位数
     * @param subSid 子频道id，无符号32位数
     */
    public void joinMedia(String uid, long sid, long subSid) {
        String _uid = uid != null ? uid : new String();
        ThunderLog.release(ThunderLog.kLogTagCall, "joinMedia %s %d %d", _uid, sid, subSid);
        ThunderNative.joinMedia(_uid, sid, subSid);
    }

    /**
     * 退出音视频服务
     */
    public void leaveMedia() {
        ThunderLog.release(ThunderLog.kLogTagCall, "leaveMedia");
        ThunderNative.leaveMedia();
    }

    /**
     * 加入直播间
     * 观众系统主动拉取直播间内的所有流 通知kYYLiveAPINotification_StreamInfoStatus
     *
     * @param roomId   直播间号
     * @param uid      用户uid
     * @param userRole 用户角色 {@link ThunderConstant.ThunderUserRole}
     */
    public boolean joinRoom(String roomId, String uid, int userRole, boolean is32BitUid) {
        if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
            return false;
        }
        if (roomId == null || uid == null) {
            ThunderLog.info(ThunderLog.kLogTagCall, "joinRoom param error");
            return false;
        }
        mRoomId = roomId;

        ThunderLog.info(ThunderLog.kLogTagCall, "joinRoom %s %s %d", roomId, uid, userRole);
        ThunderNative.joinChannel(roomId, uid, is32BitUid, userRole);
        return true;
    }

    /**
     * 退出直播间
     */
    public void leaveRoom() {
        if (!BuildConfig.__YY_VIDEO_SUPPORT__) {
            return;
        }

        resetParamsStatus();

        ThunderLog.info(ThunderLog.kLogTagCall, "leaveRoom");
        ThunderNative.leaveChannel();
    }

    public void set32BitUid(boolean is32Bit) {
        ThunderLog.release(ThunderLog.kLogTagCall, "set32BitUid %b", is32Bit);
        m32BitUid = is32Bit;
    }

    /**
     * 设置用户角色
     * <br>
     * <br>开播或观看前需要设置用户角色
     * <br>若观众跟主播连麦，则自己也变成了主播
     * <br>连麦前需要设置角色为主播，结束后设置为观众
     *
     * @param role 角色 {@link ThunderConstant.ThunderUserRole}
     * @see ThunderConstant.ThunderUserRole
     */
    public void setUserRole(int role) {
        ThunderLog.release(ThunderLog.kLogTagCall, "setUserRole %d", role);
        ThunderNative.setUserRole(role);
    }

    /**
     * 开启低延时模式
     * <br>连麦场景需要开启，以获得更及时的交互体验
     * <br>纯观看场景建议关闭，以获得更稳定流畅的播放
     * <br>在{@link ThunderPlayer#startPlayStreams(ArrayList, ArrayList)},
     * {@link ThunderPublisher#startPublishVideo(String, ArrayList, ThunderPublishVideoConfig)},
     * {@link ThunderPublisher#startPublishAudio(String, ArrayList, ThunderPublishAudioConfig)} 前调用
     *
     * @param enable 开启or关闭
     */
    public void enableLowLatency(boolean enable) {
        ThunderLog.release(ThunderLog.kLogTagCall, "enableLowLatency %b", enable);
        ThunderNative.enableLowLatency(enable);
    }

    public void setRoomConfig(int profile) {
        ThunderLog.release(ThunderLog.kLogTagCall, "setRoomConfig %d", profile);
        ThunderNative.setRoomConfig(profile);
    }

    public void setPlayVolumeInterval(int interval, int moreThanThd, int lessThanThd) {
        ThunderNative.setPlayVolumeInterval(interval, moreThanThd, lessThanThd);
    }

    public void setCaptureVolumeInterval(int interval, int moreThanThd, int lessThanThd) {
        ThunderNative.setCaptureVolumeInterval(interval, moreThanThd, lessThanThd);
    }

    /**
     * 开始将音频数据保存成aac格式的文件
     *
     * @param fileName  文件保存路径,必须是完整路径，包括文件名，文件名的后缀必须是.aac。其中保存文件的文件夹必须是已经创建好的，该接口不提供创建文件夹的功能。例如：/sdcard/helloworld.aac
     * @param saverMode =ThunderRtcConstant.AudioSaverMode.
     *                  THUNDER_AUDIO_SAVER_ONLY_CAPTURE = 只保存频道内所有的上行的音频数据。上行的音频数据指的是：无论是主播的音频数据还是是伴奏的音频数据，只有被设置成上行，才被保存
     *                  THUNDER_AUDIO_SAVER_ONLY_RENDER = 保存频道内除主播以外的音频数据，例如伴奏的音频数据和观众端的音频数据
     *                  THUNDER_AUDIO_SAVER_BOTH = 保存频道内所有的音频数据
     * @param fileMode  = ThunderRtcConstant.AudioSaverWfMode.
     *                  THUNDER_AUDIO_SAVER_FILE_APPEND = 追加打开一个文本文件，并在文件末尾写数据
     *                  THUNDER_AUDIO_SAVER_FILE_OVERRIDE = 打开一个文本文件，写入的数据会覆盖文件本身的内容
     * @return true:成功 false:失败
     */
    public boolean startAudioSaver(String fileName, int saverMode, int fileMode) {
        return ThunderNative.startAudioSaver(fileName, saverMode, fileMode);
    }

    /**
     * 停止将音频数据保存成aac格式的文件
     *
     * @return true:成功 false:失败
     */
    public boolean stopAudioSaver() {
        return ThunderNative.stopAudioSaver();
    }

    /**
     * 获取开播相关接口
     *
     * @return 开播接口对象
     * @see ThunderPublisher
     */
    public ThunderPublisher getPublisher() {
        return this.mPublisher;
    }

    /**
     * 获取直播相关接口
     *
     * @return 直播接口对象
     * @see ThunderPublishRoom
     */
    public ThunderPublishRoom getPublishRoom() {
        return mPubRoom;
    }

    /**
     * 获取播放相关接口
     *
     * @return 播放接口对象
     * @see ThunderPlayer
     */
    public ThunderPlayer getPlayer() {
        return this.mPlayer;
    }

    /**
     * 设置外部音频处理器
     *
     * @param eap, eap是个C 函数指针数组，长度为6，每个函数的类型为：
     *             typedef void (*OnCaptureStartFun)(void*obj);
     *             typedef void (*OnCaptureDataFun)(void*obj, void* pData, uint32_t len, uint32_t sampleRate, uint32_t channelCount, uint32_t bitPerSample);
     *             typedef void (*OnCaptureStopFun)(void*obj);
     *             typedef void (*OnRenderStartFun)(void*obj);
     *             typedef void (*OnRenderDataFun)(void*obj, void* pData, uint32_t len, uint32_t sampleRate, uint32_t channelCount, uint32_t bitPerSample);
     *             typedef void (*OnRenderStopFun)(void*obj);
     */
    public void setExternalAudioProcessor(long eap) {
        ThunderLog.release(ThunderLog.kLogTagCall, "setExternalAudioProcessor %d", eap);
        ThunderNative.setExternalAudioProcessor(eap);
    }

    public void enableAudioPlaySpectrum(boolean enable) {
        ThunderNative.enableAudioPlaySepctrum(enable);
    }

    public void setAudioPlaySpectrumInfo(int spectrumLen, int notifyIntervalMS) {
        ThunderNative.setAudioPlaySepctrumInfo(spectrumLen, notifyIntervalMS);
    }

    public  void setPlayerMixerCompValue(int value){
        ThunderNative.setAudioPlayerMixerCompValue(value);
    }

    public void enableAudioDataIndication(boolean enable) {
        ThunderNative.enableAudioDataIndication(enable);
    }

    public void enableCapturePcmDataCallBack(boolean enable, int sampleRate, int channel) {
        ThunderNative.enableCapturePcmDataCallBack(enable, sampleRate, channel);
    }

    public boolean enableRenderPcmDataCallBack(boolean enable, int sampleRate, int channel) {
        if (sampleRate == -1 || sampleRate == 8000 || sampleRate == 16000 || sampleRate == 441000 ||
                sampleRate == 480000) {
            if (channel == -1 || channel == 1 || channel == 2) {
                ThunderNative.enableRenderPcmDataCallBack(enable, sampleRate, channel);
                return true;
            }
        }
        return false;
    }

    public void enableVoicePosition(boolean enable) {
        ThunderNative.enableVoicePosition(enable);
    }

    public void enableDebugLoopDelay(boolean enable){
        ThunderNative.enableDebugLoopDelay(enable);
    }

    /**
     * 用户发送透传消息
     *
     * @param msgData 发送的消息
     *                <br> 调用此接口条件:
     *                1、用户在频道内
     *                2、开麦成功后调用(纯观众和开播鉴权失败都不能发)
     *                3、调用该接口的频率每秒不能超过2次,msgData的大小不能超过200Byte
     *                <br> 不满足以上条件msg都会被丢弃
     *                <br> 通过监听通知kYYLiveAPINotification_AppMsgDataFailedStatus获得发送失败原因
     *                <br> 回调 kThunderAPINotification_UserAppMsgData kThunderAPINotification_AppMsgDataFailedStatus}
     */
    public void sendUserAppMsgData(byte[] msgData) {
        if (msgData == null) {
            return;
        }
        //ThunderLog.release(ThunderLog.kLogTagCall, "sendUserAppMsgData msgData %d", msgData.length);
        ThunderNative.sendUserAppMsgData(msgData);
    }

    /**
     * 透传argo上配的transsdk配置，不对外提供
     *
     * @param cfg
     */
    public void setArgoConfig(HashMap<Integer, Integer> cfg) {
        if (mIsInited) {
            ThunderNative.setArgoConfig(cfg);
        }
    }

    /**
     * 透传https二进制响应。此接口sdk内部使用，不对外提供
     *
     * @param reqUrl     请求地址
     * @param response   响应结果
     * @param statusCode 响应状态码
     * @param errCode    错误码
     * @param target     目标
     */
    public void setHttpsBinaryResponse(String reqUrl,
                                       byte[] response,
                                       int statusCode,
                                       int errCode,
                                       int target) {
        if (mIsInited) {
            ThunderNative.setHttpsBinaryResponse(reqUrl, response, statusCode, errCode, target);
        }
    }

    /**
     * 透传https文本响应。此接口sdk内部使用，不对外提供
     *
     * @param reqUrl     请求地址
     * @param response   响应结果
     * @param statusCode 响应状态码
     * @param errCode    错误码
     * @param target     目标
     */
    public void setHttpsTextResponse(String reqUrl,
                                     String response,
                                     int statusCode,
                                     int errCode,
                                     int target) {
        if (mIsInited) {
            ThunderNative.setHttpsTextResponse(reqUrl, response, statusCode, errCode, target);
        }
    }

    public int setPreviewRenderMode(int mode) {
        return ThunderNative.setPreviewRenderMode(mode);
    }

    /**
     * {@link IAudioFrameObserver} 接口对象实例；如果传入 null，则取消注册
     *
     * @param observer
     * @return
     */
    public int registerAudioFrameObserver(IAudioFrameObserver observer) {

        ThunderNative.registerAudioFrameObserver(observer);
        return 0;
    }

    /**
     * 设置回调采集数据的格式
     *
     * @param sampleRate     指定 onRecordAudioFrame 中返回数据的采样率，可设置为 8000，16000，32000，44100 或 48000
     * @param room           指定 onRecordAudioFrame 中返回数据的通道数，可设置为 1（单声道） 或 2（双声道）
     * @param mode           指定 onRecordAudioFrame 的使用模式:{@link ThunderRtcConstant.ThunderAudioRawFrameOperationMode}
     * @param samplesPerCall 指定 onRecordAudioFrame 中返回数据的采样点数，如 RTMP 推流应用中通常为 1024。
     * @return
     */
    public int setRecordingAudioFrameParameters(int sampleRate, int room, int mode, int samplesPerCall) {
        ThunderNative.setRecordingAudioFrameParameters(sampleRate, room, mode, samplesPerCall);
        return 0;
    }

    /**
     * 设置回调播放数据的格式
     *
     * @param sampleRate     指定 onPlaybackFrame 中返回数据的采样率，可设置为 8000，16000，32000，44100 或 48000
     * @param room           指定 onPlaybackFrame 中返回数据的通道数，可设置为 1（单声道） 或 2（双声道）
     * @param mode           指定 onPlaybackFrame 的使用模式:{@link ThunderRtcConstant.ThunderAudioRawFrameOperationMode}
     * @param samplesPerCall 指定 onPlaybackFrame 中返回数据的采样点数，如 RTMP 推流应用中通常为 1024。
     * @return
     */
    public int setPlaybackAudioFrameParameters(int sampleRate, int room, int mode, int samplesPerCall) {
        ThunderNative.setPlaybackAudioFrameParameters(sampleRate, room, mode, samplesPerCall);
        return 0;
    }

    /**
     * 更新源流推流地址
     *
     * @param bAdd true：添加地址  false：删除地址
     * @param url  推流地址
     * @return 添加/删除成功返回 0  添加/删除失败返回 < 0
     */
    public int updateSrcStreamUrlSet(boolean bAdd, String url) {
        boolean ret = false;
        if (bAdd) {
            ret = mSrcStreamUrlSet.add(url);
        } else {
            ret = mSrcStreamUrlSet.remove(url);
        }

        ThunderNative.updatePublishStreamUrl(mSrcStreamUrlSet);
        return (ret ? 0 : -1);
    }

    public int addSubscribe(String roomId, String uid) {
        mSubscribeUid2RoomId.put(uid, roomId);
        ThunderNative.subscribeBroadcast(roomId, uid, true);
        HashSet<String> rooms = new HashSet<>();
        rooms.add(roomId);
        ThunderNative.queryTransCodingChannelStream(rooms);
        return 0;
    }

    public int removeSubscribe(String roomId, String uid) {
        mSubscribeUid2RoomId.remove(uid);
        if (!isSetTranscodingTask(roomId)) {
            ThunderNative.subscribeBroadcast(roomId, uid, false);
        }
        return 0;
    }

    /**
     * 更新转码流推流地址
     *
     * @param taskId 转码任务id
     * @param bAdd   true：添加地址  false：删除地址
     * @param url    推流地址ThunderAPI
     * @return 添加/删除成功返回 0  添加/删除失败返回 < 0
     */
    public int updateTranscodingStreamUrlMap(String taskId, boolean bAdd, String url) {

        if (!isValidTranscoingTaskId(taskId)) {
            return -2;
        }
        boolean ret = false;
        HashSet<String> urlSet = mTranscodingStreamUrlMap.get(taskId);
        if (bAdd) {
            if (urlSet == null) {
                urlSet = new HashSet<>();
            }
            ret = urlSet.add(url);
            mTranscodingStreamUrlMap.put(taskId, urlSet);
        } else {
            if (urlSet == null) {
                return -3;
            }
            ret = mTranscodingStreamUrlMap.get(taskId).remove(url);
        }


        if (mTranscodingParamsMap.get(taskId) != null) {
            ret = true;

            mTranscodingParamsMap.get(taskId).transcodingStreamUrls = mTranscodingStreamUrlMap.get(taskId);

            ThunderLog.info(ThunderLog.kLogTagRtcEngine, "updateTranscodingStreamUrlMap user %s",
                    mTranscodingParamsMap.get(taskId).userToString());
            ThunderNative.updateTranscodingParams(mTranscodingParamsMap);
        }

        return (ret ? 0 : -1);
    }

    /**
     * 更新转码任务
     *
     * @param taskId      转码任务id
     * @param bRemove     tru：删除转码任务  false：添加or更新转码任务
     * @param transcoding 转码任务对象
     * @return 添加/删除成功返回 0  添加/删除失败返回 < 0
     */
    public int updateTranscodingMap(String taskId, boolean bRemove, LiveTranscoding transcoding) {
        if (!isValidTranscoingTaskId(taskId)) {
            return -2;
        }
        boolean ret = true;

        if (bRemove) {
            ret = mTranscodingParamsMap.containsKey(taskId);
            if (ret) {
                updateTranscodingQuerySubScribe(true, mTranscodingParamsMap.get(taskId).usersLayout);
                //mTranscodingParamsMap.remove(taskId);
                mTranscodingParamsMap.put(taskId, new LiveTranscodingParams());
                ThunderNative.updateTranscodingParams(mTranscodingParamsMap);
                mTranscodingParamsMap.remove(taskId);
            } else {
                ThunderLog.info(ThunderLog.kLogTagCall, "can not find taskId %s", taskId.toString());
                return -3;
            }
        } else {

            LiveTranscodingParams params = new LiveTranscodingParams();
            params.transcodingCofig = getTranscodingConfigJsonFromArgo(transcoding.getTransCodingMode());
            params.transcodingStreamUrls = mTranscodingStreamUrlMap.get(taskId);
            params.usersLayout = transcoding.getUsers();

            //判断是否减少了user
            if (mTranscodingParamsMap.containsKey(taskId)) {
                LiveTranscodingParams originParams = mTranscodingParamsMap.get(taskId);

                ArrayList<LiveTranscoding.TranscodingUser> removeUsers = new ArrayList<>();
                if (params.usersLayout == null || params.usersLayout.size() == 0) {
                    removeUsers.addAll(originParams.usersLayout);

                } else {

                    for (int i = 0; i < originParams.usersLayout.size(); ++i) {

                        LiveTranscoding.TranscodingUser oriUser = originParams.usersLayout.get(i);

                        int j = 0;
                        for (; j < params.usersLayout.size(); ++j) {

                            LiveTranscoding.TranscodingUser newUser = params.usersLayout.get(j);

                            if (newUser.roomId.equals(oriUser.roomId) &&
                                    newUser.uid.equals(oriUser.uid)) {
                                break;
                            }
                        }
                        if (j == params.usersLayout.size()) {
                            removeUsers.add(oriUser);
                        }
                    }

                }

                updateTranscodingQuerySubScribe(true, removeUsers);
            }


            updateTranscodingQuerySubScribe(false, params.usersLayout);
            ThunderLog.info(ThunderLog.kLogTagRtcEngine, "user layout %s", params.userToString());
            mTranscodingParamsMap.put(taskId, params);

            ThunderNative.updateTranscodingParams(mTranscodingParamsMap);
        }

        return (ret ? 0 : -1);
    }


    public void handleTrancodingQuerySubscribeStream(ThunderNotification.StreamsNotify streamInfo) {

        HashSet<String> querySubscribeStream = new HashSet<>();

        for (Map.Entry<String, HashSet<String>> entry : mTranscodingStreamNotify.entrySet()) {

            String key = entry.getKey();
            String[] uidAndChannel = key.split("@");
            String uid = uidAndChannel[0];
            String channelId = uidAndChannel[1];


            if (streamInfo.getRoomId().equals(channelId)) {

                for (ThunderStream stream : streamInfo.getStreams()) {
                    String strUid = stream.strUid.isEmpty() ? String.valueOf(stream.speakerUid) : stream.strUid;
                    if (uid.equals(strUid)) {

                        if (!entry.getValue().contains(stream.streamName)) {

                            querySubscribeStream.add(stream.streamName);
                            entry.getValue().add(stream.streamName);
                        }
                    }
                }
            }
        }


        if (!querySubscribeStream.isEmpty()) {
            ThunderNative.updateTranscodingUserStreams(querySubscribeStream);
        }

    }

    private void updateTranscodingQuerySubScribe(boolean bRemove,
                                                 ArrayList<LiveTranscoding.TranscodingUser> users) {
        String key = null;
        HashSet<String> channelSet = new HashSet<>();
        for (LiveTranscoding.TranscodingUser user : users) {

            String uid = user.uid;
            String channelId = user.roomId;
            key = uid + "@" + channelId;

            if (!bRemove) {

                if (mTranscodingStreamNotify.containsKey(key)) {
                    continue;
                }

                HashSet<String> streamName = new HashSet<>();
                mTranscodingStreamNotify.put(key, streamName);
                ThunderNative.subscribeBroadcast(channelId, uid, true);
                channelSet.add(channelId);

            } else {

                if (mTranscodingStreamNotify.containsKey(key)) {
                    mTranscodingStreamNotify.remove(key);
                    //如果和此channelId的频道有跨频道订阅关系，则不向service退订
                    if (!mSubscribeUid2RoomId.containsValue(channelId) && !channelId.equals(mRoomId)) {
                        ThunderNative.subscribeBroadcast(channelId, uid, false);
                    }
                }

            }
        }

        if (!channelSet.isEmpty()) {
            ThunderNative.queryTransCodingChannelStream(channelSet);
        }

    }

    private String getTranscodingConfigJsonFromArgo(int transcodinMode) {
        String ret = ThunderNative.getTranscodingCfgByMode(transcodinMode);
        if (ret != null && ret.length() > 0) {
            ThunderLog.info(ThunderLog.kLogTagRtcEngine, "getTranscodingConfigJsonFromArgo %s", ret.toString());
        }
        return ret;
    }

    private void resetParamsStatus() {
        mSrcStreamUrlSet.clear();
        mTranscodingStreamUrlMap.clear();
        mTranscodingParamsMap.clear();
        mTranscodingStreamNotify.clear();
    }

    private boolean isValidTranscoingTaskId(String taskId) {
        if (taskId == null || taskId.isEmpty()) {
            return false;
        }

        int length = taskId.length();
        if (length > 20) {    //length <= 20
            return false;
        }
        int codePoint;
        int validCharCount = 0;
        for (int i = 0; i < length; i++) {
            codePoint = Character.codePointAt(taskId, i);
            if ((codePoint >= 48 && codePoint <= 57)    // [0,9]
                    || (codePoint >= 65 && codePoint <= 90)    // [A,Z]
                    || (codePoint >= 97 && codePoint <= 122))  // [a,z]
            {
                validCharCount += 1;
            }
        }
        if (validCharCount == length) {
            return true;
        }
        return false;
    }

    //判断某个频道是否在参与混画任务
    private boolean isSetTranscodingTask(String roomId) {

        if (mTranscodingParamsMap != null && !mTranscodingParamsMap.isEmpty()) {
            for (Map.Entry<String, LiveTranscodingParams> entry : mTranscodingParamsMap.entrySet()) {
                LiveTranscodingParams params = entry.getValue();
                if (params != null) {
                    ArrayList<LiveTranscoding.TranscodingUser> userList = params.usersLayout;
                    if (userList != null && !userList.isEmpty()) {
                        for (LiveTranscoding.TranscodingUser user : userList) {
                            if (user.roomId.equals(roomId)) {
                                return true;
                            }
                        }
                    }
                }
            }
        }
        return false;
    }
}
