package com.hummer.im.model.completion;

import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.shared.DispatchQueue;

import java.util.LinkedList;
import java.util.List;

public final class RichCompletion {

    public RichCompletion(@NonNull String description) {
        Looper looper = Looper.myLooper();
        if (looper == Looper.getMainLooper()) {
            this.dispatchQueue = new DispatchQueue(new DispatchQueue.LooperHandlerProvider(Looper.getMainLooper()));
            Log.i(TAG, Trace.once().method("CompletionInit")
                    .info("main dispatchQueue", this.dispatchQueue));
        } else if (looper == HMRContext.work.getHandler().getLooper()) {
            // 回到work队列
            this.dispatchQueue = HMRContext.work;
            Log.i(TAG, Trace.once().method("CompletionInit")
                    .info("work dispatchQueue", this.dispatchQueue));
        } else {
            // 默认使用Direct分派
            this.dispatchQueue = DispatchQueue.direct;
            Log.i(TAG, Trace.once().method("CompletionInit")
                    .info("internal dispatchQueue", this.dispatchQueue));
        }

        this.description = DesPrefix + description;
        HMRContext.recorder.start("[async]" + this.description);
    }

    public RichCompletion(final HMR.Completion completion, @NonNull final String description) {
        this(description);

        if (completion == null) {
            // 如果接口调用方不传入completion，sdk无法进行回调，则会出现计时的异常情况
            // 针对这种情况，这里作特别处理
            HMRContext.recorder.stop("[async]" + this.description);
        }

        this.onSuccess(new OnSuccess() {
            @Override
            public void onSuccess() {
                if (completion != null) {
                    completion.onSuccess();
                }
            }
        }).onFailure(new OnFailure() {
            @Override
            public void onFailure(Error error) {
                if (completion != null) {
                    completion.onFailed(error);
                }
            }
        });
    }

    public RichCompletion(@NonNull Handler handler, @NonNull String description) {
        this.dispatchQueue = new DispatchQueue(new DispatchQueue.LooperHandlerProvider(handler.getLooper()));
        this.description = DesPrefix + description;
        HMRContext.recorder.start("[async]" + this.description);
        Log.i(TAG, Trace.once().method("CompletionInit")
                .info("custom dispatchQueue with handler", this.dispatchQueue));
    }

    public RichCompletion decorate(@Nullable RichCompletion decorated) {
        if (this.decorated == null) {
            Log.e(TAG, Trace.once().method("decorate")
                    .info("只允许对Completion进行一次装饰", this.decorated));
            return this;
        }

        this.decorated = decorated;
        return this;
    }

    public RichCompletion beforeSuccess(@NonNull Runnable runnable) {
        this.preSuccess.add(runnable);
        return this;
    }

    public RichCompletion onSuccess(@NonNull OnSuccess success) {
        this.successHandler = success;
        return this;
    }

    public RichCompletion afterSuccess(@NonNull Runnable runnable) {
        this.postSuccess.add(0, runnable);
        return this;
    }

    public RichCompletion beforeFailure(@NonNull Runnable runnable) {
        this.preFailure.add(runnable);
        return this;
    }

    public RichCompletion onFailure(@NonNull OnFailure failure) {
        this.failureHandler = failure;
        return this;
    }

    public RichCompletion afterFailure(@NonNull Runnable runnable) {
        this.postFailure.add(0, runnable);
        return this;
    }

    public String getDescription() {
        return this.description;
    }

    void dispatchSuccess() {
        Log.i(TAG, Trace.once().method("dispatchSuccess")
                .info("dispatchQueue", this.dispatchQueue));
        dispatchQueue.async(this.description, new Runnable() {
            @Override
            public void run() {
                dispatchActions(preSuccess);

                if (successHandler != null) {
                    Log.i(TAG, Trace.once().method("dispatchSuccess")
                            .msg("dispatchSuccess"));
                    successHandler.onSuccess();
                } else {
                    Log.i(TAG, Trace.once().method("dispatchSuccess")
                            .msg("dispatchSuccess, successHandler == null"));
                }

                CompletionUtils.dispatchSuccess(decorated);

                dispatchActions(postSuccess);
            }
        });
    }

    void dispatchFailure(@NonNull final Error error) {
        Log.i(TAG, Trace.once().method("dispatchFailure")
                .info("dispatchQueue", this.dispatchQueue));
        dispatchQueue.async(this.description, new Runnable() {
            @Override
            public void run() {
                dispatchActions(preFailure);

                if (failureHandler != null) {
                    Log.i(TAG, Trace.once().method("dispatchFailure")
                            .msg("dispatchFailure"));
                    failureHandler.onFailure(error);
                } else {
                    Log.i(TAG, Trace.once().method("dispatchFailure")
                            .msg("dispatchFailure, failureHandler == null"));
                }

                CompletionUtils.dispatchFailure(decorated, error);

                dispatchActions(postFailure);
            }
        });
    }

    private static void dispatchActions(@NonNull List<Runnable> actions) {
        for (Runnable r : actions) {
            r.run();
        }
    }

    private static final String TAG = "RichCompletion";
    private static final String DesPrefix = "RichCompletion::";

    private final DispatchQueue dispatchQueue;
    private List<Runnable> preSuccess = new LinkedList<>();
    private OnSuccess successHandler;
    private List<Runnable> postSuccess = new LinkedList<>();

    private List<Runnable> preFailure = new LinkedList<>();
    private OnFailure failureHandler;
    private List<Runnable> postFailure = new LinkedList<>();

    private RichCompletion decorated;
    private String description;
}
