package com.hummer.im.model.chat;

import com.hummer.im.Error;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.shared.CodecManager;
import com.hummer.im.model.chat.contents.Audio;
import com.hummer.im.model.chat.contents.Image;
import com.hummer.im.model.chat.contents.Text;
import com.hummer.im.model.chat.states.Preparing;

/**
 * ChatContent用于表示一个聊天内容类型，具体类型应该由其子类来实现。例如Text, Voice, Image等
 * @see Text
 * @see Audio
 * @see Image
 */
public class Content {

    /**
     * Preparable接口，如果一个具体的ChatContent需要一个较长耗时的处理过程，例如网络文件上传，压缩处理等操作，则应实现该接口
     */
    public interface Preparable {
        /**
         * 进行内容准备操作
         *
         * @param callback 操作的异步回调
         */
        void prepare(PreparingCallback callback);
    }

    /**
     * Prepare过程中的状态回调。
     * 用于通知业务方Prepare过程的状态变化
     * 业务方可以实现这些接口来执行事件。如发送成功，发送失败，更新UI，进度更新等
     */
    public interface PreparingCallback {
        /**
         * 内容准备的进度回调，像图片、语音的消息在上传时，会通告该回调告知其具体的上传进度
         *
         * @param progress 一个抽象的进度实例，通常为ScalarProgress类型，包含进度的具体数值。未来还可能是用于表示
         *                 准备阶段的具体类型。
         */
        void onPrepare(Preparing.Progress progress);

        /**
         * 内容准备成功
         */
        void onPrepareSuccess();

        /**
         * 内容准备失败
         *
         * @param err err携带了相关的错误上下文信息
         */
        void onPrepareFailed(Error err);
    }

    /**
     * 所有的Hummer消息都需要支持对应的内容编码、解码器，如果需要实现业务自定义消息类型，则需要实现下述Codec接口，并在业务初始化时
     * 调用registerCodec方法进行注册，否则Hummer无法正确处理该类型的消息。
     */
    public interface Codec {
        /**
         * 消息的类型标识值，业务在实现自定义消息时，应使用大于等于CustomTypeStart(10000)的取值，避免和Hummer内置类型冲突。该值
         * 一般用于消息内容的反序列化。
         */
        int type();

        /**
         * 消息类型的类对象
         */
        Class<? extends Content> contentClass();

        /**
         * 构造网络传输需要的二进制数据
         */
        byte[] makePBBytes(Content content) throws Throwable;

        /**
         * 通过二进制数据构造具体的消息内容
         *
         * @param data 消息内容的二进制数据流
         * @return 如果格式正确，则返回消息内容实例，否则返回null
         */
        Content makeChatContent(byte[] data) throws Throwable;

        /**
         * 构造用于本地数据库存储的字符串内容
         *
         * @param content 消息内容对象
         */
        String makeDBString(Content content) throws Throwable;
        Content makeChatContent(String data) throws Throwable;
    }

    /**
     * Hummer支持Text, Image, Audio等内置消息内容，它们的type值取值范围均小于 CustomTypeStart 值，业务如果需要
     * 实现自定义消息内容，应保证其type值应大于等于CustomTypeStart值，否则，自定义类型注册将会断言失败。
     */
    @SuppressWarnings({"WeakerAccess", "unused"})
    public final static int CustomTypeStart = 10000;


    /**
     * 根据消息内容获取其类型值
     */
    public static Integer getDataType(Content content) {
        return pbCodecs.getDataType(content);
    }

    /**
     * 消息内容的网络二进制数据序列化方法
     */
    public static byte[] makeBytes(Content content) {
        return pbCodecs.encode(content);
    }

    /**
     * 消息内容的本地存储数据序列化方法
     */
    public static String makeString(Content content) {
        return dbCodecs.encode(content);
    }

    /**
     * 通过网络数据及类型值构造一个消息内容实例
     *
     * @param type 欲构造的消息内容类型
     * @param data 用于构造消息内容的二进制数据流
     */
    public static Content makeContent(int type, byte[] data) {
        return pbCodecs.decode(type, data);
    }

    /**
     * 通过数据库数据及类型构造一个消息内容实例
     *
     * @param type 欲构造的消息内容类型
     * @param string 用于构造消息内容的字符串数据
     */
    public static Content makeContent(int type, String string) {
        return dbCodecs.decode(type, string);
    }

    /**
     * 注册业务自定义消息类型的工厂对象
     *
     * @param codec 用于具体消息的编解码器
     */
    protected static void registerCodec(final Codec codec) {
        Log.i("Content", Trace.once("registerCodec")
                .info("type", codec.type()).info("codec", codec));
        pbCodecs.register(new CodecManager.Codec<Integer, byte[], Content>() {
            @Override
            public Integer getDataType() {
                return codec.type();
            }

            @Override
            public Class<? extends Content> getModelClass() {
                return codec.contentClass();
            }

            @Override
            public byte[] encode(Content content) throws Throwable {
                return codec.makePBBytes(content);
            }

            @Override
            public Content decode(byte[] data) throws Throwable {
                return codec.makeChatContent(data);
            }
        });

        dbCodecs.register(new CodecManager.Codec<Integer, String, Content>() {
            @Override
            public Integer getDataType() {
                return codec.type();
            }

            @Override
            public Class<? extends Content> getModelClass() {
                return codec.contentClass();
            }

            @Override
            public String encode(Content content) throws Throwable {
                return codec.makeDBString(content);
            }

            @Override
            public Content decode(String data) throws Throwable {
                return codec.makeChatContent(data);
            }
        });
    }

    private static final class PBCodecs extends CodecManager<Integer, byte[], Content> { }
    private static final class DBCodecs extends CodecManager<Integer, String, Content> { }

    private static final PBCodecs pbCodecs = new PBCodecs();
    private static final DBCodecs dbCodecs = new DBCodecs();
}
