package com.hummer.im._internals.shared;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;

import java.util.concurrent.Semaphore;

public final class DispatchQueue {

    public static final DispatchQueue main   = new DispatchQueue(new LooperHandlerProvider(Looper.getMainLooper()));
    public static final DispatchQueue direct = new DispatchQueue(new DispatchQueue.WorkerHandler("hmr_callBack"));
    private static final String TAG = "DispatchQueue";

    public interface HandlerProvider {
        Handler getHandler();
    }

    public static class LooperHandlerProvider implements HandlerProvider {

        public LooperHandlerProvider(@NonNull Looper looper) {
            this.handler = new Handler(looper);
        }

        @Override
        public Handler getHandler() {
            return handler;
        }

        private final Handler handler;
    }

    public static class WorkerHandler implements HandlerProvider {
        public WorkerHandler(String threadName) {
            this.threadName = threadName;
        }

        @Override
        public Handler getHandler() {
            HandlerThread thread = new HandlerThread(threadName);
            thread.start();
            return new Handler(thread.getLooper());
        }

        private final String threadName;
    }

    public void async(@NonNull final String taskName, @NonNull final Runnable runnable) {
        HMRContext.recorder.start("[async]" + taskName);

        if (handler == null) {
            Log.i(TAG, Trace.once().method("async")
                    .msg("runnable, run"));
            runnable.run();
            HMRContext.recorder.stop("[async]" + taskName);
            return;
        }

        handler.post(new Runnable() {
            @Override
            public void run() {
                runnable.run();
                HMRContext.recorder.stop("[async]" + taskName);
            }
        });
    }

    public void sync(@NonNull final String taskName, @NonNull final Runnable runnable) {
        HMRContext.recorder.start("[sync]" + taskName);

        if (handler == null) {
            runnable.run();
            HMRContext.recorder.stop("[sync]" + taskName);
            return;
        }

        boolean sameLooper = Looper.myLooper() == handler.getLooper();

        if (sameLooper) {
            runnable.run();
            HMRContext.recorder.stop("[sync]" + taskName);
        } else {
            final Semaphore semaphore = new Semaphore(0);
            handler.post(new Runnable() {
                @Override
                public void run() {
                    runnable.run();
                    HMRContext.recorder.stop("[sync]" + taskName);
                    semaphore.release();
                }
            });
            semaphore.acquireUninterruptibly(1);
        }
    }

    public void asyncAfter(@NonNull final String taskName, int delayMillis, @NonNull Runnable runnable) {
        HMRContext.recorder.start("[asyncAfter]" + taskName);

        if (handler == null) {
            try {
                Thread.sleep(delayMillis);
            } catch (InterruptedException e) {
                // 无需处理
            }

            runnable.run();
            HMRContext.recorder.stop("[asyncAfter]" + taskName);
        } else {
            handler.postDelayed(runnable, delayMillis);
            HMRContext.recorder.stop("[asyncAfter]" + taskName);
        }
    }

    public Handler getHandler() {
        return handler;
    }

    private final Handler handler;

    @SuppressWarnings("unused") // Meant not for calling
    public DispatchQueue() {
        handler = null;
    }

    public DispatchQueue(HandlerProvider provider) {
        this.handler = provider.getHandler();
    }
}
