package com.hummer.im._internals.services.upload.YYaliOSS;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.alibaba.sdk.android.oss.ClientConfiguration;
import com.alibaba.sdk.android.oss.ClientException;
import com.alibaba.sdk.android.oss.OSS;
import com.alibaba.sdk.android.oss.OSSClient;
import com.alibaba.sdk.android.oss.ServiceException;
import com.alibaba.sdk.android.oss.callback.OSSProgressCallback;
import com.alibaba.sdk.android.oss.callback.OSSRetryCallback;
import com.alibaba.sdk.android.oss.model.PutObjectRequest;
import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.services.upload.Uploader;
import com.hummer.im._internals.shared.FileUtils;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;


/**
 * Oss方式上传的策略实现
 */
public final class AliOSS implements Uploader {

    private static final String TAG = "AliOSS";
    private static final String DEFAULT_OSS_REGION = "cn-shenzhen";

    private static final Map<String, String> REGION_BUCKET_MAP = new HashMap<>();
    static {
        // 中国
        REGION_BUCKET_MAP.put("china", "cn-shenzhen");
        // 深圳
        REGION_BUCKET_MAP.put("shenzhen", "cn-shenzhen");
        // 印尼
        REGION_BUCKET_MAP.put("indonesia", "ap-southeast-5");
        // 北美
        REGION_BUCKET_MAP.put("america", "us_east-1");
        // 巴西
        REGION_BUCKET_MAP.put("brazil", "us-east-1");
        // 印度
        REGION_BUCKET_MAP.put("india", "ap-south-1");
        // 迪拜
        REGION_BUCKET_MAP.put("Dubai", "me-east-1");
        // 吉隆坡
        REGION_BUCKET_MAP.put("KL", "ap-southeast-3");
    }

    @Override
    public void uploadFile(@NonNull final String filePath,
                           @NonNull final Uploader.UploadCallback<String> clientCallback) {
        HMRContext.work.async("AliOSS::uploadFile", new Runnable() {
            @Override
            public void run() {
                if (!FileUtils.exists(filePath)) {
                    clientCallback.onFailure(new Error(Error.Code.IOError, "File not existed!"));
                    return;
                }

                startUploading(filePath, new UploadCallback<String>() {
                    @Override
                    public void onProgress(float value) {
                        clientCallback.onProgress(value);
                    }

                    @Override
                    public void onSuccess(String destUrl) {
                        clientCallback.onSuccess(destUrl);
                    }

                    @Override
                    public void onFailure(Error err) {
                        clientCallback.onFailure(err);
                    }
                });
            }
        });
    }

    @Override
    public void cancelUploadFile(@NonNull String filePath) {

    }

    @Override
    public @Nullable String acquireThumbImageUrl(String originImageUrl, int width, int height) {
        if  (originImageUrl.matches("^https?://cim-oss.*/\\d+/[^?]*$")) {
            return String.format(Locale.US, "%s?x-oss-process=image/resize,m_lfit,h_%d,w_%d",
                    originImageUrl, height, width);
        } else {
            return null;
        }
    }

    @SuppressWarnings({"SameParameterValue", "ConstantConditions"})
    private static void startUploading(@NonNull final String filePath, @NonNull final UploadCallback<String> callback) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                long startTS = System.currentTimeMillis();

                // 对region进行适当的防御和映射处理
                String region = REGION_BUCKET_MAP.get(HMRContext.region.area);
                if (TextUtils.isEmpty(region)) {
                    Log.e(TAG, Trace.once().method("startUploading")
                            .info("region.area", HMRContext.region.area));
                    region = DEFAULT_OSS_REGION;
                }

                String fileExt = "";
                int extIndex = filePath.lastIndexOf(".");
                if (extIndex > 0) {
                    fileExt = filePath.substring(extIndex);
                }

                final String uuid = UUID.randomUUID().toString();
                final String uid = String.valueOf(HMR.getMe().getId());
                final String dstPath = uid + "/" + uuid + fileExt;

                // oss client
                final String endpoint = "https://oss-" + region + ".aliyuncs.com";
                final String bucketPrefix = "cim-" + region;
                final String bucket = bucketPrefix + "-" + HMRContext.appId;

                // 初始化OSS的token的获取方式
                final OSSCredentialProviderImpl mOSSCreditProvider = new OSSCredentialProviderImpl(region);
                ClientConfiguration configuration = new ClientConfiguration();
                configuration.setMaxErrorRetry(3);
                OSS oss = new OSSClient(HMRContext.getAppContext(), endpoint, mOSSCreditProvider, configuration);

                // 进度回调处理，实测5M左右的文件会产生2400多次回调，全部抛给work队列会导致极大延迟。使得上传耗时约为30s
                // 在做了下面的进度合并后，最多回调100次，同样为5M大小的文件，上传耗时压缩为4s~5s
                final float[] progress = {0.0f};
                PutObjectRequest put = new PutObjectRequest(bucket, dstPath, filePath);
                put.setProgressCallback(new OSSProgressCallback<PutObjectRequest>() {
                    @Override
                    public void onProgress(PutObjectRequest request, long currentSize, long totalSize) {
                        float current = (totalSize == 0) ? 0.0f : (float) currentSize / (float) totalSize;
                        if (current - progress[0] >= 0.01f) {
                            progress[0] = ((float) (int) (current * 100.0f)) / 100.0f;
                            HMRContext.work.async("AliOSS::progressCallback", new Runnable() {
                                @Override
                                public void run() {
                                    callback.onProgress(progress[0]);
                                }
                            });
                        }
                    }
                });

                // 上传失败aliOSS的SDK会回调retry接口，我们需要在retry接口里把retry设为true来重新生成新的token
                put.setRetryCallback(new OSSRetryCallback() {
                    @Override
                    public void onRetryCallback() {
                        mOSSCreditProvider.setRetrying(true);
                    }
                });

                // 执行上传
                try {
                    oss.putObject(put);
                    mOSSCreditProvider.setRetrying(false);

                    final String originalUrl = "https://cim-oss-" + region + "-" + HMRContext.appId +
                            ".bs2dl.yy.com/" + dstPath;

                    final long elapsed = System.currentTimeMillis() - startTS;
                    HMRContext.work.async("AliOSS::startUploading", new Runnable() {
                        @Override
                        public void run() {
                            Log.i(TAG, Trace.once().method("startUploading")
                                    .msg("Success")
                                    .info("duration", "" + elapsed + "ms"));
                            callback.onSuccess(originalUrl);
                        }
                    });
                } catch (ClientException e) {
                    Log.e(TAG, Trace.once().method("startUploading").msg("Failed")
                            .info("localPath", filePath)
                            .info("remotePath", dstPath)
                            .info("exception", e.getMessage()));

                    HMRContext.work.async("AliOSS::ClientException", new Runnable() {
                        @Override
                        public void run() {
                            callback.onFailure(new Error(
                                    Error.Code.ThirdPartyServiceError,
                                    "Upload failed"
                            ));
                        }
                    });
                } catch (ServiceException e) {
                    Log.e(TAG, Trace.once().method("startUploading").msg("Failed")
                            .info("localPath", filePath)
                            .info("remotePath", dstPath)
                            .info("requestId", e.getRequestId())
                            .info("host", e.getHostId())
                            .info("msg", e.getRawMessage()));

                    HMRContext.work.async("AliOSS::ServiceException", new Runnable() {
                        @Override
                        public void run() {
                            callback.onFailure(new Error(
                                    Error.Code.ThirdPartyServiceError,
                                    "Upload failed"
                            ));
                        }
                    });
                }
            }
        }, "hmr_upload").start();
    }
}
