package com.hummer.im._internals.chatsvc;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.google.protobuf.ByteString;
import com.hummer.im.Error;
import com.hummer.im._internals.HummerException;
import com.hummer.im._internals.IMRPC;
import com.hummer.im._internals.proto.Chat;
import com.hummer.im.model.chat.Content;
import com.hummer.im.model.chat.Message;
import com.hummer.im._internals.shared.StringChain;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.completion.CompletionUtils;

public class RPCSendAppSessionMessage extends
        IMRPC<Chat.AppSessionChatRequest, Chat.AppSessionChatRequest.Builder, Chat.AppSessionChatResponse> {

    public RPCSendAppSessionMessage(Message message, RichCompletion completion) {
        this.message    = message;
        this.completion = completion;
    }

    @Override
    public void buildHummerRequest(@NonNull Chat.AppSessionChatRequest.Builder builder) throws Throwable {
        String extension = message.getAppExtra();
        if (extension == null) {
            extension = "";
        }

        builder.setFromUid(message.getSender().getId())
                .setToId(message.getReceiver().getId())
                .setToIdType(IdentifiableHelper.makeStringFrom(message.getReceiver()))
                .setUuid(message.getUuid())
                .setExtension(extension);

        byte[] contentBytes = Content.makeBytes(message.getContent());
        Integer messageType = Content.getDataType(message.getContent());
        if (contentBytes == null || messageType == null) {
            throw new HummerException(Error.Code.ClientExceptions, "Encode failed");
        }

        builder.setMsgType(messageType).setContent(ByteString.copyFrom(contentBytes));
    }

    @Override
    public void handleHummerSuccess(@NonNull Chat.AppSessionChatResponse res) {
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void handleHummerError(@Nullable Chat.AppSessionChatResponse res, @NonNull Error error) {
        CompletionUtils.dispatchFailure(completion, error);
    }

    @Override
    public String getHummerFunction() {
        return "AppSessionChat";
    }

    @Override
    public String describeHummerRequest(Chat.AppSessionChatRequest req) {
        return new StringChain().acceptNullElements()
                .add("message", message)
                .toString();
    }

    @Override
    public String describeHummerResponse(@NonNull Chat.AppSessionChatResponse res) {
        return null;
    }

    private final Message message;
    private final RichCompletion completion;
}
