package com.hummer.im._internals.chatsvc;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.mq.Source;
import com.hummer.im._internals.proto.Im;
import com.hummer.im._internals.shared.DispatchQueue;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im.model.chat.Content;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.states.Archived;
import com.hummer.im.model.chat.states.Delivering;
import com.hummer.im.model.chat.states.Failed;
import com.hummer.im.model.chat.states.Init;
import com.hummer.im.model.chat.states.Preparing;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.OnFailure;
import com.hummer.im.model.completion.OnSuccess;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.completion.RichCompletionArg;
import com.hummer.im.model.id.Identifiable;
import com.hummer.im.service.Channel;
import com.hummer.im.service.ChatService;
import com.hummer.im.service.MQService;

import java.util.ArrayList;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

public final class ChatServiceImpl implements
        ChatService,
        MQService.MsgParser,
        Channel.NotificationHandler,
        ServiceProvider.Service {

    public static final String TAG = "ChatService";
    private static final int MaxReceivedUUIDs = 100;

    @Override
    public Class[] staticDependencies() {
        return new Class[]{MQService.class};
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{Channel.class};
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return new Class[]{MQService.class};
    }

    @Override
    public void initService() {
        HMR.getService(MQService.class).registerMsgParser(this);
    }

    @Override
    public void openService(@NonNull RichCompletion completion) {
        HMR.getService(Channel.class).addNotificationHandler(this);
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void closeService() {
        HMR.getService(Channel.class).removeNotificationHandler(this);
    }

    @Override
    public void loadManually() {
        HMRContext.work.async("ChatService::loadManually", new Runnable() {
            @Override
            public void run() {
                HMR.getService(MQService.class).pullManually();
            }
        });
    }

    @Override
    public void send(@NonNull final Message message, @Nullable final HMR.Completion completion) {
        HMRContext.work.async("ChatService::send", new SendingFlow() {
            final Trace.Flow flow = new Trace.Flow();
            final RichCompletion rCompletion = new RichCompletion(completion, "ChatService::send");

            @Override
            public void run() {
                Error error = null;
                if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
                    error = new Error(Error.Code.BadUser, "User not login, or using anonymous mode");
                }

                if (message.getReceiver() == null) {
                    error = new Error(Error.Code.InvalidParameters, "Bad message, no receiver found");
                }

                if (!(message.getState() == null || message.getState() instanceof Failed)) {
                    error = new Error(
                            Error.Code.InvalidParameters,
                            "Bad message, unexpected state: " + message.getState()
                    );
                }

                if (message.getState() != null) {
                    // && 排版太丑，使用两个if来实现同样的逻辑，将优化交给编译器
                    if (message.getSender() == null || message.getUuid() == null) {
                        error = new Error(Error.Code.InvalidParameters, "Bad message: " + message.toString());
                    }
                }

                if (message.getContent() == null) {
                    error = new Error(Error.Code.InvalidParameters, "Bad message: Null content");
                }

                if (error != null) {
                    finish(error);
                    return;
                }

                supplementPropertiesIfNeeded();
                start();
            }

            private void supplementPropertiesIfNeeded() {
                if (message.getState() != null) {
                    return;
                }

                message.setSender(HMR.getMe());                   // 覆盖发送方，只可能是由本地用户发出
                message.setTimestamp(System.currentTimeMillis()); // 单位是ms，与协议中response保持一致
                message.setUuid(UUID.randomUUID().toString());    // 使用系统提供的UUID生成器
                message.setState(new Init());
            }

            private void start() {
                Log.i(TAG, Trace.once().method("send").msg("start").info("msg", message));

                notifyBeforeSendingMessage(message);
                setMessageState(new Preparing());

                if (message.getContent() instanceof Content.Preparable) {
                    // this的类型是SendingFlow，不是ImplChatService!
                    ((Content.Preparable) message.getContent()).prepare(this);
                } else {
                    performSending();
                }
            }

            // Overrides: Message.PreparingProgress

            @Override
            public void onPrepare(Preparing.Progress progressing) {
                setMessageState(new Preparing(progressing));
            }

            @Override
            public void onPrepareSuccess() {
                Log.i(TAG, flow.trace().method("send").msg("onPrepareSuccess"));

                HMRContext.work.async("ChatService::onPrepareSuccess", new Runnable() {
                    @Override
                    public void run() {
                        performSending();
                    }
                });
            }

            @Override
            public void onPrepareFailed(Error err) {
                Log.i(TAG, flow.trace().method("send").msg("onPrepareFailed").info("error", err));

                setMessageState(new Failed(err));

                // ！！！重要，否则无法结束SendingFlow
                finish(err);
            }

            private void performSending() {
                Log.i(TAG, flow.trace().method("send").msg("performSending"));

                setMessageState(new Delivering());

                // 将消息缓存在本地（内存），并在拉取到消息队列中的消息之后进行uuid相等判断，
                // 这样就可以避免用户收到自己发送的onSendingFailed消息。
                cacheMessageUUID(message.getUuid());

                final RichCompletion cmpl = new RichCompletion("ChatService::performSending")
                        .onSuccess(new OnSuccess() {
                            @Override
                            public void onSuccess() {
                                Log.i(TAG, flow.trace().method("send").msg("onSendingSucceed"));

                                setMessageState(new Archived());
                                finish(null);
                            }
                        })
                        .onFailure(new OnFailure() {
                            @Override
                            public void onFailure(Error error) {
                                Log.e(TAG, flow.trace().method("send").msg("onSendingFailed")
                                        .info("error", error));

                                // 如果发送失败，则消息不会进入消息队列，则无需进行去重跟踪
                                // 如果不清理uuid，且失败请求过多，会导致uuid数据积累。因此需要在失败时进行清理。
                                // 但是，如果是因为请求超时而导致的失败，服务器实际上是可能已经收到并进行消息下发的
                                // 因此，不应该移除因超时而失败的缓存uuid
                                if (error.code != Error.Code.OperationTimeout) {
                                    uncacheMessageUUID(message.getUuid());
                                }

                                setMessageState(new Failed(error));
                                finish(error);
                            }
                        });

                Channel.RPC rpc = makeSendingRPC(message, cmpl);
                if (rpc == null) {
                    Error error = new Error(Error.Code.InvalidParameters, "该消息未被支持，请检查插件是否注册");
                    uncacheMessageUUID(message.getUuid());

                    setMessageState(new Failed(error));
                    finish(error);
                } else {
                    HMR.getService(Channel.class).run(rpc);
                }
            }

            private void setMessageState(final Message.State state) {
                // message.state的变化很可能会联动UI渲染逻辑，所以不能在worker线程中进行设置，必须抛到UI线程中
                DispatchQueue.main.sync("CHatService::setMessageState", new Runnable() {
                    @Override
                    public void run() {
                        message.setState(state);
                        notifyMessageStateChange(message, state);
                    }
                });
            }

            private void finish(final @Nullable Error error) {
                notifyAfterSendingMessage(message);

                if (error == null) {
                    Log.i(TAG, flow.trace().method("send").msg("finish"));
                    CompletionUtils.dispatchSuccess(rCompletion);
                } else {
                    Log.e(TAG, error, flow.trace().method("send").msg("finish"));
                    CompletionUtils.dispatchFailure(rCompletion, error);
                }
            }
        });
    }

    @Override
    public void forward(@NonNull Message message, @NonNull List<Identifiable> receivers) {
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            Log.e(TAG, Trace.once("forward").msg("User not login, or using anonymous mode"));
            return;
        }

        for (Identifiable target : receivers) {
            Message chatMessage = new Message(target, message.getContent());
            send(chatMessage, null);
        }
    }

    public void countRambleMessages(@NonNull Identifiable target,
                                    long before,
                                    long after,
                                    @Nullable RichCompletionArg<Long> completion) {
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            Log.e(TAG, Trace.once("forward").msg("User not login, or using anonymous mode"));
            return;
        }

        HMR.getService(Channel.class).run(new RPCFetchMessageSize(target, before, after, completion));
    }

    @Override
    public void addMessageListener(@Nullable Identifiable target, @NonNull MessageListener l) {
        synchronized (mMessageListeners) {
            String lisKey = DEFAULT_MESSAGE_LISTENER_KEY;
            if (target != null) {
                lisKey = String.valueOf(target.getId());
            }

            HashSet<MessageListener> listeners = mMessageListeners.get(lisKey);
            if (listeners == null) {
                listeners = new HashSet<>();
                mMessageListeners.put(lisKey, listeners);
            }
            listeners.add(l);

            Log.d(TAG, Trace.once()
                    .method("addMessageListener")
                    .info("name", l.getClass().getSimpleName())
                    .info("size", listeners.size()));
        }
    }

    @Override
    public void removeMessageListener(@Nullable Identifiable target, @NonNull MessageListener l) {
        synchronized (mMessageListeners) {
            String lisKey = DEFAULT_MESSAGE_LISTENER_KEY;
            if (target != null) {
                lisKey = String.valueOf(target.getId());
            }

            HashSet<MessageListener> listeners = mMessageListeners.get(lisKey);
            if (listeners != null) {
                listeners.remove(l);
            }

            Log.d(TAG, Trace.once()
                    .method("removeMessageListener")
                    .info("name", l.getClass().getSimpleName())
                    .info("size", listeners == null ? "<null>" : listeners.size()));
        }
    }

    @Override
    public void addStateListener(@Nullable Message message, @NonNull StateListener listener) {
        synchronized (mStateListeners) {
            String lisKey = DEFAULT_STATE_LISTENER_KEY;
            if (message != null) {
                lisKey = message.getUuid();
            }

            HashSet<StateListener> listeners = mStateListeners.get(lisKey);
            if (listeners == null) {
                listeners = new HashSet<>();
                mStateListeners.put(lisKey, listeners);
            }
            listeners.add(listener);
            Log.d(TAG, Trace.once()
                    .method("addStateListener")
                    .info("name", listener.getClass().getSimpleName())
                    .info("size", listeners.size()));
        }
    }

    @Override
    public void removeStateListener(@Nullable Message message, @NonNull StateListener listener) {
        synchronized (mStateListeners) {
            String lisKey = DEFAULT_STATE_LISTENER_KEY;
            if (message != null) {
                lisKey = message.getUuid();
            }

            HashSet<StateListener> listeners = mStateListeners.get(lisKey);
            if (listeners != null) {
                listeners.remove(listener);
            }

            Log.d(TAG, Trace.once()
                    .method("removeStateListener")
                    .info("name", listener.getClass().getSimpleName())
                    .info("size", listeners == null ? "<null>" : listeners.size()));
        }
    }

    private void cacheMessageUUID(final String uuid) {
        mUUIDsSent.add(uuid);
    }

    private void uncacheMessageUUID(final String uuid) {
        mUUIDsSent.remove(uuid);
    }

    private void cacheReceivedMessage(final Message message) {
        if (mUUIDsReceivedList.size() >= MaxReceivedUUIDs) {
            // 缓存不应该是无限增长的，最多只支持MaxReceivedUUIDs数量的接收消息去重。因此，需要将我们认为
            // 已经没有去重匹配价值的uuid从命中表中移除。这里使用的评权算法是FIFO
            String overdue = mUUIDsReceivedList.pop();
            mUUIDsReceivedSet.remove(overdue);
        }

        String newUUID = message.getUuid();
        mUUIDsReceivedList.push(newUUID);
        mUUIDsReceivedSet.add(newUUID);
    }

    private boolean alreadyReceived(final Message message) {
        // 这里应该使用哈希表进行查重，而不是链表结构。后者存在严重的迭代效率问题。
        return mUUIDsReceivedSet.contains(message.getUuid());
    }

    // conflate: http://dict.cn/conflate
    // 尝试与本地发送出去的消息进行'合并去重'操作，以便过滤掉那些本地发送出去的消息
    private boolean conflateLocalSent(Message message) {
        if (mUUIDsSent.contains(message.getUuid())) {
            // 移除它对应的缓存uuid，否则该就会一直积累，一来造成内存泄露，二来降低去重命中效率
            mUUIDsSent.remove(message.getUuid());
            return true;
        } else {
            return false;
        }
    }

    // ----- MessageState notify

    interface MessageListenersVisitor {
        void visit(MessageListener listener);
        String visitorName();
    }

    private void iterateMessageListeners(final Message message, final MessageListenersVisitor visitor) {
        DispatchQueue.main.sync("ChatService::" + visitor.visitorName(), new Runnable() {
            @Override
            public void run() {
                synchronized (mMessageListeners) {
                    // 避免同一个listener被通知多次，先将它们收集到一个set内，然后在发起访问
                    String key = String.valueOf(message.getTarget() == null ? "" : message.getTarget().getId());
                    HashSet<MessageListener> listeners = mMessageListeners.get(key);
                    HashSet<MessageListener> hashLis;

                    if (listeners == null) {
                        hashLis = new HashSet<>();
                    } else {
                        hashLis = (HashSet<MessageListener>) listeners.clone();
                    }

                    HashSet<MessageListener> fullListeners = mMessageListeners.get(DEFAULT_MESSAGE_LISTENER_KEY);

                    if (fullListeners != null) {
                        hashLis.addAll((HashSet<MessageListener>) fullListeners.clone());
                    }

                    for (MessageListener l : hashLis) {
                        visitor.visit(l);
                    }
                }
            }
        });
    }

    private void notifyBeforeSendingMessage(final Message message) {
        iterateMessageListeners(message, new MessageListenersVisitor() {
            @Override
            public void visit(MessageListener listener) {
                listener.beforeSendingMessage(message);
            }

            @Override
            public String visitorName() {
                return "notifyBeforeSendingMessage";
            }
        });
    }

    private void notifyAfterSendingMessage(final Message message) {
        iterateMessageListeners(message, new MessageListenersVisitor() {
            @Override
            public void visit(MessageListener listener) {
                listener.afterSendingMessage(message);
            }

            @Override
            public String visitorName() {
                return "notifyAfterSendingMessage";
            }
        });
    }

    private void notifyBeforeReceiveMessage(final Message message) {
        iterateMessageListeners(message, new MessageListenersVisitor() {
            @Override
            public void visit(MessageListener listener) {
                listener.beforeReceivingMessage(message);
            }

            @Override
            public String visitorName() {
                return "notifyBeforeReceiveMessage";
            }
        });
    }

    private void notifyAfterReceiveMessage(final Message message) {
        iterateMessageListeners(message, new MessageListenersVisitor() {
            @Override
            public void visit(MessageListener listener) {
                listener.afterReceivingMessage(message);
            }

            @Override
            public String visitorName() {
                return "notifyAfterReceiveMessage";
            }
        });
    }

    // ----- StateChange notify

    private void notifyMessageStateChange(final Message message, final Message.State state) {
        DispatchQueue.main.sync("ChatService::notifyMessageStateChange", new Runnable() {
            @Override
            public void run() {
                synchronized (mStateListeners) {
                    HashSet<StateListener> listeners = mStateListeners.get(message.getUuid());

                    // 对listener进行深拷贝，避免多并发访问导致崩溃。
                    if (listeners != null) {
                        HashSet<StateListener> hashLis = (HashSet<StateListener>) listeners.clone();
                        for (final StateListener l : hashLis) {
                            l.onUpdateMessageState(message, state);
                        }
                    }

                    HashSet<StateListener> fullListeners = mStateListeners.get(DEFAULT_STATE_LISTENER_KEY);
                    if (fullListeners != null) {
                        HashSet<StateListener> hashFullLis = (HashSet<StateListener>) fullListeners.clone();
                        for (final StateListener l : hashFullLis) {
                            l.onUpdateMessageState(message, state);
                        }
                    }
                }
            }
        });
    }

    // Mark - 解析MQ消息

    @Override
    public void parse(Im.Msg msg, Source fromSource) {
        dispatchMessage(makeMessage(msg, fromSource));
    }

    // Mark - 解析service消息

    @Override
    public void onNotify(String serviceName, String functionName, byte[] data) {
        dispatchMessage(makeMessage(serviceName, functionName, data));
    }

    private void dispatchMessage(final Message message) {
        if (message == null) {
            return;
        }

        HMRContext.work.async("ChatService::dispatchMessage", new Runnable() {
            @Override
            public void run() {
                // 由本客户端发出的消息，不作接收处理
                if (ChatServiceImpl.this.conflateLocalSent(message)) {
                    return;
                }

                // 在弱网情况下（100 kbps，海外常见），通道层可能会触发网络重试，导致非幂等的RPC请求操作。进而使得消息队列
                // 中存储多条内容重复的消息（UUID重复）。为了在通道层过滤掉此类无意义的消息，引入了短表去重手段。
                if (ChatServiceImpl.this.alreadyReceived(message)) {
                    Log.w(TAG, Trace.once().method("dispatchMessage").msg("重复消息").info("uuid", message.getUuid()));
                    return;
                }

                // 为了严谨区分before/after两个过程，将它们分开通知。例如消息存储处理时，需要在beforeReceive中
                // 创建消息表，并在afterReceive中进行消息插入。如果只是一个listener，该问题并不存在，但一旦有多个
                // listeners，则将两个过程分开是必须的
                ChatServiceImpl.this.notifyBeforeReceiveMessage(message);
                ChatServiceImpl.this.notifyAfterReceiveMessage(message);

                // 对确认接收的消息，需要进行短表缓存，以便对后续的重发消息进行过滤
                ChatServiceImpl.this.cacheReceivedMessage(message);
            }
        });
    }

    private interface SendingFlow extends Runnable, Content.PreparingCallback {
    }

    private final Map<String, HashSet<MessageListener>> mMessageListeners = new HashMap<>();
    private final Map<String, HashSet<StateListener>> mStateListeners = new HashMap<>();
    private final static String DEFAULT_STATE_LISTENER_KEY = "default_state_listener_key";
    private final static String DEFAULT_MESSAGE_LISTENER_KEY = "default_message_listener_key";

    // 去重缓冲命中对效率要求较高，使用HashSet比较合适
    private final Set<String> mUUIDsSent = new HashSet<>();
    private final Deque<String> mUUIDsReceivedList = new LinkedList<>();
    // 如果只是用链表，会有较大的命中检测效率问题，因此同时使用链表和哈希表的方式来解决命中效率的问题，属于空间换时间的trade-off
    private final Set<String> mUUIDsReceivedSet = new HashSet<>();


    // ----- Extensions

    public interface SendingExtension {
        Channel.RPC makeSendingRPC(Message message, RichCompletion completion);
    }

    public interface ParserExtension {
        Message parseMessage(String serviceName, String functionName, byte[] data) throws Exception;

        Message parseMessage(Im.Msg msg, Source fromSource) throws Exception;
    }

    public static void registerSendingExtension(SendingExtension ext) {
        sendingExtensions.add(ext);
    }

    public static void registerParserExtension(ParserExtension parser) {
        parserExtensions.add(parser);
    }

    private static Channel.RPC makeSendingRPC(Message message, RichCompletion completion) {
        for (SendingExtension ext : sendingExtensions) {
            Channel.RPC rpc = ext.makeSendingRPC(message, completion);

            if (rpc != null) {
                return rpc;
            }
        }

        return null;
    }

    private static Message makeMessage(String serviceName, String functionName, byte[] data) {
        for (ParserExtension parser : parserExtensions) {
            try {
                Message msg = parser.parseMessage(serviceName, functionName, data);
                if (msg != null) {
                    Log.d("Core", Trace.once().method("makeMessage1")
                            .info("parser", parser)
                            .info("message", msg));

                    return msg;
                }
            } catch (Exception e) {
                // Ignore this parser
            }
        }

        return null;
    }

    private static Message makeMessage(Im.Msg msg, Source fromSource) {
        for (ParserExtension parser : parserExtensions) {
            try {
                Message message = parser.parseMessage(msg, fromSource);
                if (message != null) {
                    Log.d("Core", Trace.once().method("makeMessage2")
                            .info("parser", parser)
                            .info("message", message));

                    return message;
                }
            } catch (Exception e) {
                // Ignore this parser
            }
        }

        return null;
    }

    private static List<SendingExtension> sendingExtensions = new ArrayList<>();
    private static List<ParserExtension> parserExtensions = new ArrayList<>();
}
