package com.hummer.im._internals.channel;

import android.os.Handler;
import android.support.annotation.NonNull;

import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.shared.DispatchQueue;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.service.Channel;
import com.hummer.im.service.ChannelStateService;
import com.yy.platform.baseservice.ConstCode;
import com.yy.platform.baseservice.IChannelListener;
import com.yy.platform.baseservice.YYServiceCore;

import java.util.HashSet;

public final class ChannelStateServiceImp implements
        ChannelStateService,
        ServiceProvider.Service,
        IChannelListener.IChannelStatusNotify {

    // Mark - ServiceProvider
    @Override
    public Class[] staticDependencies() {
        return null;
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{ Channel.class };
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return null;
    }

    @Override
    public void initService() {

    }

    @Override
    public void openService(@NonNull RichCompletion completion) {
        YYServiceCore.getInstance().registChannelStatusListener(this, new Handler());
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void closeService() {
        YYServiceCore.getInstance().unregistChannelStatusListener(this);
    }

    // Mark - ChannelStateService

    @Override
    public ChannelState getState() {
        return mState;
    }

    @Override
    public void addChannelStateListener(@NonNull ChannelStateListener listener) {
        synchronized (mListeners) {
            mListeners.add(listener);

            Log.i(TAG, Trace.once()
                    .method("addChannelStateListener")
                    .info("name", listener.getClass().getSimpleName())
                    .info("size", mListeners.size()));
        }
    }

    @Override
    public void removeChannelStateListener(@NonNull ChannelStateListener listener) {
        synchronized (mListeners) {
            mListeners.remove(listener);

            Log.i(TAG, Trace.once()
                    .method("removeStateListener")
                    .info("name", listener.getClass().getSimpleName())
                    .info("size", mListeners.size()));
        }
    }

    private void updateChannelState(final ChannelState uState) {
        if (uState == mState) {
            return;
        }

        final ChannelState oldState = mState;
        mState = uState;

        DispatchQueue.main.async("ChannelStateServiceImp::updateChannelState", new Runnable() {
            @Override
            public void run() {
                synchronized (mListeners) {

                    Log.i(TAG, Trace.once()
                            .method("updateChannelState")
                            .info("fromState", oldState)
                            .info("toState", mState)
                            .info("size", mListeners.size()));
                    HashSet<ChannelStateListener> copyListener = (HashSet<ChannelStateListener>) mListeners.clone();

                    for (ChannelStateListener l : copyListener) {
                        l.onUpdateChannelState(oldState, mState);
                    }
                }
            }
        });
    }

    // Mark - IChannelListener.IChannelStatusNotify

    @Override
    public void onStatus(int status) {
        if (status == ConstCode.Status.Binded) {
            updateChannelState(ChannelState.Connected);
        } else if (status == ConstCode.Status.Init ||
                status == ConstCode.Status.Close ||
                status == ConstCode.Status.OnClosed ||
                status == ConstCode.Status.OnConnecttimeout) {
            updateChannelState(ChannelState.Disconnected);
        } else {
            updateChannelState(ChannelState.Connecting);
        }
    }

    private static final String TAG = "ChannelStateService";
    private ChannelState mState = ChannelState.Unavailable;
    private final HashSet<ChannelStateListener> mListeners = new HashSet<>();
}
