package com.hummer.im._internals.blacklist;

import android.support.annotation.NonNull;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.mq.Source;
import com.hummer.im._internals.proto.BuddyOuterClass;
import com.hummer.im._internals.proto.Im;
import com.hummer.im._internals.shared.DispatchQueue;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.OnSuccessArg;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.completion.RichCompletionArg;
import com.hummer.im.model.id.Identifiable;
import com.hummer.im.model.id.User;
import com.hummer.im.service.BlacklistService;
import com.hummer.im.service.Channel;
import com.hummer.im.service.MQService;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public final class BlacklistServiceImpl implements BlacklistService, ServiceProvider.Service, MQService.MsgParser {

    private static final String TAG = "BlacklistService";
    private final Set<User> blockedUsers = new HashSet<>();
    private final Set<BlacklistService.BlacklistListener> observers = new HashSet<>();

    @Override
    public Class[] staticDependencies() {
        return new Class[]{MQService.class};
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{
                Channel.class
        };
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return new Class[]{MQService.class};
    }

    @Override
    public void initService() {
        HMR.getService(MQService.class).registerMsgParser(this);
    }

    @Override
    public void openService(@NonNull RichCompletion completion) {
        if (!HMR.getMe().isAnonymous()) {
            ServiceProvider.get(Channel.class).run(new RPCGetBlackList(
                    new RichCompletionArg<List<User>>("BlacklistService::RPCGetBlackList")
                    .onSuccess(new OnSuccessArg<List<User>>() {
                        @Override
                        public void onSuccess(List<User> users) {
                            blockedUsers.addAll(users);
                            BlacklistServiceImpl.this.notifyBlacklistChanged();
                        }
                    })));
        }

        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void closeService() {
        blockedUsers.clear();
    }

    @Override
    public void listBlacklist(HMR.CompletionArg<List<User>> completion) {
        RichCompletionArg rCompletion = new RichCompletionArg(completion, "BlacklistService::listBlacklist");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(
                    rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode")
            );
            return;
        }

        CompletionUtils.dispatchSuccess(rCompletion, new ArrayList<>(blockedUsers));
    }

    @Override
    public void isBlocked(@NonNull User user, HMR.CompletionArg<Boolean> completion) {
        RichCompletionArg rCompletion = new RichCompletionArg(completion, "BlacklistService::isBlocked");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(
                    rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode")
            );
            return;
        }

        CompletionUtils.dispatchSuccess(rCompletion, blockedUsers.contains(user));
    }

    @Override
    public void block(@NonNull final User user, @NonNull final HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion, "BlacklistService::block");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(
                    rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode")
            );
            return;
        }

        ServiceProvider.get(Channel.class).run(new RPCBlockUser(user, rCompletion
                .beforeSuccess(new Runnable() {
                    @Override
                    public void run() {
                        BlacklistServiceImpl.this.blockUser(user);
                    }
                })));
    }

    @Override
    public void unblock(@NonNull final User user, @NonNull final HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion, "BlacklistService::unblock");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(
                    rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode")
            );
            return;
        }

        ServiceProvider.get(Channel.class).run(new RPCUnblockUser(user, rCompletion
                .beforeSuccess(new Runnable() {
                    @Override
                    public void run() {
                        BlacklistServiceImpl.this.unblockUser(user);
                    }
                })));
    }

    @Override
    public void addListener(@NonNull final BlacklistListener listener) {
        HMRContext.work.async("BlacklistService::addListener", new Runnable() {
            @Override
            public void run() {
                observers.add(listener);
                Log.i(TAG, Trace.once().method("addListener")
                        .info("listener", listener.getClass().getSimpleName())
                        .info("size", observers.size()));

                BlacklistServiceImpl.this.notifyBlacklistChanged();
            }
        });
    }

    @Override
    public void removeListener(@NonNull final BlacklistListener listener) {
        HMRContext.work.async("BlacklistService::removeListener", new Runnable() {
            @Override
            public void run() {
                observers.remove(listener);
                Log.i(TAG, Trace.once().method("removeListener")
                        .info("listener", listener.getClass().getSimpleName())
                        .info("size", observers.size()));
            }
        });
    }

    @Override
    public void parse(Im.Msg msg, Source fromSource) throws Exception {
        if (msg.getAction() == Im.Action.kNotifyBlacklistAddedRequest) {
            handleAddingMessage(msg);
        } else if (msg.getAction() == Im.Action.kNotifyBlacklistRemovedRequest) {
            handleRemovingMessage(msg);
        }
    }

    private void handleAddingMessage(Im.Msg msg) throws Exception {
        BuddyOuterClass.NotifyBlacklistAddedRequest req = BuddyOuterClass.NotifyBlacklistAddedRequest.newBuilder()
                .mergeFrom(msg.getContent()).build();

        User initiator = new User(req.getInitiateUid());
        User target = new User(req.getAddedUid());

        if (Identifiable.equals(initiator, HMR.getMe())) {
            blockUser(target);
        } else {
            notifyBlockedBy(initiator);
        }
    }

    private void handleRemovingMessage(Im.Msg msg) throws Exception {
        BuddyOuterClass.NotifyBlacklistRemovedRequest req = BuddyOuterClass.NotifyBlacklistRemovedRequest.newBuilder()
                .mergeFrom(msg.getContent())
                .build();

        User initiator = new User(req.getInitiateUid());
        User target = new User(req.getRemovedUid());

        if (Identifiable.equals(initiator, HMR.getMe())) {
            unblockUser(target);
        } else {
            notifyUnblockedBy(initiator);
        }
    }

    private void notifyBlacklistChanged() {
        final List<User> newBlacklist = new ArrayList<>(blockedUsers);
        DispatchQueue.main.async("BlacklistService::notifyBlacklistChanged", new Runnable() {
            @Override
            public void run() {
                for (BlacklistListener l : observers) {
                    l.onUpdateBlacklist(newBlacklist);
                }
            }
        });
    }

    private void notifyBlockedBy(final User user) {
        DispatchQueue.main.async("BlacklistService::notifyBlockedBy", new Runnable() {
            @Override
            public void run() {
                for (BlacklistListener l : observers) {
                    l.onBlockedByUser(user);
                }
            }
        });
    }

    private void notifyUnblockedBy(final User user) {
        DispatchQueue.main.async("BlacklistService::notifyUnblockedBy", new Runnable() {
            @Override
            public void run() {
                for (BlacklistListener l : observers) {
                    l.onUnblockedByUser(user);
                }
            }
        });
    }

    private void blockUser(final User user) {
        if (!blockedUsers.add(user)) {
            // ignore if already in list
            return;
        }

        DispatchQueue.main.async("BlacklistService::blockUser", new Runnable() {
            @Override
            public void run() {
                for (BlacklistListener l : observers) {
                    l.onBlockUser(user);
                }
            }
        });

        notifyBlacklistChanged();
    }

    private void unblockUser(final User user) {
        if (!blockedUsers.remove(user)) {
            // ignore if user not in list
            return;
        }

        DispatchQueue.main.async("BlacklistService::unblockUser", new Runnable() {
            @Override
            public void run() {
                for (BlacklistListener l : observers) {
                    l.onUnblockUser(user);
                }
            }
        });

        notifyBlacklistChanged();
    }
}
