package com.hummer.im._internals;

import android.support.annotation.NonNull;

import com.hummer.im.HMR;
import com.hummer.im._internals.blacklist.BlacklistServiceImpl;
import com.hummer.im._internals.channel.ChannelStateServiceImp;
import com.hummer.im._internals.chatsvc.ChatServiceImpl;
import com.hummer.im._internals.chatsvc.MessageParser;
import com.hummer.im._internals.chatsvc.RPCSendAppSessionMessage;
import com.hummer.im._internals.chatsvc.RPCSendMessage;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.mq.MQServiceImpl;
import com.hummer.im._internals.mq.Source;
import com.hummer.im._internals.proto.Im;
import com.hummer.im._internals.roaming.RoamingServiceImpl;
import com.hummer.im._internals.services.upload.UploadServiceImp;
import com.hummer.im._internals.services.user.UserService;
import com.hummer.im._internals.shared.AsciiTable;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im._internals.user.UserServiceImpl;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.PushContent;
import com.hummer.im.model.chat.contents.Audio;
import com.hummer.im.model.chat.contents.Image;
import com.hummer.im.model.chat.contents.Text;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.id.AppSession;
import com.hummer.im.model.id.Group;
import com.hummer.im.model.id.IDFactory;
import com.hummer.im.model.id.Identifiable;
import com.hummer.im.model.id.User;
import com.hummer.im.service.BlacklistService;
import com.hummer.im.service.Channel;
import com.hummer.im.service.ChannelStateService;
import com.hummer.im.service.ChatService;
import com.hummer.im.service.MQService;
import com.hummer.im.service.RoamingService;
import com.hummer.im.service.UploadService;
import com.yy.yylogger.Logger;
import com.yy.yylogger.mode.Delegate;
import com.yy.yylogger.mode.IChannelMode;
import com.yy.yylogger.scope.ScopeAll;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

@SuppressWarnings("unused")
public final class SPCoreModuleLoader implements ServiceProvider.ModuleLoader {

    static class LoggerService implements ServiceProvider.Service {
        @Override
        public void initService() {
            registerChatContentCodecs();
            registerIdExtensions();
            registerChatServiceExtensions();

            MQService mqService = HMR.getService(MQService.class);
            mqService.addSource(new Source(new Source.Private(null, null, 5 * 60 * 1000L)));
            mqService.addSource(new Source(new Source.Shared(1L, null)));
        }

        @Override
        public Class[] staticDependencies() {
            return  null;
        }

        @Override
        public Class[] inherentDynamicDependencies() {
            return new Class[]{Channel.class};
        }

        @Override
        public Class[] plantingDynamicDependencies() {
            return null;
        }

        @Override
        public void openService(@NonNull RichCompletion completion) {
            Logger.enablePassiveReport(
                    new Delegate(),
                    HMR.getMe().getId(),
                    new ScopeAll(),
                    new IChannelMode.LoginCallback() {
                        @Override
                        public void success(String message) {
                        }

                        @Override
                        public void failed(int sdkCode, int srvCode, String message) {
                        }
                    });

            CompletionUtils.dispatchSuccess(completion);
        }

        @Override
        public void closeService() {
            Logger.disablePassiveReport(new IChannelMode.LogoutCallback() {
                @Override
                public void success(String message) {
                }

                @Override
                public void failed(int sdkCode, int srvCode, String message) {
                }
            });
        }
    }

    @Override
    public void initModule() {
        showInitMessage();
    }

    @Override
    public Map<Class, ServiceProvider.Service> getServices() {
        return new HashMap<Class, ServiceProvider.Service>() {{
            put(BlacklistService.class, new BlacklistServiceImpl());
            put(PrefStorage.class, PrefStorage.storage());
            put(UserService.class, new UserServiceImpl());
            put(MQService.class, new MQServiceImpl());
            put(ChatService.class, new ChatServiceImpl());
            put(LoggerService.class, new LoggerService());
            put(UploadService.class, new UploadServiceImp());
            put(RoamingService.class, new RoamingServiceImpl());
            put(ChannelStateService.class, new ChannelStateServiceImp());
        }};
    }

    private static void showInitMessage() {
        AsciiTable.log(40, "Welcome to Hummer", new String[][]{
                {"version", HMR.getVersion()},
                {"appId", String.valueOf(HMRContext.appId)},
        });
    }

    private static void registerChatContentCodecs() {
        Text.registerCodecs();
        Image.registerCodecs();
        Audio.registerCodecs();
    }

    private static void registerIdExtensions() {
        registerUserIdExtension();
    }

    private static void registerUserIdExtension() {
        IDFactory.registerExtension(new IDFactory.Extension() {
            @Override
            public Identifiable makeId(String idString) {
                if (idString.matches("^user_\\d+$")) {
                    return new User(Long.parseLong(idString.substring(5)));
                }

                if (idString.matches("^group_\\d+$")) {
                    return new Group(Long.parseLong(idString.substring(6)));
                }

                if (idString.contains("app_")) {
                    String[] parts = idString.split("_");
                    if (parts.length < 2) {
                        Log.e("SPCoreModuleLoader", Trace.once().method("IDFactory.Extension").msg("AppSession")
                                .info("error", "[Format error]: format should be app/xxx"));
                        return null;
                    }
                    long idValue = Long.parseLong(parts[1]);
                    String typeValue = parts[2];

                    return new AppSession(typeValue, idValue);
                }

                return null;
            }

            @Override
            public String makeString(Identifiable id) {
                if (id instanceof User) {
                    return String.format(Locale.US, "%s_%d", "user", id.getId());
                } else if (id instanceof Group) {
                    return String.format(Locale.US, "%s_%d", "group", id.getId());
                } else if (id instanceof AppSession) {
                    return String.format(Locale.US, "%s_%d_%s", "app", id.getId(),
                            ((AppSession) id).getType());
                }

                return null;
            }
        });
    }

    private static void registerChatServiceExtensions() {
        ChatServiceImpl.registerSendingExtension(new ChatServiceImpl.SendingExtension() {
            @Override
            public Channel.RPC makeSendingRPC(Message message, RichCompletion completion) {
                // 目前可靠离线消息仅支持，P2P消息
                if (message.getSender() instanceof User && message.getReceiver() instanceof User) {
                    return new RPCSendMessage(message, completion);
                }

                if (message.getReceiver() instanceof AppSession) {
                    return new RPCSendAppSessionMessage(message, completion);
                }

                return null;
            }
        });

        ChatServiceImpl.registerParserExtension(new ChatServiceImpl.ParserExtension() {
            @Override
            public String toString() {
                return "Core";
            }

            @Override
            public Message parseMessage(String serviceName, String functionName, byte[] data) {
                return null;
            }

            @Override
            public Message parseMessage(Im.Msg msg, Source fromSource) {
                return MessageParser.parseContent(msg.getContent(), msg.getTimestamp(), msg.getAction());
            }

            private PushContent parsePushContent(Im.OsPushMsg pushMsg) {
                if (pushMsg == null) {
                    return null;
                }

                return new PushContent(
                        pushMsg.getTitle(),
                        pushMsg.getContent(),
                        pushMsg.getPayload().toByteArray(),
                        pushMsg.getIcon());
            }
        });
    }

}
