package com.hummer.im._internals;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.google.protobuf.MessageLite;
import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.proto.Im;
import com.hummer.im._internals.shared.HiidoReporter;
import com.hummer.im._internals.shared.StringChain;
import com.hummer.im.service.Channel;

import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.util.Locale;

public abstract class IMRPC<Request, RequestBuilder, Response> implements Channel.RPC {

    public abstract void buildHummerRequest(@NonNull RequestBuilder builder) throws Throwable;

    public String describeHummerRequest(Request req) {
        return req.toString();
    }

    public abstract void handleHummerSuccess(@NonNull Response res) throws Throwable;

    public abstract void handleHummerError(@Nullable Response res, @NonNull Error error);

    public abstract String getHummerFunction();

    public String describeHummerResponse(@NonNull Response res) {
        return res.toString();
    }

    @Override
    public String serviceName() {
        HMRContext.Region env = HMRContext.region;
        return String.format(Locale.US, "svc_%s_cim_%s_%s_proxy", env.area, env.type, env.name);
    }

    @Override
    public String getFunctionName() {
        return FunctionNamePrefix + "." + getHummerFunction();
    }

    @Override
    public String protoType() {
        return "yy-rp-pb";
    }

    public long getLogId() {
        return flow.logId;
    }

    @Override
    @SuppressWarnings("unchecked")
    public byte[] getRequestBytes() throws Throwable {
        Class requestClazz = (Class) ((ParameterizedType) this.getClass()
                .getGenericSuperclass())
                .getActualTypeArguments()[0];

        Method newBuilder = requestClazz.getMethod("newBuilder");

        RequestBuilder builder = (RequestBuilder) newBuilder.invoke(null);
        if (builder == null) {
            Log.e("IMRPC", Trace.once().method("getRequestBytes")
                    .info("RequestBuilder", null));
        }

        Method setLogId = builder.getClass().getMethod("setLogId", long.class);
        if (setLogId == null) {
            Log.e("IMRPC", Trace.once().method("getRequestBytes")
                    .info("Method setLogId", null));
        }
        setLogId.invoke(builder, getLogId());

        Method setAppId = builder.getClass().getMethod("setAppId", long.class);
        if (setAppId == null) {
            Log.e("IMRPC", Trace.once().method("getRequestBytes")
                    .info("Method setAppId", null));
        }
        setAppId.invoke(builder, HMRContext.appId);

        try {
            Method setSelfUid = builder.getClass().getMethod("setSelfUid", long.class);
            setSelfUid.invoke(builder, HMR.getMe().getId());
        } catch (NoSuchMethodException e) {
            // setSelfUid 不是必须的
        }

        Method build = builder.getClass().getMethod("build");
        if (builder == null) {
            Log.e("IMRPC", Trace.once().method("getRequestBytes")
                    .info("Methodbuilder", null));
        }

        buildHummerRequest(builder);
        MessageLite request = (MessageLite) build.invoke(builder);

        Log.i("IMRPC", flow.trace()
                .method(getHummerFunction())
                .msg("-> {" + describeHummerRequest((Request) request)) + "}");

        startTs = System.currentTimeMillis();

        return request.toByteArray();
    }

    @Override
    public void handleError(@NonNull Error err) {
        handleHummerError(null, err);
    }

    @Override
    @SuppressWarnings("unchecked")
    public void handleResponse(@NonNull byte[] responseBytes) throws Throwable {
        Class responseClass = (Class) ((ParameterizedType) this.getClass()
                .getGenericSuperclass())
                .getActualTypeArguments()[2];

        if (responseClass == null) {
            Log.e("IMRPC", Trace.once().method("handleResponse")
                    .info("错误的HummerRPC继承实现: ", null));
        }

        Method newBuilder = responseClass.getMethod("newBuilder");
        if (newBuilder == null) {
            Log.e("IMRPC", Trace.once().method("handleResponse")
                    .info("Method newBuilder: ", null));
        }

        Object builder = newBuilder.invoke(null);
        if (builder == null) {
            Log.e("IMRPC", Trace.once().method("handleResponse")
                    .info("Object builder: ", null));
        }

        Method mergeFrom = builder.getClass().getMethod("mergeFrom", byte[].class);
        if (mergeFrom == null) {
            Log.e("IMRPC", Trace.once().method("handleResponse")
                    .info("Method mergeFrom: ", null));
        }

        Method build = builder.getClass().getMethod("build");
        if (build == null) {
            Log.e("IMRPC", Trace.once().method("handleResponse")
                    .info("Method build: ", null));
        }

        mergeFrom.invoke(builder, (Object) responseBytes);
        Response response = (Response) build.invoke(builder);

        Method getCode = response.getClass().getMethod("getCode");
        Method getMsg = response.getClass().getMethod("getMsg");
        if (getCode == null) {
            Log.e("IMRPC", Trace.once().method("handleResponse")
                    .info("Method getCode: ", getCode));
        }

        if (getMsg == null) {
            Log.e("IMRPC", Trace.once().method("handleResponse")
                    .info("Method getMsg: ", getMsg));
        }

        Integer code = (Integer) getCode.invoke(response);
        String msg = (String) getMsg.invoke(response);


        long rtt = System.currentTimeMillis() - startTs;

        if (code == Im.Code.kOk_VALUE) {
            Log.i("IMRPC", flow.trace()
                    .method(getHummerFunction())
                    .msg("<- Succeed {" + describeHummerResponse(response)) + "}");

            handleHummerSuccess(response);
        } else {
            rtt = 0;
            Error error = new Error(code, msg);
            Log.e("IMRPC", error, flow.trace().method(getHummerFunction())
                    .msg("<- Failed {" + new StringChain()
                            .add("logId", getLogId())
                            .add("response", describeHummerResponse(response))
                            .toString() + "}"));

            handleHummerError(response, error);
        }

        HiidoReporter.reportReturnCodeTemporary(getHummerFunction(), rtt, code);
    }

    private Trace.Flow flow = new Trace.Flow();
    private long startTs;
    private static final String FunctionNamePrefix = "cim.proto.ProxyService";
}
