package com.hummer.im;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.util.SparseIntArray;

import com.google.protobuf.InvalidProtocolBufferException;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.HummerException;
import com.hummer.im._internals.IMRPC;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.services.user.UserService;
import com.hummer.im._internals.shared.HiidoReporter;
import com.hummer.im.model.auth.TokenProvider;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.OnFailure;
import com.hummer.im.model.completion.OnSuccess;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.id.User;
import com.hummer.im.service.Channel;
import com.yy.platform.baseservice.ConstCode;
import com.yy.platform.baseservice.IChannelListener;
import com.yy.platform.baseservice.IRPCChannel;
import com.yy.platform.baseservice.YYServiceCore;
import com.yy.platform.baseservice.profile.ChannelProfile;
import com.yy.platform.baseservice.profile.LogProfile;
import com.yy.platform.baseservice.profile.ServiceProfileFactory;
import com.yy.platform.baseservice.task.BindTask;
import com.yy.platform.baseservice.task.BroadSubOrUnSubTaskV2;
import com.yy.platform.baseservice.task.RPCTask;
import com.yy.platform.baseservice.task.UnBindTask;
import com.yy.platform.baseservice.utils.UserGroupTypeString;
import com.yy.yylogger.Logger;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 用于实现Hummer内部通信的具体通道
 */
public class ServiceChannel implements Channel {

    @Override
    public Class[] staticDependencies() {
        return null;
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{UserService.class};
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return null;
    }

    @Override
    public void initService() {
        if (mode instanceof AutonomousMode) {
            Logger.createModule("Service", false);

            //如果未有中间件或其他模块初始化ServiceSDK，则由本模块启动初始化ServiceSDK并上报数据
            if (YYServiceCore.getInstance() == null) {
                YYServiceCore.initWithGSLB(HMRContext.getAppContext(), HMRContext.appId,
                        "", new ServiceProfileFactory() {
                            @Override
                            public LogProfile logProfile() {
                                return new LogProfile() {
                                    @Override
                                    public ILog getLog() {
                                        return new ILog() {
                                            @Override
                                            public void outputLog(String log) {
                                                Logger.i("Service", log);
                                            }
                                        };
                                    }

                                    @Override
                                    public String logPath() {
                                        return null;
                                    }

                                    @Override
                                    public boolean isLogCat() {
                                        return false;
                                    }
                                };
                            }

                            @Override
                            public ChannelProfile channelProfile() {
                                return null;
                            }
                        }, null);
            }
        }
    }

    public interface Mode {
        void performOpening(@NonNull User me, @NonNull RichCompletion completion);
        void performClosing();
    }

    /**
     * 该模式下，Hummer只是简单地使用通道进行通信，不会进行鉴权处理。如果通同一个App内有多个子系统都接入了Service通道，则鉴权处理应该在更高层的地方进行处理，并且使用该模式进行工作
     */
    @SuppressWarnings("unused")
    public static final class DelegateMode implements Mode {
        @Override
        public void performOpening(@NonNull User me, @NonNull RichCompletion completion) {
            CompletionUtils.dispatchSuccess(completion);
        }

        @Override
        public void performClosing() {
        }
    }

    /**
     * 该模式下，Hummer会拥有通道的完整控制权，包括如何进行用户鉴权等。如果整个App只接入了一个使用Service通道的SDK，可以使用该种模式
     */
    @SuppressWarnings("unused")
    public static final class AutonomousMode implements Mode {
        /**
         * 国际版UDB的service token，使用用yyloginlitesdk 的getServiceToken 接口得到token
         */
        public final static int NEW_UDB_TOKEN = 0;
        /**
         * 第三方产的service token
         */
        public final static int THIRD_USER_TOKEN = 1;
        /**
         * YY体系（国内UDB）下的linkd token
         */
        public final static int YY_UDB_TOKEN = 2;

        public AutonomousMode(int tokenType) {
            this.tokenType = tokenType;
        }

        public AutonomousMode(int tokenType, TokenProvider tokenProvider) {
            this.tokenProvider = tokenProvider;
            this.tokenType = tokenType;
        }

        @Override
        public void performOpening(@NonNull User me, @NonNull final RichCompletion completion) {
            if (me.isAnonymous()) {
                CompletionUtils.dispatchSuccess(completion);
                return;
            }

            Log.i(TAG, Trace.once().method("performOpening").msg("bind service start"));
            YYServiceCore.getInstance().bind(
                    me.getId(),
                    tokenType,
                    new IChannelListener.ITokenProvider() {
                        @Override
                        public byte[] getToken(long uid) {
                            return tokenProvider.getToken(uid);
                        }
                    },
                    new IRPCChannel.RPCCallback<BindTask.ResponseParam>() {

                        @Override
                        public void onSuccess(int requestId, BindTask.ResponseParam response) {
                            Log.i(TAG, Trace.once().method("performOpening").msg("bind service done"));
                            CompletionUtils.dispatchSuccess(completion);
                        }

                        @Override
                        public void onFail(int requestId, int sdkResCode, int srvResCode,
                                           Exception e) {
                            Log.i(TAG, Trace.once().method("performOpening").msg("bind service fail"));
                            CompletionUtils.dispatchFailure(completion,
                                    new Error(sdkResCode, e.getLocalizedMessage()));
                        }
                    });
        }

        @Override
        public void performClosing() {
            if (HMR.getMe().isAnonymous()) {
                return;
            }

            // 发现callback不能设置为null否则会崩溃
            Log.i(TAG, Trace.once().method("performClosing").msg("unbind service start"));
            YYServiceCore.getInstance()
                    .unBind(new IRPCChannel.RPCCallback<UnBindTask.ResponseParam>() {
                        @Override
                        public void onSuccess(int requestId, UnBindTask.ResponseParam response) {
                            Log.i(TAG, Trace.once().method("performClosing").msg("unbind service done"));
                        }

                        @Override
                        public void onFail(int requestId, int sdkResCode, int srvResCode,
                                           Exception e) {
                            Log.i(TAG, Trace.once().method("performClosing").msg("unbind service fail"));
                        }
                    });
        }

        private TokenProvider tokenProvider;
        private final int tokenType;

        public TokenProvider getTokenProvider() {
            return tokenProvider;
        }

        public void setTokenProvider(TokenProvider tokenProvider) {
            this.tokenProvider = tokenProvider;
        }
    }

    public ServiceChannel(Mode mode) {
        if (mode == null) {
            Log.e("ServiceChannel", Trace.once().method("ServiceChannel")
                    .info("mode", "null"));
        }

        this.mode = mode;
    }

    @Override
    public void openService(@NonNull final RichCompletion completion) {
        YYServiceCore.getInstance().registUnicastListener(unicastNotifyHandler);
        YYServiceCore.getInstance().registBroadcastListener(bcNotifyHandler);
        YYServiceCore.getInstance().registBroadcastListener(strGroupBroadcastNotifyHandler);

        HMRContext.work.async("ServiceChannel::openService", new Runnable() {
            @Override
            public void run() {
                mode.performOpening(HMR.getMe(), new RichCompletion("")
                        .onSuccess(new OnSuccess() {
                            @Override
                            public void onSuccess() {
                                CompletionUtils.dispatchSuccess(completion);
                            }
                        })
                        .onFailure(new OnFailure() {
                            @Override
                            public void onFailure(Error error) {
                                YYServiceCore.getInstance().unregistBroadcastListener(bcNotifyHandler);
                                YYServiceCore.getInstance().unregistUnicastListener(unicastNotifyHandler);
                                YYServiceCore.getInstance().unregistBroadcastListener(strGroupBroadcastNotifyHandler);
                                CompletionUtils.dispatchFailure(completion, error);
                            }
                        }));
            }
        });
    }

    @Override
    public void closeService() {
        YYServiceCore.getInstance().unregistBroadcastListener(bcNotifyHandler);
        YYServiceCore.getInstance().unregistUnicastListener(unicastNotifyHandler);
        YYServiceCore.getInstance().unregistBroadcastListener(strGroupBroadcastNotifyHandler);
        this.mode.performClosing();
    }

    @Override
    public void run(final Channel.RPC rpc) {
        Error error = null;

        final User currentMe = HMR.getMe();
        if (currentMe == null) {
            error = new Error(Error.Code.InvalidParameters, "Missing login user!");
        }

        if (!isNetworkConnected()) {
            error = new Error(Error.Code.NetworkNotFound, "Network disconnected");
        }

        byte[] requestPayLoad = null;
        try {
            requestPayLoad = rpc.getRequestBytes();
        } catch (Throwable t) {
            // 异常时 requestPayload是null，交给下面的统一判断处理
        }

        if (requestPayLoad == null) {
            error = new Error(Error.Code.InvalidParameters, "Failed parsing data from request", rpc);
        }

        if (error != null) {
            Log.e(TAG, Trace.once().method("runRPC").msg(error.toString()));
            rpc.handleError(error);
            return;
        }

        RPCTask.RequestParam requestParam = new RPCTask.RequestParam(
                "",
                rpc.serviceName(),
                rpc.getFunctionName(),
                requestPayLoad,
                rpc.protoType(),
                null,
                null,
                null);

        final long startTs = System.currentTimeMillis();
        final String shortFunctionName = shortFunctionName(rpc);
        Log.i(TAG, Trace.once("start call rpc timestamp:%s, shortFunctionName:%s", startTs, shortFunctionName));

        YYServiceCore.getInstance()
                .rpcCall(requestParam, null, new IRPCChannel.RPCCallback<RPCTask.ResponseParam>() {
                    @Override
                    public void onSuccess(int requestId, final RPCTask.ResponseParam response) {
                        HMRContext.work.async("ServiceChannel::rpcSuccess:" + shortFunctionName, new Runnable() {
                            @Override
                            public void run() {
                                Error error = null;

                                try {
                                    if (!HMR.isMe(currentMe)) {
                                        error = new Error(Error.Code.ClientExceptions, "RPC overdue!",
                                                rpc);
                                    } else {
                                        rpc.handleResponse(response.mResponseData);
                                    }
                                } catch (InvalidProtocolBufferException e) {
                                    error = new Error(Error.Code.ProtocolExceptions,
                                            "Protocol exceptions", e);
                                } catch (HummerException e) {
                                    error = e.error;
                                } catch (Throwable t) {
                                    error = new Error(Error.Code.UndefinedExceptions,
                                            "Undefined exceptions", t);
                                }

                                if (error != null) {
                                    rpc.handleError(error);

                                    HiidoReporter.reportReturnCodeTemporary(shortFunctionName, 0, error.code);
                                }
                            }
                        });
                    }

                    @Override
                    public void onFail(int requestId, final int sdkResCode, final int srvResCode, Exception e) {
                        HMRContext.work.async("ServiceChannel::rpcFail:" + shortFunctionName, new Runnable() {
                            @Override
                            public void run() {
                                if (!HMR.isMe(currentMe)) {
                                    Error err = new Error(Error.Code.ClientExceptions, "RPC overdue!");

                                    Log.w(TAG, err.desc);
                                    rpc.handleError(err);
                                    return;
                                }

                                Error error;

                                if (sdkResCode != ConstCode.SdkResCode.RECV_RESPONSE) {
                                    error = errorFromServiceFailure(true, sdkResCode);
                                } else {
                                    error = errorFromServiceFailure(false, srvResCode);
                                }

                                rpc.handleError(error);
                                HiidoReporter.reportReturnCodeTemporary(shortFunctionName, 0, error.code);
                            }
                        });
                    }
                });
    }

    @Override
    public void addNotificationHandler(final Channel.NotificationHandler handler) {
        notifyHandlers.add(handler);
    }

    @Override
    public void removeNotificationHandler(Channel.NotificationHandler handler) {
        notifyHandlers.remove(handler);
    }

    @Override
    public void subscribeGroupcast(String group, final RichCompletion completion) {
        Set<UserGroupTypeString> grps = new HashSet<>();
        grps.add(new UserGroupTypeString(group));
        YYServiceCore.getInstance().subscribeStrBroadcast(grps,
                new IRPCChannel.RPCCallback<BroadSubOrUnSubTaskV2.ResponseParam>() {
                    @Override
                    public void onSuccess(int requestId, BroadSubOrUnSubTaskV2.ResponseParam response) {
                        CompletionUtils.dispatchSuccess(completion);
                    }

                    @Override
                    public void onFail(int requestId, int sdkResCode, int srvResCode, Exception e) {
                        CompletionUtils.dispatchFailure(completion,
                                new Error(sdkResCode, e.getLocalizedMessage()));
                    }
                });
    }

    @Override
    public void unSubscribeGroupcast(String group, final RichCompletion completion) {
        Set<UserGroupTypeString> grps = new HashSet<>();
        grps.add(new UserGroupTypeString(group));
        YYServiceCore.getInstance().unSubscribeStrBroadcast(grps,
                new IRPCChannel.RPCCallback<BroadSubOrUnSubTaskV2.ResponseParam>() {
                    @Override
                    public void onSuccess(int requestId, BroadSubOrUnSubTaskV2.ResponseParam response) {
                        CompletionUtils.dispatchSuccess(completion);
                    }

                    @Override
                    public void onFail(int requestId, int sdkResCode, int srvResCode, Exception e) {
                        CompletionUtils.dispatchFailure(completion,
                                new Error(sdkResCode, e.getLocalizedMessage()));
                    }
                });
    }

    @Override
    public void addStateListener(final StateChangedListener listener) {
        IChannelListener.IChannelStatusNotify stateHandler = new IChannelListener.IChannelStatusNotify() {
            @Override
            public void onStatus(int status) {
                if (status == ConstCode.Status.Binded) {
                    listener.onChannelConnected();
                } else {
                    listener.onChannelDisconnected();
                }
            }
        };

        stateListeners.put(listener, stateHandler);

        YYServiceCore.getInstance().registChannelStatusListener(stateHandler, new Handler());
    }

    @Override
    public void removeStateListener(StateChangedListener listener) {
        IChannelListener.IChannelStatusNotify stateHandler = stateListeners.get(listener);
        if (stateHandler == null) {
            return;
        }

        YYServiceCore.getInstance().unregistChannelStatusListener(stateHandler);
        stateListeners.remove(listener);
    }

    private static String shortFunctionName(Channel.RPC rpc) {
        return (rpc instanceof IMRPC) ? ((IMRPC) rpc).getHummerFunction() : rpc.getFunctionName();
    }

    private void onNotify(final String serviceName, final String functionName, final byte[] data) {
        HMRContext.work.async("ServiceChannel::onNotify:" + serviceName + "/" + functionName, new Runnable() {
            @Override
            public void run() {
                for (NotificationHandler handler : notifyHandlers) {
                    handler.onNotify(serviceName, functionName, data);
                }
            }
        });
    }

    private boolean isNetworkConnected() {
        ConnectivityManager connectivityManager = (ConnectivityManager) HMRContext.getAppContext()
                .getSystemService(Context.CONNECTIVITY_SERVICE);

        if (connectivityManager == null) {
            return false;
        }

        NetworkInfo mNetworkInfo = connectivityManager.getActiveNetworkInfo();
        if (mNetworkInfo == null) {
            return false;
        }

        return mNetworkInfo.isAvailable();
    }

    private Error errorFromServiceFailure(boolean fromSDK, int code) {
        String message;
        if (fromSDK) {
            SparseIntArray codeMapping = new SparseIntArray() {{
                put(ConstCode.SdkResCode.NO_CONNECTED, Error.Code.ConnectionFailed);
                put(ConstCode.SdkResCode.NEED_BIND, Error.Code.Unauthorized);
                put(ConstCode.SdkResCode.TIMEOUT, Error.Code.OperationTimeout);
            }};

            message = ConstCode.SdkResCode.desc(code);
            code = codeMapping.get(code, Error.Code.IOError);
        } else {
            message = ConstCode.SrvResCode.desc(code);
            code = Error.Code.InternalServerExceptions;
        }

        return new Error(code, message);
    }

    private static final String TAG = "ServiceChannel";

    @SuppressWarnings("CodeBlock2Expr")
    private final IChannelListener.IServiceBroadcastNotify bcNotifyHandler =
            new IChannelListener.IServiceBroadcastNotify() {
                @Override
                public void onBroadCast(long uid, long grpType,
                                        long grpId, String serviceName,
                                        String functionName, String protoType, byte[] data) {
                    ServiceChannel.this.onNotify(serviceName, functionName, data);
                }
            };

    @SuppressWarnings("CodeBlock2Expr")
    private final IChannelListener.IServiceUnicastNotify unicastNotifyHandler =
            new IChannelListener.IServiceUnicastNotify() {
                @Override
                public void onUnicast(long uid, String serviceName,
                                      String functionName, String protoType, byte[] data) {
                    ServiceChannel.this.onNotify(serviceName, functionName, data);
                }
            };

    private final IChannelListener.IServiceStrGroupBroadcastNotify strGroupBroadcastNotifyHandler =
            new IChannelListener.IServiceStrGroupBroadcastNotify() {
                @Override
                public void onBroadCastFromStrGroup(long uid, String grpStr, String serviceName, String functionName,
                                                    String protoType, byte[] data) {
                    ServiceChannel.this.onNotify(serviceName, functionName, data);
                }
            };

    private static Map<StateChangedListener, IChannelListener.IChannelStatusNotify> stateListeners =
            new HashMap<>();

    private static List<NotificationHandler> notifyHandlers = new LinkedList<>();

    private Mode mode;

    public Mode getMode() {
        return mode;
    }
}
