package com.hummer.im;

import java.util.Locale;

/**
 * 该Error类型是Hummer中用于表示通用错误的类型，常见于消息收发等需要网络通讯的请求中.
 */
@SuppressWarnings({"unused", "WeakerAccess"})
public final class Error {

    /**
     * code属性主要用于进行机器判别、日志快速定位等作用
     */
    public final int code;

    /**
     * reason属性通过人可读的文本形式对mCode进行了描述
     */
    public final String desc;

    /**
     * extra对象用于附着一些特殊的错误上下文信息，例如网络错误、JSON解析错误等造成的异常。将它们附着到extra
     * 中，可以帮助使用者找到更全的信息，以便快速定位问题发生的具体原因。
     *
     * 请注意，extra对象是易挥发的，只在运行时有效，不会被存放到任何持久化数据中，例如数据库
     */
    public final Object extra;

    /**
     * 构造Error实例，
     *
     * @param code 处理结果的代码
     * @param desc 处理结果的文本描述
     */
    public Error(int code, String desc) {
        this(code, desc, null);
    }

    /**
     * 构造Error实例
     *
     * @param code  处理结果的代码
     * @param desc  处理结果的文本描述
     * @param extra 额外的处理结果上下文
     */
    public Error(int code, String desc, Object extra) {
        this.code  = code;
        this.desc  = desc;
        this.extra = extra;
    }

    @Override
    public String toString() {
        return String.format(Locale.US, "Error{%d, %s}", code, desc);
    }

    public @interface Code {
        int Success = 0;

        /**
         * 通用客户端错误，如果在下面的错误码表中没有更匹配的错误类型，则会使用该错误码进行表示
         */
        int ClientExceptions = 1000;

        /**
         * 表示Hummer未被正确初始化
         */
        int Uninitialized = 1001;

        /**
         * 请求Hummer服务时，提供了无效的错误参数
         */
        int InvalidParameters = 1002;

        /**
         * 表示遇到了本地IO操作相关的错误，例如数据库访问异常等
         */
        int IOError = 1003;

        /**
         * 如果发起需要网络服务，但是请求时网络不可达，则会返回该错误，建议提示用户检查网络连接并重试
         */
        int NetworkNotFound = 1004;

        /**
         * 网络操作超时，建议提示用户检查网络连接并重试
         */
        int OperationTimeout = 1005;

        /**
         * 网络连接超时，通常是由于网络虽然连通，但是因为网络条件比较恶劣导致的，建议提示用户检查网络连接并重试
         */
        int ConnectionTimeout = 1006;

        /**
         * 无法建立可用的网络连接
         */
        int ConnectionFailed = 1007;

        /**
         * 服务请求调用过于频繁，建议业务端需要进行频率控制
         */
        int Throttling = 1008;

        /**
         * 操作鉴权失败，通常意味着当前用户鉴权失败
         */
        int Unauthorized = 1009;

        /**
         * Hummer内调用第三方服务发生错误
         */
        int ThirdPartyServiceError = 1010;

        /**
         * 如果业务没有正确处理用户上下文切换，例如业务已经注销了用户，但是没有调用Hummer.close，则会产生该错误
         */
        int BadUser = 1011;

        /**
         * 传输协议异常，例如协议版本错误等
         */
        int ProtocolExceptions = 2000;

        /**
         * 协议内容校验失败，例如消息内容过长等
         */
        int InvalidContent = 2001;

        /**
         * 用于进行权限验证的Token无效
         */
        int TokenInvalid = 2002;

        /**
         * 用于进行权限验证的Token已经失效
         */
        int TokenExpired = 2003;

        /**
         * 请求访问的资源不存在
         */
        int ResourceNotFound = 2004;

        /**
         * 请求访问的资源已存在，通常在创建房间等场景出现
         */
        int ResourceAlreadyExist = 2005;

        /**
         * 资源、关系数量超出了限定值
         */
        int LimitExceeded = 2006;

        /**
         * 消息长度超出了上限
         */
        int MessageSizeLimitExceeded = 2007;

        /**
         * 访问被拒绝，经常出现在通信通道协议uid和实际请求业务uid不匹配的情况下
         */
        int AccessDenied = 3000;

        /**
         * 用户被列入黑名单，无法获得CIM服务
         */
        int Blacklisted = 3001;

        int kTemporarilyDeniedException = 3002; // 暂时没有权限
        int kForbiddenException = 3003; // 操作被禁止
        int kUserForbiddenException = 3004; // 用户操作被禁止
        int kBannedException = 3005;  // 操作被封禁
        int kChallengeException = 3006; // 需要输入参数进行验证
        int kInspectionFailedException = 3007; //审查失败

        /**
         * 服务器内部异常
         */
        int InternalServerExceptions = 4000;

        /**
         * 暂时无法提供IM服务，一般为服务器进程重启等原因导致
         */
        int ServiceUnavailable = 4001;

        int kBusinessServerError = 4002;
        int kServiceThrottling = 4003;

        int kElapseExceptionLevel1 = 5001;
        int kElapseExceptionLevel2 = 5002;
        int kElapseExceptionLevel3 = 5003;

        /**
         * 其它未定义异常类型
         */
        int UndefinedExceptions = -1;
    }
}
