package com.hummer.im.chatroom._internals;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.chatsvc.ChatServiceImpl;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.mq.Source;
import com.hummer.im._internals.proto.Im;
import com.hummer.im._internals.services.user.UserService;
import com.hummer.im._internals.shared.DispatchQueue;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im._internals.yyp.packet.Receiver;
import com.hummer.im._internals.yyp.packet.Uint64;
import com.hummer.im.chatroom.Challenges;
import com.hummer.im.chatroom.ChatRoomInfo;
import com.hummer.im.chatroom.ChatRoomService;
import com.hummer.im.chatroom._internals.proto.ChatRoomProto;
import com.hummer.im.chatroom._internals.rpc.RPCChangeChatRoomInfo;
import com.hummer.im.chatroom._internals.rpc.RPCChangeChatRoomRole;
import com.hummer.im.chatroom._internals.rpc.RPCChatRoomTextChat;
import com.hummer.im.chatroom._internals.rpc.RPCCreateChatRoom;
import com.hummer.im.chatroom._internals.rpc.RPCDisableUserText;
import com.hummer.im.chatroom._internals.rpc.RPCDismissChatRoom;
import com.hummer.im.chatroom._internals.rpc.RPCFetchAdminList;
import com.hummer.im.chatroom._internals.rpc.RPCFetchChatRoomInfo;
import com.hummer.im.chatroom._internals.rpc.RPCFetchMemberList;
import com.hummer.im.chatroom._internals.rpc.RPCFetchMutedMemberList;
import com.hummer.im.chatroom._internals.rpc.RPCJoinChatRoom;
import com.hummer.im.chatroom._internals.rpc.RPCKickOffUser;
import com.hummer.im.chatroom._internals.rpc.RPCLeaveChatRoom;
import com.hummer.im.chatroom._internals.rpc.RPCPullAllAdminUser;
import com.hummer.im.chatroom._internals.rpc.RPCSendBroadcast;
import com.hummer.im.chatroom._internals.rpc.RPCSendUnicast;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.contents.Text;
import com.hummer.im.model.chat.states.Archived;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.OnFailure;
import com.hummer.im.model.completion.OnSuccess;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.completion.RichCompletionArg;
import com.hummer.im.model.id.ChatRoom;
import com.hummer.im.model.id.User;
import com.hummer.im.model.kick.KickOff;
import com.hummer.im.model.kick.KickOffEnum;
import com.hummer.im.service.Channel;
import com.yy.platform.baseservice.IRPCChannel;
import com.yy.platform.baseservice.YYServiceCore;
import com.yy.platform.baseservice.task.BroadSubOrUnSubTask;
import com.yy.platform.baseservice.utils.UserGroupType;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

public class ChatRoomServiceImpl implements
        ChatRoomService,
        Channel.StateChangedListener,
        Channel.NotificationHandler,
        ServiceProvider.Service {
    private static final long kRejoinConditionGap = 60 * 1000;
    private static final long kReferenceValue = 946656000;
    private long mInitialTimeInterval = 0L;

    private static final int kickTimeSec = 300;
    private static final String TAG = "ChatRoomService";
    // 在线人数以及成员进出事件的 grpType
    private static final long onlineBCGrpType = 2147483650L;
    // 鉴权服务 （踢人，改频道消息等） 的 grpType
    private static final long authBCGrpType = 2147483649L;
    // 公屏消息 的 grpType
    private static final long textChatBCGrpType = 2147483648L;

    private static final int MUTED_TIME_SEC = 3 * 24 * 60 * 60;

    @Override
    public Class[] staticDependencies() {
        return new Class[]{Channel.class};
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{UserService.class, Channel.class};
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return null;
    }

    @Override
    public void initService() {
        registerChatServiceExtensions();
    }

    @Override
    public void openService(@NonNull RichCompletion completion) {
        HMR.getService(Channel.class).addStateListener(this);
        HMR.getService(Channel.class).addNotificationHandler(this);

        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void closeService() {
        if (chatRooms.size() > 0) {
            for (ChatRoom chatRoom : chatRooms) {
                Log.i(TAG, String.format(Locale.US,
                        "leave chatRoom after closeService, roomId:%d",
                        chatRoom.getId()));
                leaveChatRoom(chatRoom, null);
            }

            chatRooms.clear();
        }
        resetInitialTimeInterval();
        HMR.getService(Channel.class).removeNotificationHandler(this);
        HMR.getService(Channel.class).removeStateListener(this);
    }

    private void registerChatServiceExtensions() {
        ChatServiceImpl.registerSendingExtension(new ChatServiceImpl.SendingExtension() {
            @Override
            public Channel.RPC makeSendingRPC(final Message message, RichCompletion completion) {
                if (!(message.getSender() instanceof User && message.getReceiver() instanceof ChatRoom)) {
                    return null;
                }

                ChatRoom room = (ChatRoom) message.getReceiver();

                if (message.getContent() instanceof Text) {
                    return new RPCChatRoomTextChat(
                            room.getId(),
                            ((Text) message.getContent()).getText(),
                            new HashMap<String, String>() {{
                                put("uuid", message.getUuid());

                                if (message.getAppExtra() != null) {
                                    put("extra", message.getAppExtra());
                                }
                            }},
                            null,
                            completion);
                } else if (message.getContent() instanceof Signal) {
                    Signal sig = (Signal) message.getContent();
                    if (sig.user == null) {
                        return new RPCSendBroadcast(room.getId(), sig.content, completion);
                    } else {
                        return new RPCSendUnicast(room.getId(), sig.content, sig.user.getId(), completion);
                    }
                }
                return null;
            }
        });

        ChatServiceImpl.registerParserExtension(new ChatServiceImpl.ParserExtension() {
            @Override
            public String toString() {
                return "ChatRoom";
            }

            @Override
            public Message parseMessage(String service, String function, byte[] data) {
                if ("chatroom_textchat".equals(service) && "TextChatBc".equals(function)) {
                    Receiver receiver = new Receiver(data);
                    receiver.parseHeader();
                    if (receiver.getUri().intValue() != ChatRoomProto.PCS_TextChatBc.uri) {
                        return null;
                    }

                    ChatRoomProto.PCS_TextChatBc bc = new ChatRoomProto.PCS_TextChatBc();
                    receiver.unmarshallWrap(bc);

                    ChatRoom chatRoom = new ChatRoom(bc.roomId.longValue());
                    Text text = new Text(bc.chat);
                    Message chatMessage = new Message(chatRoom, text);
                    chatMessage.setState(new Archived());
                    chatMessage.setSender(new User(bc.uid.longValue()));
                    String uuid = bc.chatProps.get("uuid");
                    if (TextUtils.isEmpty(uuid)) {
                        chatMessage.setUuid(UUID.randomUUID().toString());
                    } else {
                        chatMessage.setUuid(bc.chatProps.get("uuid"));
                    }
                    chatMessage.setAppExtra(bc.chatProps.get("extra"));
                    chatMessage.setTimestamp(System.currentTimeMillis());
                    return chatMessage;
                } else if ("chatroom_auther".equals(service) && "SendUnicastUc".equals(function)) {
                    Receiver receiver = new Receiver(data);
                    receiver.parseHeader();
                    if (receiver.getUri().intValue() != ChatRoomProto.PCS_SendUnicastUc.uri) {
                        return null;
                    }

                    ChatRoomProto.PCS_SendUnicastUc uc = new ChatRoomProto.PCS_SendUnicastUc();
                    receiver.unmarshallWrap(uc);

                    Message msg = new Message(new ChatRoom(
                            uc.roomId.longValue()),
                            ChatRoomService.Signal.unicast(new User(uc.receiver.longValue()), uc.content)
                    );

                    msg.setSender(new User(uc.uid.longValue()));
                    msg.setUuid(UUID.randomUUID().toString());
                    msg.setTimestamp(System.currentTimeMillis());
                    return msg;
                } else if ("chatroom_auther".equals(service) && "SendBroadcastBc".equals(function)) {
                    Receiver receiver = new Receiver(data);
                    receiver.parseHeader();
                    if (receiver.getUri().intValue() != ChatRoomProto.PCS_SendBroadcastBc.uri) {
                        return null;
                    }

                    ChatRoomProto.PCS_SendBroadcastBc bc = new ChatRoomProto.PCS_SendBroadcastBc();
                    receiver.unmarshallWrap(bc);

                    Message msg = new Message(new ChatRoom(
                            bc.roomId.longValue()),
                            ChatRoomService.Signal.broadcast(bc.content)
                    );
                    msg.setSender(new User(bc.uid.longValue()));
                    msg.setUuid(UUID.randomUUID().toString());
                    msg.setTimestamp(System.currentTimeMillis());
                    return msg;
                }

                return null;
            }

            @Override
            public Message parseMessage(Im.Msg msg, Source fromSource) {
                return null;
            }
        });
    }

    @Override
    public void switchRegion(String region) {
        HMRContext.chatRoomRegion = region.toLowerCase(Locale.US);
    }

    @Override
    public void createChatRoom(@NonNull ChatRoomInfo chatRoomInfo,
                               @NonNull final HMR.CompletionArg<ChatRoom> completion) {
        RichCompletionArg<ChatRoom> rCompletion = new RichCompletionArg<>(completion,
                "CHatRoomService::createChatRoom");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(
                    rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode")
            );
            return;
        }

        Log.i(TAG, Trace.once("createChatRoom subject:%s", chatRoomInfo.getName()));
        ServiceProvider.get(Channel.class).run(new RPCCreateChatRoom(chatRoomInfo, rCompletion));
    }

    @Override
    public void dismissChatRoom(@NonNull ChatRoom chatRoom,
                                @Nullable HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion, "ChatRoomService::dismissChatRoom");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(
                    rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode")
            );
            return;
        }

        Log.i(TAG, Trace.once("deleteChatRoom chatRoom: %s", chatRoom.getId()));
        ServiceProvider.get(Channel.class).run(new RPCDismissChatRoom(
                Long.valueOf(chatRoom.getId()).intValue(),
                rCompletion
        ));
    }

    @Override
    public void join(@NonNull final ChatRoom chatRoom,
                     @NonNull final Map<String, String> joinProps,
                     @NonNull final Challenges.JoiningCompletion completion) {
        Log.i(TAG, Trace.once("join chatRoom: %s", chatRoom.getId()));
        if (HMR.getState() != HMR.State.Opened) {
            DispatchQueue.main.async("ChatRoomService::joinFailCallback", new Runnable() {
                @Override
                public void run() {
                    completion.onFailure(new Error(
                            Error.Code.BadUser, "User not login")
                    );
                }
            });
            return;
        }

        if (HMR.getMe().isAnonymous()) {
            HMRContext.work.async("ChatRoomService::subscribe", new Runnable() {
                @Override
                public void run() {
                    ChatRoomServiceImpl.this._subscribeChatRoomBC(chatRoom,
                            new RichCompletion("ChatRoomService::subscribe")
                                    .onSuccess(new OnSuccess() {
                                        @Override
                                        public void onSuccess() {
                                            ChatRoomServiceImpl.this.addChatRoom(chatRoom);
                                            DispatchQueue.main.async("ChatRoomService::subscribeSuccessCallback",
                                                    new Runnable() {
                                                        @Override
                                                        public void run() {
                                                            completion.onSucceed();
                                                        }
                                                    });
                                        }
                                    })
                                    .onFailure(new OnFailure() {
                                        @Override
                                        public void onFailure(final Error error) {
                                            DispatchQueue.main.async("ChatRoomService::subscribeFailureCallback", new
                                                    Runnable() {
                                                        @Override
                                                        public void run() {
                                                            completion.onFailure(error);
                                                        }
                                                    });
                                        }
                                    }));
                }
            });
        } else {
            HMRContext.work.async("ChatRoomService::join", new Runnable() {
                @Override
                public void run() {
                    ServiceProvider.get(Channel.class).run(new RPCJoinChatRoom(
                            (int) chatRoom.getId(),
                            joinProps,
                            new Challenges.JoiningCompletion() {
                                @Override
                                public void onReceiveChallenge(final Challenges.Password challenge) {
                                    DispatchQueue.main.async("ChatRoomService::onReceiveChallengePassword",
                                            new Runnable() {
                                                @Override
                                                public void run() {
                                                    completion.onReceiveChallenge(challenge);
                                                }
                                            });
                                }

                                @Override
                                public void onReceiveChallenge(final Challenges.AppChallenge challenge) {
                                    DispatchQueue.main
                                            .async("ChatRoomService::onReceiveChallengeAppChallenge", new Runnable() {
                                                @Override
                                                public void run() {
                                                    completion.onReceiveChallenge(challenge);
                                                }
                                            });
                                }

                                @Override
                                public void onSucceed() {
                                    addChatRoom(chatRoom);
                                    _subscribeChatRoomBC(chatRoom, new RichCompletion("ChatRoomService::joinSuccess")
                                            .onSuccess(new OnSuccess() {
                                                @Override
                                                public void onSuccess() {
                                                    DispatchQueue.main.async("ChatRoomService::joinSuccess", new
                                                            Runnable() {
                                                                @Override
                                                                public void run() {
                                                                    completion.onSucceed();
                                                                }
                                                            });
                                                }
                                            })
                                            .onFailure(new OnFailure() {
                                                @Override
                                                public void onFailure(Error error) {
                                                    onFailure(error);
                                                }
                                            })
                                    );
                                }

                                @Override
                                public void onFailure(@NonNull final Error error) {
                                    DispatchQueue.main.async("ChatRoomService::joinFailure", new Runnable() {
                                        @Override
                                        public void run() {
                                            completion.onFailure(error);
                                        }
                                    });
                                }
                            }));
                }
            });
        }
    }

    @Override
    public void leave(@NonNull final ChatRoom chatRoom, @Nullable final HMR.Completion completion) {
        Log.i(TAG, Trace.once("leave chatRoom: %s", chatRoom.getId()));
        final RichCompletion rCompletion = new RichCompletion(completion, "ChatRoomService::leave");

        if (HMR.getState() != HMR.State.Opened) {
            CompletionUtils.dispatchFailure(rCompletion, new Error(
                    Error.Code.BadUser, "User not login")
            );
            return;
        }

        Log.i(TAG, Trace.once("leave chatRoom: %s", chatRoom.getId()));
        HMRContext.work.async("ChatRoomService::leave", new Runnable() {
            @Override
            public void run() {
                ChatRoomServiceImpl.this.leaveChatRoom(chatRoom, rCompletion);
                ChatRoomServiceImpl.this.removeChatRoom(chatRoom);
            }
        });
    }

    @Override
    public void kick(@NonNull final ChatRoom chatRoom,
                     @NonNull final User member,
                     @Nullable final Map<EKickInfo, String> extraInfo,
                     @NonNull final HMR.Completion completion) {
        final RichCompletion rCompletion = new RichCompletion(completion, "ChatRoomService::kick");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion, new Error(
                    Error.Code.BadUser, "User not login, or using anonymous mode")
            );
            return;
        }

        Log.i(TAG, Trace.once("kick id:%s fellow:%s", chatRoom.getId(), member.getId()));
        HMRContext.work.async("ChatRoomService::kick", new Runnable() {
            @Override
            public void run() {
                long secs = kickTimeSec;
                String reason = "";
                if (extraInfo != null && extraInfo.containsKey(EKickInfo.Time)) {
                    secs = Long.parseLong(extraInfo.get(EKickInfo.Time));
                }
                if (extraInfo != null && extraInfo.containsKey(EKickInfo.Reason)) {
                    reason = extraInfo.get(EKickInfo.Reason);
                }

                ServiceProvider.get(Channel.class).run(new RPCKickOffUser(
                        member.getId(),
                        (int) chatRoom.getId(),
                        secs,
                        reason,
                        rCompletion));
            }
        });
    }

    @Override
    public void addRole(@NonNull final ChatRoom chatRoom,
                        @NonNull final User member,
                        @NonNull final String role,
                        @NonNull final HMR.Completion completion) {
        final RichCompletion rCompletion = new RichCompletion(completion, "ChatRoomService::addRole");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion, new Error(
                    Error.Code.BadUser, "User not login, or using anonymous mode")
            );
            return;
        }

        Log.i(TAG, Trace.once("addRole id:%s fellow:%s role:%s",
                chatRoom.getId(),
                member.getId(),
                role));

        ServiceProvider.get(Channel.class).run(new RPCChangeChatRoomRole(
                chatRoom.getId(),
                member.getId(),
                new ChatRoomProto.AdminRoleTypeEnum(role),
                ChatRoomProto.ChatRoomOpEnum.ADD,
                rCompletion));
    }

    @Override
    public void removeRole(@NonNull final ChatRoom chatRoom, @NonNull final User member,
                           @NonNull final String role,
                           @NonNull final HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion, "ChatRoomService::removeRole");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion, new Error(
                    Error.Code.BadUser, "User not login, or using anonymous mode")
            );
            return;
        }

        Log.i(TAG, Trace.once("removeRole id:%s fellow:%s role:%s",
                chatRoom.getId(),
                member.getId(),
                role));

        ServiceProvider.get(Channel.class).run(new RPCChangeChatRoomRole(
                chatRoom.getId(),
                member.getId(),
                new ChatRoomProto.AdminRoleTypeEnum(role),
                ChatRoomProto.ChatRoomOpEnum.REMOVE,
                rCompletion));
    }

    @Override
    public void fetchMembers(@NonNull final ChatRoom chatRoom,
                             final int num,
                             final int offset,
                             @NonNull final HMR.CompletionArg<List<User>> completion) {
        final RichCompletionArg<List<User>> rCompletion = new RichCompletionArg<>(completion,
                "ChatRoomService::fetchMembers");
        if (HMR.getState() != HMR.State.Opened) {
            CompletionUtils.dispatchFailure(rCompletion, new Error(
                    Error.Code.BadUser, "User not login")
            );
            return;
        }

        Log.i(TAG, Trace.once("fetchMembers id:%s num:%s pos:%s", chatRoom.getId(), num, offset));
        HMRContext.work.async("ChatRoomService::fetchMembers", new Runnable() {
            @Override
            public void run() {
                ServiceProvider.get(Channel.class)
                        .run(new RPCFetchMemberList((int) chatRoom.getId(),
                                num, offset, rCompletion));
            }
        });
    }

    @Override
    public void fetchRoleMembers(@NonNull final ChatRoom chatRoom, final boolean online,
                                 @NonNull final HMR.CompletionArg<Map<String, List<User>>> completion) {
        final RichCompletionArg<Map<String, List<User>>> rCompletion =
                new RichCompletionArg<>(completion, "ChatRoomService::fetchRoleMembers");
        if (HMR.getState() != HMR.State.Opened) {
            CompletionUtils.dispatchFailure(rCompletion, new Error(
                    Error.Code.BadUser, "User not login")
            );
            return;
        }

        Log.i(TAG, Trace.once("fetch all roler Members id:%s, online:%s", chatRoom.getId(), online));

        if (online) {
            ServiceProvider.get(Channel.class).run(new RPCFetchAdminList(
                    (int) chatRoom.getId(), "all",
                    rCompletion));
        } else {
            ServiceProvider.get(Channel.class)
                    .run(new RPCPullAllAdminUser((int) chatRoom.getId(), "all",
                            rCompletion));
        }
    }

    @Override
    public void fetchBasicInfo(@NonNull final ChatRoom chatRoom,
                               @NonNull final HMR.CompletionArg<ChatRoomInfo> completion) {
        RichCompletionArg<ChatRoomInfo> rCompletion =
                new RichCompletionArg<>(completion, "ChatService::fetchBasicInfo");
        if (HMR.getState() != HMR.State.Opened) {
            CompletionUtils.dispatchFailure(rCompletion, new Error(
                    Error.Code.BadUser, "User not login")
            );
            return;
        }

        Log.i(TAG, Trace.once("fetchBasicInfo id:%s", chatRoom.getId()));

        ServiceProvider.get(Channel.class).run(new RPCFetchChatRoomInfo((int) chatRoom.getId(), rCompletion));
    }

    @Override
    public void changeBasicInfo(@NonNull final ChatRoom chatRoom,
                                @NonNull final Map<ChatRoomInfo.BasicInfoType, String> propInfo,
                                @Nullable final HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion, "ChatRoomService::changeBasicInfo");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion, new Error(
                    Error.Code.BadUser, "User not login")
            );
            return;
        }

        Log.i(TAG, Trace.once("changeBasicInfo id:%s", chatRoom.getId()));

        ServiceProvider.get(Channel.class).run(new RPCChangeChatRoomInfo(
                (int) chatRoom.getId(),
                propInfo,
                rCompletion));
    }

    @Override
    public void muteMember(@NonNull ChatRoom chatRoom,
                           @NonNull User member,
                           @Nullable String reason,
                           @NonNull HMR.Completion completion) {

        doMuteOrUnmute(chatRoom, member, reason, true, completion);
    }

    @Override
    public void unmuteMember(@NonNull ChatRoom chatRoom,
                             @NonNull User member,
                             @Nullable String reason,
                             @NonNull HMR.Completion completion) {
        doMuteOrUnmute(chatRoom, member, reason, false, completion);
    }

    private void doMuteOrUnmute(ChatRoom chatRoom,
                                User member,
                                String reason,
                                boolean disable,
                                HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion, "ChatRoomService::doMuteOrUnmute");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion, new Error(
                    Error.Code.BadUser, "User not login")
            );
            return;
        }
        Log.i(TAG, Trace.once((disable ? "mute" : "unmute") + " id:%s member:%s",
                chatRoom.getId(),
                member.getId()));

        HMR.getService(Channel.class).run(new RPCDisableUserText(
                HMR.getMe().getId(),
                member.getId(),
                (int) chatRoom.getId(),
                MUTED_TIME_SEC,
                disable,
                reason == null ? "" : reason,
                rCompletion));
    }

    @Override
    public void fetchMutedUsers(@NonNull ChatRoom chatRoom, @NonNull HMR.CompletionArg<Set<User>> completion) {
        RichCompletionArg<Set<User>> rCompletion =
                new RichCompletionArg<>(completion, "ChatRoomService::fetchMutedUsers");
        if (HMR.getState() != HMR.State.Opened) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "User not login"));
            return;
        }
        Log.i(TAG, Trace.once("fetchMutedMembers roomId:%s", chatRoom.getId()));

        HMR.getService(Channel.class)
                .run(new RPCFetchMutedMemberList((int) chatRoom.getId(), rCompletion));
    }

    @Override
    public void isMuted(@NonNull ChatRoom chatRoom, @NonNull final User member,
                        @NonNull final HMR.CompletionArg<Boolean> completion) {
        final RichCompletionArg<Boolean> rCompletion = new RichCompletionArg<>(completion, "ChatRoomService::isMuted");
        fetchMutedUsers(chatRoom, new HMR.CompletionArg<Set<User>>() {
            @Override
            public void onSuccess(Set<User> mutedMembers) {
                if (mutedMembers.size() > 0 && mutedMembers.contains(member)) {
                    CompletionUtils.dispatchSuccess(rCompletion, true);
                } else {
                    CompletionUtils.dispatchSuccess(rCompletion, false);
                }
            }

            @Override
            public void onFailed(Error error) {
                CompletionUtils.dispatchFailure(rCompletion, error);
            }
        });
    }

    @Override
    public void addListener(@NonNull final ChatRoomListener listener) {
        listeners.add(listener);
        Log.d(TAG, Trace.once()
                .method("addListener")
                .info("name", listener.getClass().getSimpleName())
                .info("size", listeners.size()));
    }

    @Override
    public void removeListener(@NonNull final ChatRoomListener listener) {
        listeners.remove(listener);
        Log.d(TAG, Trace.once()
                .method("removeListener")
                .info("name", listener.getClass().getSimpleName())
                .info("size", listeners.size()));
    }

    @Override
    public void addMemberListener(@NonNull final MemberListener listener) {
        memberListeners.add(listener);
        Log.d(TAG, Trace.once()
                .method("addMemberListener")
                .info("name", listener.getClass().getSimpleName())
                .info("size", memberListeners.size()));
    }

    @Override
    public void removeMemberListener(
            @NonNull MemberListener listener) {
        memberListeners.remove(listener);
        Log.d(TAG, Trace.once()
                .method("removeMemberListener")
                .info("name", listener.getClass().getSimpleName())
                .info("size", memberListeners.size()));
    }

    @Override
    public void onNotify(String service, String function, byte[] data) {
        if ("chatroom_online_count_d".equals(service) && "UserCountBc".equals(function)) {
            handleUserCountChanged(data);
        } else if ("chatroom_auther".equals(service) && "UpdateChatRoomInfoBc".equals(function)) {
            handleBasicInfoChanged(data);
        } else if ("chatroom_auther".equals(service) && "UpdateRolerBc".equals(function)) {
            handleMemberRoleChanged(data);
        } else if ("chatroom_auther".equals(service) && "KickOffUserBc".equals(function)) {
            handleMemberKicked(data);
        } else if ("chatroom_auther".equals(service) && "DismissChatRoomBc".equals(function)) {
            handleChatRoomDeleted(data);
        } else if ("chatroom_online_repair_d".equals(service) && "UserOnlineChangeBc".equals(function)) {
            handleMembersChanged(data);
        } else if ("chatroom_auther".equals(service) && "DisableUserTextBc".equals(function)) {
            handleChatMutation(data);
        }
    }

    private void handleChatMutation(byte[] data) {
        Receiver receiver = new Receiver(data);
        receiver.parseHeader();
        if (receiver.getUri().intValue() != ChatRoomProto.PCS_DisableUserTextBc.URI) {
            return;
        }

        final ChatRoomProto.PCS_DisableUserTextBc bc = new ChatRoomProto.PCS_DisableUserTextBc();
        receiver.unmarshallWrap(bc);

        final ChatRoom chatRoom = new ChatRoom(bc.roomid.longValue());
        final User operator = new User(bc.opUid.longValue());
        final Set<User> users = new HashSet<>();
        for (Uint64 uid : bc.tuids) {
            users.add(new User(uid.longValue()));
        }

        DispatchQueue.main.async("ChatRoomService::handleChatMutation", new Runnable() {
            @Override
            public void run() {
                for (MemberListener listener : memberListeners) {
                    if (bc.disable) {
                        listener.onMemberMuted(chatRoom, operator, users, bc.reason);
                    } else {
                        listener.onMemberUnmuted(chatRoom, operator, users, bc.reason);
                    }
                }
            }
        });
    }

    private void handleMembersChanged(byte[] data) {
        Receiver receiver = new Receiver(data);
        receiver.parseHeader();
        if (receiver.getUri().intValue() != ChatRoomProto.PCS_ChatRoomUserBc.uri) {
            return;
        }

        ChatRoomProto.PCS_ChatRoomUserBc bc = new ChatRoomProto.PCS_ChatRoomUserBc();
        receiver.unmarshallWrap(bc);

        final ChatRoom chatRoom = new ChatRoom(bc.roomId.longValue());
        final List<User> joinUsers = new ArrayList<>();
        final List<User> leaveUsers = new ArrayList<>();

        if (bc.joinUsers.size() > 0) {
            for (Uint64 uid : bc.joinUsers) {
                joinUsers.add(new User(uid.longValue()));
            }
        }

        if (bc.leaveUsers.size() > 0) {
            for (Uint64 uid : bc.leaveUsers) {
                leaveUsers.add(new User(uid.longValue()));
            }
        }

        DispatchQueue.main.async("ChatRoomService::handleMembersChanged", new Runnable() {
            @Override
            public void run() {
                for (MemberListener listener : memberListeners) {
                    if (joinUsers.size() > 0) {
                        Log.i(TAG, Trace
                                .once().method("onReceiveMessage").msg("member join, size = " + joinUsers.size()));
                        listener.onMemberJoined(chatRoom, joinUsers);
                    }

                    if (leaveUsers.size() > 0) {
                        Log.i(TAG,
                                Trace.once().method("onReceiveMessage")
                                        .msg("member leave, size = " + leaveUsers.size()));
                        listener.onMemberLeaved(chatRoom, leaveUsers);
                    }
                }
            }
        });
    }

    private void handleChatRoomDeleted(byte[] data) {
        Receiver receiver = new Receiver(data);
        receiver.parseHeader();
        if (receiver.getUri().intValue() != ChatRoomProto.PCS_DismissChatRoomBc.uri) {
            return;
        }

        final ChatRoomProto.PCS_DismissChatRoomBc bc = new ChatRoomProto.PCS_DismissChatRoomBc();
        receiver.unmarshallWrap(bc);

        DispatchQueue.main.async("ChatRoomService::handleChatRoomDeleted", new Runnable() {
            @Override
            public void run() {
                for (ChatRoomListener listener : listeners) {
                    listener.onChatRoomDismissed(new ChatRoom(bc.roomid.longValue()), new User(bc.uid.longValue()));
                }
            }
        });
    }

    private void handleMemberKicked(byte[] data) {
        Receiver receiver = new Receiver(data);
        receiver.parseHeader();
        if (receiver.getUri().intValue() != ChatRoomProto.PCS_KickOffUserBc.uri) {
            return;
        }

        final ChatRoomProto.PCS_KickOffUserBc bc = new ChatRoomProto.PCS_KickOffUserBc();
        receiver.unmarshallWrap(bc);
        final ChatRoom chatRoom = new ChatRoom(bc.topsid.longValue());
        final User admin = new User(bc.admin.longValue());
        final ArrayList<User> tuids = new ArrayList<>();
        for (Uint64 uid : bc.tuids) {
            tuids.add(new User(uid.longValue()));
        }

        final KickOff kickOff = new KickOff(bc.mReason, KickOffEnum.getByType(bc.type.intValue()));

        if (tuids.contains(HMR.getMe())) {
            _unsubscribeChatRoomBC(chatRoom);
        }

        DispatchQueue.main.async("ChatRoomService::handleMemberKicked", new Runnable() {
            @Override
            public void run() {
                for (MemberListener listener : memberListeners) {
                    listener.onMemberKicked(chatRoom, admin, tuids, kickOff);
                }
            }
        });
    }

    private void handleMemberRoleChanged(byte[] data) {
        Receiver receiver = new Receiver(data);
        receiver.parseHeader();
        if (receiver.getUri().intValue() != ChatRoomProto.PCS_UpdateChatRoomRolerBc.uri) {
            return;
        }

        final ChatRoomProto.PCS_UpdateChatRoomRolerBc bc = new ChatRoomProto.PCS_UpdateChatRoomRolerBc();
        receiver.unmarshallWrap(bc);

        if (ChatRoomProto.ChatRoomOpEnum.values()[bc.op.intValue()] !=
                ChatRoomProto.ChatRoomOpEnum.ADD &&
                ChatRoomProto.ChatRoomOpEnum.values()[bc.op.intValue()] !=
                        ChatRoomProto.ChatRoomOpEnum.REMOVE) {
            return;
        }

        final ChatRoom chatRoom = new ChatRoom(bc.roomid.longValue());
        final User user = new User(bc.uid.longValue());
        final User admin = new User(bc.admin.longValue());

        final boolean isAdd = ChatRoomProto.ChatRoomOpEnum.values()[bc.op.intValue()] ==
                ChatRoomProto.ChatRoomOpEnum.ADD;

        DispatchQueue.main.async("ChatRoomService::handleMemberRoleChanged", new Runnable() {
            @Override
            public void run() {
                for (MemberListener listener : memberListeners) {
                    if (isAdd) {
                        listener.onRoleAdded(chatRoom, bc.roler, admin, user);
                    } else {
                        listener.onRoleRemoved(chatRoom, bc.roler, admin, user);
                    }
                }
            }
        });
    }

    private void handleBasicInfoChanged(byte[] data) {
        Receiver receiver = new Receiver(data);
        receiver.parseHeader();
        if (receiver.getUri().intValue() != ChatRoomProto.PCS_UpdateChatRoomInfoBc.uri) {
            return;
        }

        ChatRoomProto.PCS_UpdateChatRoomInfoBc bc = new ChatRoomProto.PCS_UpdateChatRoomInfoBc();
        receiver.unmarshallWrap(bc);
        final ChatRoom chatRoom = new ChatRoom(bc.roomid.intValue());
        final Map<ChatRoomInfo.BasicInfoType, String> basicInfo = new HashMap<>();
        for (Map.Entry<String, String> entry : bc.props.entrySet()) {
            String key = entry.getKey();
            if ("Extention".equals(key)) {
                key = ChatRoomInfo.BasicInfoType.AppExtra.name();
            }
            basicInfo.put(ChatRoomInfo.BasicInfoType.valueOf(key), entry.getValue());
        }

        DispatchQueue.main.async("ChatRoomService::handleBasicInfoChanged", new Runnable() {
            @Override
            public void run() {
                for (ChatRoomListener listener : listeners) {
                    listener.onBasicInfoChanged(chatRoom, basicInfo);
                }
            }
        });
    }

    private void handleUserCountChanged(byte[] data) {
        Receiver receiver = new Receiver(data);
        receiver.parseHeader();
        if (receiver.getUri().intValue() != ChatRoomProto.PCS_PushChatRoomUserCountBC.uri) {
            return;
        }

        ChatRoomProto.PCS_PushChatRoomUserCountBC bc = new ChatRoomProto.PCS_PushChatRoomUserCountBC();
        receiver.unmarshallWrap(bc);

        final ChatRoom chatRoom = new ChatRoom(bc.topsid.longValue());
        final int count = bc.totalCount.intValue();

        DispatchQueue.main.async("ChatRoomService::handleUserCountChanged", new Runnable() {
            @Override
            public void run() {
                for (MemberListener listener : memberListeners) {
                    listener.onMemberCountChanged(chatRoom, count);
                }
            }
        });
    }

    private void addChatRoom(ChatRoom chatRoom) {
        chatRooms.add(chatRoom);
    }

    private void removeChatRoom(ChatRoom chatRoom) {
        chatRooms.remove(chatRoom);
    }

    private void leaveChatRoom(final ChatRoom chatRoom, final RichCompletion completion) {
        if (HMR.getMe().isAnonymous()) {
            _unsubscribeChatRoomBC(chatRoom);
            CompletionUtils.dispatchSuccess(completion);
        } else {
            ServiceProvider.get(Channel.class).run(new RPCLeaveChatRoom(
                    (int) chatRoom.getId(), new RichCompletion("ChatRoomService::leaveChatRoom")
                    .onSuccess(new OnSuccess() {
                        @Override
                        public void onSuccess() {
                            ChatRoomServiceImpl.this._unsubscribeChatRoomBC(chatRoom);
                            CompletionUtils.dispatchSuccess(completion);
                        }
                    }).onFailure(new OnFailure() {
                        @Override
                        public void onFailure(Error err) {
                            CompletionUtils.dispatchFailure(completion, err);
                        }
                    })));
        }
    }

    private void _subscribeChatRoomBC(ChatRoom chatRoom, @NonNull final RichCompletion completion) {
        final long roomId = chatRoom.getId();
        UserGroupType groupType = new UserGroupType(textChatBCGrpType, roomId);
        UserGroupType groupType2 = new UserGroupType(authBCGrpType, roomId);
        UserGroupType groupType3 = new UserGroupType(onlineBCGrpType, roomId);
        ArrayList<UserGroupType> lists = new ArrayList<>();
        lists.add(groupType);
        lists.add(groupType2);
        lists.add(groupType3);
        YYServiceCore.getInstance().subscribeBroadcast(lists,
                new IRPCChannel.RPCCallback<BroadSubOrUnSubTask.ResponseParam>() {
                    @Override
                    public void onSuccess(int var1, BroadSubOrUnSubTask.ResponseParam response) {
                        Log.i(TAG, Trace.once().method("subscribeBroadcast success")
                                .msg("resCode:%d, roomId:%d", response.mResCode, roomId));
                        CompletionUtils.dispatchSuccess(completion);
                    }

                    @Override
                    public void onFail(int requestId, int sdkResCode, int srvResCode, Exception e) {
                        Log.e(TAG, Trace.once().method("subscribeBroadcast failed")
                                .msg("sdkResCode:%d, srvResCode:%d, roomId:%d", sdkResCode,
                                        srvResCode,
                                        roomId));
                        CompletionUtils
                                .dispatchFailure(completion, new Error(srvResCode, "service err"));
                    }
                });
    }

    private void _unsubscribeChatRoomBC(ChatRoom chatRoom) {
        final long roomId = chatRoom.getId();
        UserGroupType groupType = new UserGroupType(textChatBCGrpType, roomId);
        UserGroupType groupType2 = new UserGroupType(authBCGrpType, roomId);
        UserGroupType groupType3 = new UserGroupType(onlineBCGrpType, roomId);
        ArrayList<UserGroupType> lists = new ArrayList<>();
        lists.add(groupType);
        lists.add(groupType2);
        lists.add(groupType3);
        YYServiceCore.getInstance().unSubscribeBroadcast(lists,
                new IRPCChannel.RPCCallback<BroadSubOrUnSubTask.ResponseParam>() {
                    @Override
                    public void onSuccess(int var1, BroadSubOrUnSubTask.ResponseParam response) {
                        Log.i(TAG, Trace.once().method("unSubscribeBroadcast success")
                                .msg("resCode:%d, roomId:%d",
                                        response.mResCode, roomId));
                    }

                    @Override
                    public void onFail(int requestId, int sdkResCode, int srvResCode, Exception e) {
                        Log.i(TAG, Trace.once().method("unSubscribeBroadcast failed")
                                .msg("sdkResCode:%d, srvResCode:%d, roomId:%d",
                                        sdkResCode, srvResCode, roomId));
                    }
                });
    }

    @Override
    public void onChannelConnected() {
        Log.i(TAG, Trace.once().method("onChannelConnected"));
        rejoinChatRoomIfneed();
        resetInitialTimeInterval();
    }

    @Override
    public void onChannelDisconnected() {
        Log.i(TAG, Trace.once().method("onChannelDisconnected"));
        recordInitialTimeIntervalIfNeeded();
    }

    private void rejoinChatRoomIfneed() {
        // YYServiceSDK 的在线是通过 YYServiceSDK 的心跳来进行维持的
        // 退到后台后或者断网的时候, Service 那边的心跳停止发送
        // 频道的后台会读取这个在线状态，当超过75秒（跟后台确认的时间）没有发送心跳，则会认为这个用户离线
        // 频道台会踢这个用户出频道，所以当失效时间段小于60秒的时候，则无需处理
        long now = System.currentTimeMillis();

        if (mInitialTimeInterval <= kReferenceValue || (now - mInitialTimeInterval) < kRejoinConditionGap) {
            return;
        }

        if (chatRooms.size() > 0) {
            for (ChatRoom chatRoom : chatRooms) {
                Log.i(TAG, String.format(Locale.US,
                        "rejoin chatRoom after channel connected, roomId:%d",
                        chatRoom.getId()));

                // 仅在实名登录时，才需要重新进频道
                if (HMR.getMe() != null) {
                    ServiceProvider.get(Channel.class).run(new RPCJoinChatRoom(
                            (int) chatRoom.getId(),
                            new HashMap<String, String>(), null
                    ));
                }
            }
        }
    }

    private void recordInitialTimeIntervalIfNeeded() {
        if (mInitialTimeInterval < kReferenceValue) {
            mInitialTimeInterval = System.currentTimeMillis();
        }
    }

    private void resetInitialTimeInterval() {
        mInitialTimeInterval = 0L;
    }

    interface NotifyHandler {
        /**
         * 通知处理器
         *
         * @param data 数据
         */
        void handle(byte[] data);
    }

    private final Set<MemberListener> memberListeners = new HashSet<>();
    private final Set<ChatRoomListener> listeners = new HashSet<>();
    private final Set<ChatRoom> chatRooms = new HashSet<>();
}
