package com.hummer.im.chatroom;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im.HMR;
import com.hummer.im.model.chat.Content;
import com.hummer.im.model.id.ChatRoom;
import com.hummer.im.model.id.User;
import com.hummer.im.model.kick.KickOff;

import java.util.List;
import java.util.Map;
import java.util.Set;

public interface ChatRoomService {

    /**
     * chatroom 切换区域
     * @param region 国家区域
     */
    void switchRegion(String region);

    /**
     * 创建聊天室
     *
     * @param chatRoomInfo 聊天室信息
     * @param completion 回调
     */
    void createChatRoom(@NonNull ChatRoomInfo chatRoomInfo,
                        @NonNull HMR.CompletionArg<ChatRoom> completion);

    /**
     * 解散聊天室
     *
     * @param chatRoom   聊天室标识符
     * @param completion 回调
     */
    void dismissChatRoom(@NonNull ChatRoom chatRoom,
                         @Nullable HMR.Completion completion);

    /**
     * 加入聊天室
     *
     * @param chatRoom   聊天室标识符
     * @param joinProps  预留属性，传入约定数据。没有则直接new HashMap()。该属性对匿名登录无效
     * @param completion 回调
     */
    void join(@NonNull ChatRoom chatRoom,
              @NonNull Map<String, String> joinProps,
              @NonNull Challenges.JoiningCompletion completion);

    /**
     * 离开聊天室
     *
     * @param chatRoom   聊天室标识符
     * @param completion 回调
     */
    void leave(@NonNull ChatRoom chatRoom, @Nullable HMR.Completion completion);

    enum EKickInfo {
        Time,
        Reason,
    }

    /**
     * 踢用户出聊天室
     *
     * @param chatRoom   聊天室标识符
     * @param member     踢出的聊天室成员
     * @param extraInfo  踢人额外信息
     * @param completion 回调
     */
    void kick(@NonNull ChatRoom chatRoom,
              @NonNull User member,
              @Nullable Map<EKickInfo, String> extraInfo,
              @NonNull HMR.Completion completion);


    final class Roles {
        public static final String Admin = "admin";
        public static final String Owner = "owner";
    }

    /**
     * 添加聊天室权限角色
     *
     * @param chatRoom   聊天室标识符
     * @param member     成员
     * @param role       角色类型,具体值可参考{@link Roles},业务也可以自己定义扩展角色
     * @param completion 回调
     */
    void addRole(@NonNull ChatRoom chatRoom,
                 @NonNull User member,
                 @NonNull String role,
                 @NonNull HMR.Completion completion);

    /**
     * 移除聊天室权限角色
     *
     * @param chatRoom   聊天室标识符
     * @param member     成员
     * @param role       角色类型,具体值可参考{@link Roles},业务也可以自己定义扩展角色
     * @param completion 回调
     */
    void removeRole(@NonNull ChatRoom chatRoom,
                    @NonNull User member,
                    @NonNull String role,
                    @NonNull HMR.Completion completion);

    /**
     * 获取聊天室成员列表
     *
     * @param chatRoom   聊天室标识符
     * @param num        拉取的条数
     * @param offset     拉取的位置，第一页从0开始
     * @param completion 回调,List<Fellow> 聊天室成员列表
     */
    void fetchMembers(@NonNull ChatRoom chatRoom, int num, int offset,
                      @NonNull HMR.CompletionArg<List<User>> completion);

    /**
     * 获取聊天室的所有角色列表
     *
     * @param chatRoom   聊天室标识符
     * @param online     true获取在线的角色列表，false获取全部角色列表
     * @param completion 回调, Map<String, List<Fellow> 角色成员列表,key 对应的取值是DefaultRoleType类型，或者是业务自定义角色类型
     */
    void fetchRoleMembers(@NonNull ChatRoom chatRoom, boolean online,
                          @NonNull HMR.CompletionArg<Map<String, List<User>>> completion);

    /**
     * 获取聊天室基本属性信息
     *
     * @param chatRoom   聊天室标识符
     * @param completion 回调, RoomInfo 聊天室属性信息
     */
    void fetchBasicInfo(@NonNull ChatRoom chatRoom,
                        @NonNull HMR.CompletionArg<ChatRoomInfo> completion);


    /**
     * 修改聊天室基本属性信息
     *
     * @param chatRoom   聊天室标识符
     * @param propInfo   修改属性值
     * @param completion 回调
     */
    void changeBasicInfo(@NonNull ChatRoom chatRoom,
                         @NonNull Map<ChatRoomInfo.BasicInfoType, String> propInfo,
                         @NonNull HMR.Completion completion);

    /**
     * 聊天室公屏禁言
     *
     * <pre>
     * // 禁言 mChatRoom 下的 member 成员，无原因
     * HMR.getService(ChatRoomService.class).muteMember(mChatRoom, fellow, null,
     *          new RichCompletion()
     *                  .onSuccess(() -> {
     *                      // doSomething
     *                  })
     *                  .onFailure(err -> {
     *                      // doSomething
     *                  }));
     * </pre>
     *
     * @param chatRoom   聊天室
     * @param member     被禁言的成员, 成员必须在聊天室内
     * @param reason     禁言原因
     * @param completion 回调
     */
    void muteMember(@NonNull ChatRoom chatRoom,
                    @NonNull User member,
                    @Nullable String reason,
                    @NonNull HMR.Completion completion);

    /**
     * 聊天室解除禁言
     *
     * <pre>
     * HMR.getService(ChatRoomService.class).unmuteMember(mChatRoom, fellow, null,
     *          new RichCompletion()
     *                  .onSuccess(() -> {
     *                      // doSomething
     *                  })
     *                  .onFailure(err -> {
     *                      // doSomething
     *                  }));
     * </pre>
     *
     * @param chatRoom   聊天室标识
     * @param member     被解禁成员，必须在聊天室内
     * @param reason     解禁原因
     * @param completion 回调
     */
    void unmuteMember(@NonNull ChatRoom chatRoom,
                      @NonNull User member,
                      @Nullable String reason,
                      @NonNull HMR.Completion completion);

    /**
     * 获取禁言用户列表：所有在chatroom中被禁言过的用户
     *
     * <pre>
     * HMR.getService(ChatRoomService.class).fetchMutedUsers(mChatRoom, new RichCompletionArg<Set<User>>()
     *                 .onSuccess((users) -> {
     *                     // doSomething
     *                 })
     *                 .onFailure(error -> {
     *                     // doSomething
     *                 }));
     * </pre>
     *
     * @param chatRoom   聊天室
     * @param completion 回调
     */
    void fetchMutedUsers(@NonNull ChatRoom chatRoom, @NonNull HMR.CompletionArg<Set<User>> completion);

    /**
     * 用户是否被禁言：true 表示是; false 表示否
     *
     * <pre>
     * HMR.getService(ChatRoomService.class).isMuted(mChatRoom, user,
     *             new RichCompletionArg<Boolean>()
     *                     .onSuccess(isMuted -> {
     *                          doSomething
     *                     })
     *                     .onFailure(error -> {
     *                         doSomething
     *                     }));
     * </pre>
     *
     * @param chatRoom   聊天室
     * @param member     被查询是否禁言的用户
     * @param completion 回调
     */
    void isMuted(@NonNull ChatRoom chatRoom,
                 @NonNull User member,
                 @NonNull HMR.CompletionArg<Boolean> completion);

    /**
     * 添加聊天室回调监听器
     *
     * @param listener 监听器对象
     */
    void addListener(@NonNull final ChatRoomListener listener);

    /**
     * 移除聊天室回调监听器
     *
     * @param listener 监听器对象
     */
    void removeListener(@NonNull final ChatRoomListener listener);

    /**
     * 添加聊天室成员相关回调监听器
     *
     * @param listener 监听器对象
     */
    void addMemberListener(@NonNull final MemberListener listener);

    /**
     * 移除聊天室成员相关回调监听器
     *
     * @param listener 监听器对象
     */
    void removeMemberListener(@NonNull final MemberListener listener);

    interface ChatRoomListener {

        /**
         * 聊天室基本信息变化的回调通知
         */
        void onBasicInfoChanged(@NonNull ChatRoom chatRoom,
                                @NonNull Map<ChatRoomInfo.BasicInfoType, String> propInfo);

        /**
         * 聊天室被解散的通知
         *
         * @param member   操作者
         * @param chatRoom 聊天室
         */
        void onChatRoomDismissed(@NonNull ChatRoom chatRoom, @NonNull User member);
    }

    interface MemberListener {
        /**
         * 用户加入聊天室的回调通知
         */
        void onMemberJoined(@NonNull ChatRoom chatRoom, @NonNull List<User> members);

        /**
         * 用户离开聊天室的回调通知
         */
        void onMemberLeaved(@NonNull ChatRoom chatRoom, @NonNull List<User> members);

        /**
         * 聊天室成员数变化的回调通知
         */
        void onMemberCountChanged(@NonNull ChatRoom chatRoom, int count);

        /**
         * 添加聊天室权限角色的回调通知
         */
        void onRoleAdded(@NonNull ChatRoom chatRoom,
                         @NonNull String role,
                         @NonNull User admin,
                         @NonNull User fellow);

        /**
         * 移除聊天室权限角色的回调通知
         */
        void onRoleRemoved(@NonNull ChatRoom chatRoom,
                           @NonNull String role,
                           @NonNull User admin,
                           @NonNull User fellow);

        /**
         * 用户被踢出聊天室的回调通知
         *
         * @param chatRoom 聊天室
         * @param admin    管理员
         * @param member   被踢用户
         * @param kickOff  踢人类型和原因等
         */
        void onMemberKicked(@NonNull ChatRoom chatRoom,
                            @NonNull User admin,
                            @NonNull List<User> member,
                            @NonNull KickOff kickOff);

        /**
         * 用户被禁言通知
         *
         * @param chatRoom 聊天室
         * @param operator 禁言者
         * @param members  被禁言者们
         * @param reason   原因
         */
        void onMemberMuted(@NonNull ChatRoom chatRoom,
                           @NonNull User operator,
                           @NonNull Set<User> members,
                           @Nullable String reason);

        /**
         * 用户解禁通知
         *
         * @param chatRoom 聊天室
         * @param operator 解禁操作人
         * @param members  被解禁者们
         * @param reason   原因
         */
        void onMemberUnmuted(@NonNull ChatRoom chatRoom,
                             @NonNull User operator,
                             @NonNull Set<User> members,
                             @Nullable String reason);
    }


    /**
     * 聊天室信令消息
     */
    final class Signal extends Content {
        public final User   user;
        public final String content;

        public static Signal unicast(@NonNull User user, @NonNull String content) {
            return new Signal(user, content);
        }

        public static Signal broadcast(@NonNull String content) {
            return new Signal(null, content);
        }

        private Signal(User user, @NonNull String content) {
            this.user    = user;
            this.content = content;
        }

        @Override
        public String toString() {
            return "ChatRoom.Signal{" + content + "}";
        }
    }

}




