package com.hummer.im.chatroom._internals.rpc;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im.chatroom.Challenges;
import com.hummer.im.chatroom._internals.ChallengeImpl;
import com.hummer.im.chatroom._internals.packet.Marshallable;
import com.hummer.im.chatroom._internals.packet.Uint32;
import com.hummer.im.chatroom._internals.packet.Uint64;
import com.hummer.im.chatroom._internals.proto.ChatRoomProto;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.completion.CompletionUtils;

import java.util.Map;

public final class RPCJoinChatRoom extends
        ChatRoomRPC<ChatRoomProto.PCS_CommonOperatorAuth2Res> {
    public RPCJoinChatRoom(int roomId, Map<String, String> joinProps, Challenges.JoiningCompletion completion) {
        this.roomId         = roomId;
        this.joinProps      = joinProps;
        this.authCompletion = completion;
        this.bTwice         = false;
        this.password       = "";
        this.completion     = null;
    }

    public RPCJoinChatRoom(int roomId, String password, Map<String, String> joinProps, RichCompletion completion) {
        this.roomId     = roomId;
        this.joinProps  = joinProps;
        this.password   = password;
        this.completion = completion;
        this.bTwice     = true;
    }

    @Override
    public String getFunctionName() {
        return "JoinChatRoom";
    }

    @Override
    public String serviceName() {
        return "chatroom_auther";
    }

    @Override
    public Marshallable requestObj() {
        ChatRoomProto.PCS_JoinChatRoomReq req = new ChatRoomProto.PCS_JoinChatRoomReq();
        req.appkey = Uint32.toUInt(HMRContext.appId);
        req.roomid = Uint32.toUInt(roomId);
        req.uid = Uint64.toUInt(HMR.getMe().getId());
        req.joinProps = joinProps;

        // TODO: What if password is empty?

        return req;
    }

    @Override
    public void handleSuccess(@NonNull ChatRoomProto.PCS_CommonOperatorAuth2Res res) {
        HMRContext.work.async("RPCJoinChatRoom::handleSuccess", new Runnable() {
            @Override
            public void run() {
                if (authCompletion != null) {
                    authCompletion.onSucceed();
                }
            }
        });
    }

    @Override
    public void handleError(@Nullable final ChatRoomProto.PCS_CommonOperatorAuth2Res res,
                            @NonNull final Error error) {
        if (res == null) {
            Log.e("RPCJoinChatRoom", Trace.once().method("handleError")
                    .info("res", null));
        }

        if (res.res.intValue() == Error.Code.Success) {
            return;
        }

        if (res.res.intValue() == Error.Code.kChallengeException) {
            if (!bTwice) {
                final ChallengeImpl.PasswordImpl passwordImpl =
                        new ChallengeImpl.PasswordImpl(res.roomId.longValue());

                HMRContext.work.async("RPCJoinChatRoom::handleChallengeException", new Runnable() {
                    @Override
                    public void run() {
                        if (authCompletion != null) {
                            authCompletion.onReceiveChallenge(passwordImpl);
                        }
                    }
                });
            } else {
                CompletionUtils.dispatchFailure(completion, error);
            }
        } else {
            if (!bTwice) {
                HMRContext.work.async("RPCJoinChatRoom::handleauthCompletion", new Runnable() {
                    @Override
                    public void run() {
                        if (authCompletion != null) {
                            authCompletion.onFailure(error);
                        }
                    }
                });
            } else {
                CompletionUtils.dispatchFailure(completion, error);
            }
        }
    }

    @Override
    public void handleError(@NonNull final Error err) {
        if (!bTwice) {
            HMRContext.work.async("RPCJoinChatRoom::handleError", new Runnable() {
                @Override
                public void run() {
                    if (authCompletion != null) {
                        authCompletion.onFailure(err);
                    }
                }
            });
        } else {
            CompletionUtils.dispatchFailure(completion, err);
        }
    }

    private Challenges.JoiningCompletion authCompletion = null;
    private Map<String, String> joinProps;
    private boolean bTwice;
    private String password;
    private RichCompletion completion;
    private int roomId;
}
