package com.hummer.im.chatroom._internals.rpc;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im.Error;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.shared.HiidoReporter;
import com.hummer.im._internals.shared.StringChain;
import com.hummer.im.chatroom._internals.packet.Marshallable;
import com.hummer.im.chatroom._internals.packet.Receiver;
import com.hummer.im.chatroom._internals.packet.Sender;
import com.hummer.im.chatroom._internals.packet.Uint32;
import com.hummer.im.service.Channel;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;

public abstract class ChatRoomRPC<Response extends Marshallable> implements Channel.RPC {

    public abstract Marshallable requestObj();

    public abstract void handleSuccess(@NonNull Response res) throws Throwable;

    @Override
    public String protoType() {
        return null;
    }

    @Override
    public byte[] getRequestBytes() {
        Sender sender = new Sender(requestObj());
        sender.endPack();
        startTs = System.currentTimeMillis();
        Log.i("ChatRoomRPC", flow.trace()
                .method(getFunctionName())
                .msg("-> {" + requestObj().toString() + "}"));
        return sender.getBytes();
    }

    @Override
    @SuppressWarnings("unchecked")
    public void handleResponse(@NonNull byte[] responseBytes) throws Throwable {
        Class responseClass = (Class) ((ParameterizedType) this.getClass()
                .getGenericSuperclass())
                .getActualTypeArguments()[0];

        Response res = (Response) responseClass.newInstance();
        Receiver receiver = new Receiver(responseBytes);
        receiver.unmarshallWrap2(res);

        Field resCodeField;
        try {
            resCodeField = res.getClass().getDeclaredField("res");
        } catch (Exception e) {
            resCodeField = res.getClass().getDeclaredField("rescode");
        }

        if (resCodeField == null) {
            Log.e(TAG, Trace.once().method("handleResponse").msg("resCodeField is <null>"));
        }

        int resCode = ((Uint32) resCodeField.get(res)).intValue();

        Log.i(TAG, Trace.once().method("ChatRoomRPC.handleResponse")
                .info("code", resCode)
                .info("resCls", res.getClass().getSimpleName()));

        long rtt = System.currentTimeMillis() - startTs;
        if (resCode == Error.Code.Success) {
            handleSuccess(res);
            Log.i("ChatRoomRPC", flow.trace()
                    .method(getFunctionName())
                    .msg("<- Succeed {" + res.toString() + "}"));
        } else {
            Field msgField = res.getClass().getDeclaredField("msg");
            String msg = "service error";

            if (msgField != null) {
                msg = msgField.get(res) == null ? msg : (String) msgField.get(res);
            }

            Error error = new Error(resCode, msg);
            handleError(res, error);
            Log.e("ChatRoomRPC", error, flow.trace().method(getFunctionName())
                    .msg("<- Failed {" + new StringChain()
                            .add("logId",    flow.logId)
                            .add("response", res.toString())
                            .add("msg", msg)
                            .toString() + "}"));
        }

        HiidoReporter.reportReturnCodeTemporary(getFunctionName(), rtt, resCode);
    }

    public void handleError(@Nullable Response res, @NonNull Error error) {
        handleError(error);
    }

    private long startTs;
    private Trace.Flow flow = new Trace.Flow();
    private static final String TAG = "ChatRoomRPC";
}
