package com.hummer.im.chatroom._internals.rpc;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im.chatroom.Challenges;
import com.hummer.im.chatroom._internals.ChallengeImpl;
import com.hummer.im.chatroom._internals.packet.Marshallable;
import com.hummer.im.chatroom._internals.packet.Uint32;
import com.hummer.im.chatroom._internals.packet.Uint64;
import com.hummer.im.chatroom._internals.proto.ChatRoomProto;
import com.hummer.im.shared.completion.Completion;
import com.hummer.im.shared.completion.CompletionUtils;

import java.util.Map;

public final class RPCJoinChatRoom extends
        ChatRoomRPC<ChatRoomProto.PCS_CommonOperatorAuth2Res> {
    public RPCJoinChatRoom(int roomId, Map<String, String> joinProps, Challenges.JoiningCompletion completion) {
        this.roomId         = roomId;
        this.joinProps      = joinProps;
        this.authCompletion = completion;
        this.bTwice         = false;
        this.password       = "";
        this.completion     = null;
    }

    public RPCJoinChatRoom(int roomId, String password, Map<String, String> joinProps, Completion completion) {
        this.roomId     = roomId;
        this.joinProps  = joinProps;
        this.password   = password;
        this.completion = completion;
        this.bTwice     = true;
    }

    @Override
    public String getFunctionName() {
        return "JoinChatRoom";
    }

    @Override
    public String serviceName() {
        return "chatroom_auther";
    }

    @Override
    public Marshallable requestObj() {
        ChatRoomProto.PCS_JoinChatRoomReq req = new ChatRoomProto.PCS_JoinChatRoomReq();
        req.appkey = Uint32.toUInt(HMRContext.appId);
        req.roomid = Uint32.toUInt(roomId);
        req.uid = Uint64.toUInt(HMR.getMe().getId());
        req.joinProps = joinProps;

        // TODO: What if password is empty?

        return req;
    }

    @Override
    public void handleSuccess(@NonNull ChatRoomProto.PCS_CommonOperatorAuth2Res res) {
        HMRContext.work.async(() -> {
            if (authCompletion != null) {
                authCompletion.onSucceed();
            }
        });
    }

    @Override
    public void handleError(@Nullable ChatRoomProto.PCS_CommonOperatorAuth2Res res,
                            @NonNull Error error) {
        assert res != null;
        if (res.res.intValue() == Error.Code.Success) {
            return;
        }

        if (res.res.intValue() == Error.Code.kChallengeException) {
            if (!bTwice) {
                final ChallengeImpl.PasswordImpl passwordImpl =
                        new ChallengeImpl.PasswordImpl(res.roomId.longValue());

                HMRContext.work.async(() -> {
                    if (authCompletion != null) {
                        authCompletion.onReceiveChallenge(passwordImpl);
                    }
                });
            } else {
                CompletionUtils.dispatchFailure(completion, new Error(res.res.intValue(), "service err"));
            }
        } else {
            if (!bTwice) {
                HMRContext.work.async(() -> {
                    if (authCompletion != null) {
                        authCompletion.onFailed(new Error(
                                res.res.intValue(),
                                "service err"
                        ));
                    }
                });
            } else {
                CompletionUtils.dispatchFailure(completion, new Error(res.res.intValue(), "service err"));
            }
        }
    }

    @Override
    public void handleError(@NonNull Error err) {
        if (!bTwice) {
            HMRContext.work.async(() -> {
                if (authCompletion != null) {
                    authCompletion.onFailed(err);
                }
            });
        } else {
            CompletionUtils.dispatchFailure(completion, err);
        }
    }

    private Challenges.JoiningCompletion authCompletion = null;
    private Map<String, String> joinProps;
    private boolean bTwice;
    private String password;
    private Completion completion;
    private int roomId;
}
