package com.hummer.im.chatroom._internals;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.mq.MQService;
import com.hummer.im.channel.Channel;
import com.hummer.im.chatroom.Challenges;
import com.hummer.im.chatroom.ChatRoomInfo;
import com.hummer.im.chatroom.ChatRoomService;
import com.hummer.im.chatroom._internals.msgparser.BroadcastMsgParser;
import com.hummer.im.chatroom._internals.msgparser.ChatRoomDismissMsgParser;
import com.hummer.im.chatroom._internals.msgparser.ChatRoomTextChatMsgParser;
import com.hummer.im.chatroom._internals.msgparser.KickOffMembersMsgParser;
import com.hummer.im.chatroom._internals.msgparser.MemberChangeMsgParser;
import com.hummer.im.chatroom._internals.msgparser.MemberCountChangeMsgParser;
import com.hummer.im.chatroom._internals.msgparser.MemberUpdateRoleMsgParser;
import com.hummer.im.chatroom._internals.msgparser.UnicastMsgParser;
import com.hummer.im.chatroom._internals.msgparser.UpdateChatRoomMsgParser;
import com.hummer.im.chatroom._internals.proto.ChatRoomProto;
import com.hummer.im.chatroom._internals.rpc.RPCChangeChatRoomInfo;
import com.hummer.im.chatroom._internals.rpc.RPCChangeChatRoomRole;
import com.hummer.im.chatroom._internals.rpc.RPCChatRoomTextChat;
import com.hummer.im.chatroom._internals.rpc.RPCCreateChatRoom;
import com.hummer.im.chatroom._internals.rpc.RPCDismissChatRoom;
import com.hummer.im.chatroom._internals.rpc.RPCFetchAdminList;
import com.hummer.im.chatroom._internals.rpc.RPCFetchChatRoomInfo;
import com.hummer.im.chatroom._internals.rpc.RPCFetchMemberList;
import com.hummer.im.chatroom._internals.rpc.RPCJoinChatRoom;
import com.hummer.im.chatroom._internals.rpc.RPCKickOffUser;
import com.hummer.im.chatroom._internals.rpc.RPCLeaveChatRoom;
import com.hummer.im.chatroom._internals.rpc.RPCPullAllAdminUser;
import com.hummer.im.chatroom._internals.rpc.RPCSendBroadcast;
import com.hummer.im.chatroom._internals.rpc.RPCSendUnicast;
import com.hummer.im.id.ChatRoom;
import com.hummer.im.id.ScopeAll;
import com.hummer.im.id.ScopeSignal;
import com.hummer.im.id.User;
import com.hummer.im.services.chat.ChatMessage;
import com.hummer.im.services.chat.contents.Text;
import com.hummer.im.services.notification.NotificationMessage;
import com.hummer.im.services.user.UserService;
import com.hummer.im.shared.DispatchQueue;
import com.hummer.im.shared.ServiceProvider;
import com.hummer.im.shared.completion.Completion;
import com.hummer.im.shared.completion.CompletionArg;
import com.hummer.im.shared.completion.CompletionUtils;
import com.yy.platform.baseservice.IRPCChannel;
import com.yy.platform.baseservice.YYServiceCore;
import com.yy.platform.baseservice.task.BroadSubOrUnSubTask;
import com.yy.platform.baseservice.utils.UserGroupType;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

public class ChatRoomServiceImpl implements ChatRoomService, MQService.MQSubscriber {
    private static final int kickTimeSec = 300;
    private static final String TAG = "ChatRoomService";
    // 在线人数以及成员进出事件的 grpType
    private static final long onlineBCGrpType = 2147483650L;
    // 鉴权服务 （踢人，改频道消息等） 的 grpType
    private static final long authBCGrpType = 2147483649L;
    // 公屏消息 的 grpType
    private static final long textChatBCGrpType = 2147483648L;

    @Override
    public Class[] staticDependencies() {
        return new Class[]{Channel.class};
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{UserService.class};
    }

    @Override
    public void initService() {
        MQService mqService = HMR.getService(MQService.class);

        mqService.registerMessageParser(new ChatRoomTextChatMsgParser());
        mqService.registerMessageParser(new MemberCountChangeMsgParser());
        mqService.registerMessageParser(new UpdateChatRoomMsgParser());
        mqService.registerMessageParser(new MemberUpdateRoleMsgParser());
        mqService.registerMessageParser(new KickOffMembersMsgParser());
        mqService.registerMessageParser(new ChatRoomDismissMsgParser());
        mqService.registerMessageParser(new MemberChangeMsgParser());
        mqService.registerMessageParser(new BroadcastMsgParser());
        mqService.registerMessageParser(new UnicastMsgParser());

        ChatMessage.registerCustomMsgAdapt(ChatRoom.class, (t, completion1) -> {
            ChatMessage msg = (ChatMessage) t;
            Text content = (Text) msg.getContent();
            Map<String, String> chatProps = new HashMap<>();
            chatProps.put("uuid", msg.getUuid());
            if (msg.getAppExtra() != null) {
                chatProps.put("extra", msg.getAppExtra());
            }
            return new RPCChatRoomTextChat(
                    msg.getReceiver().getId(),
                    content.getText(),
                    chatProps,
                    null,
                    completion1);
        });

        NotificationMessage.registerCustomMsgAdapt(ScopeSignal.class, (t, completion2) -> {
            NotificationMessage msg = (NotificationMessage) t;
            ScopeSignal receiver = (ScopeSignal) msg.getReceiver();
            String content = msg.getNotifyMessage();
            return new RPCSendUnicast(
                    receiver.getId(),
                    content,
                    receiver.getUser().getId(),
                    completion2);
        });

        NotificationMessage.registerCustomMsgAdapt(ScopeAll.class, (t, completion3) -> {
            NotificationMessage msg = (NotificationMessage) t;
            ScopeAll receiver = (ScopeAll) msg.getReceiver();
            String content = msg.getNotifyMessage();
            return new RPCSendBroadcast(
                    receiver.getId(),
                    content,
                    completion3);
        });
    }

    @Override
    public void openService(@NonNull Completion completion) {
        channelStateListener = new Channel.StateChangedListener() {
            @Override
            public void onChannelConnected() {
                HMRContext.work.async(() -> {
                    Log.i(TAG, "service channel reconnected");
                    _onChannelConnected();
                });
            }

            @Override
            public void onChannelDisconnected() {
                Log.i(TAG, "service channel disconnected");
            }
        };

        HMR.getService(Channel.class).addStateListener(channelStateListener);

        HMR.getService(MQService.class).addSubscriber(this);

        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void closeService() {
        HMR.getService(MQService.class).removeSubscriber(this);
        HMR.getService(Channel.class).removeStateListener(channelStateListener);
    }

    @Override
    public void createChatRoom(@NonNull ChatRoomInfo chatRoomInfo,
                               @NonNull  final CompletionArg<ChatRoom> completion) {
        Log.i(TAG, Trace.once("createChatRoom subject:%s", chatRoomInfo.getName()));
        ServiceProvider.get(Channel.class).run(new RPCCreateChatRoom(chatRoomInfo, completion));
    }

    @Override
    public void deleteChatRoom(@NonNull ChatRoom chatRoom,
                               @Nullable CompletionArg<ChatRoom> completion) {
        Log.i(TAG, Trace.once("deleteChatRoom chatRoom: %s", chatRoom.getId()));
        ServiceProvider.get(Channel.class).run(new RPCDismissChatRoom(
                Long.valueOf(chatRoom.getId()).intValue(),
                completion
        ));
    }

    @Override
    public void join(@NonNull final ChatRoom chatRoom,
                     @NonNull final Challenges.JoiningCompletion completion) {
        Log.i(TAG, Trace.once("join chatRoom: %s", chatRoom.getId()));
        HMRContext.work.async(() -> ServiceProvider.get(Channel.class).run(new RPCJoinChatRoom(
                (int) chatRoom.getId(),
                new HashMap<>(),
                new Challenges.JoiningCompletion() {
                    @Override
                    public void onReceiveChallenge(final Challenges.Password challenge) {
                        DispatchQueue.main
                                .async(() -> completion.onReceiveChallenge(challenge));
                    }

                    @Override
                    public void onReceiveChallenge(final Challenges.AppChallenge challenge) {
                        DispatchQueue.main
                                .async(() -> completion.onReceiveChallenge(challenge));
                    }

                    @Override
                    public void onSucceed() {
                        addChatRoom(chatRoom);
                        _subscribeChatRoomBC(chatRoom, new Completion()
                                .onSuccess(() -> DispatchQueue.main.async(completion::onSucceed))
                                .onFailure(this::onFailed)
                        );
                    }

                    @Override
                    public void onFailed(@NonNull Error error) {
                        DispatchQueue.main.async(() -> completion.onFailed(error));
                    }
                })));
    }

//    public void respondChallenge(@NonNull final Challenges.Challenge challenge,
//                                 @Nullable final String response,
//                                 @NonNull final Completion completion) {
//        Log.i(TAG, Trace.once("respondChallenge"));
//        if (!(challenge instanceof ChallengeImpl.PasswordImpl)) {
//            return;
//        }
//
//        ChallengeImpl.PasswordImpl password = (ChallengeImpl.PasswordImpl) challenge;
//        ServiceProvider.get(Channel.class).run(new RPCJoinChatRoom(
//                (int) password.getRoomId(),
//                password.getPassword(),
//                new HashMap<>(),
//                completion));
//    }

    @Override
    public void leave(@NonNull final ChatRoom chatRoom,
                      @Nullable final CompletionArg<ChatRoom> completion) {
        Log.i(TAG, Trace.once("leave chatRoom: %s", chatRoom.getId()));
        HMRContext.work.async(() -> {
            removeChatRoom(chatRoom);
            ServiceProvider.get(Channel.class).run(new RPCLeaveChatRoom(
                    (int) chatRoom.getId(), new CompletionArg<ChatRoom>()
                    .onSuccess((arg) -> {
                        _unsubscribeChatRoomBC(chatRoom);
                        CompletionUtils.dispatchSuccess(completion, chatRoom);
                    }).onFailure((err) -> CompletionUtils.dispatchFailure(completion, err))));
        });
    }

    @Override
    public void kick(@NonNull final ChatRoom chatRoom,
                     @NonNull final User member,
                     @Nullable final Map<EKickInfo, String> extraInfo,
                     @NonNull final CompletionArg<ChatRoom> completion) {
        Log.i(TAG, Trace.once("kick id:%s fellow:%s", chatRoom.getId(), member.getId()));
        HMRContext.work.async(() -> {
            long secs = kickTimeSec;
            String reason = "";
            if (extraInfo != null && extraInfo.containsKey(EKickInfo.Time)) {
                secs = Long.parseLong(extraInfo.get(EKickInfo.Time));
            }
            if (extraInfo != null && extraInfo.containsKey(EKickInfo.Reason)) {
                reason = extraInfo.get(EKickInfo.Reason);
            }

            ServiceProvider.get(Channel.class).run(new RPCKickOffUser(
                    member.getId(),
                    (int) chatRoom.getId(),
                    secs,
                    reason,
                    completion));
        });
    }

    @Override
    public void addRole(@NonNull final ChatRoom chatRoom, @NonNull final User member,
                        @NonNull final String role,
                        @NonNull final CompletionArg<ChatRoom> completion) {
        Log.i(TAG, Trace.once("addRole id:%s fellow:%s role:%s",
                chatRoom.getId(),
                member.getId(),
                role));

        ServiceProvider.get(Channel.class).run(new RPCChangeChatRoomRole(
                chatRoom.getId(),
                member.getId(),
                new ChatRoomProto.AdminRoleTypeEnum(role),
                ChatRoomProto.ChatRoomOpEnum.ADD,
                completion));
    }

    @Override
    public void removeRole(@NonNull final ChatRoom chatRoom, @NonNull final User member,
                           @NonNull final String role,
                           @NonNull final CompletionArg<ChatRoom> completion) {
        Log.i(TAG, Trace.once("removeRole id:%s fellow:%s role:%s",
                chatRoom.getId(),
                member.getId(),
                role));

        ServiceProvider.get(Channel.class).run(new RPCChangeChatRoomRole(
                chatRoom.getId(),
                member.getId(),
                new ChatRoomProto.AdminRoleTypeEnum(role),
                ChatRoomProto.ChatRoomOpEnum.REMOVE,
                completion));
    }

    @Override
    public void fetchMembers(@NonNull final ChatRoom chatRoom,
                             final int num,
                             final int offset,
                             @NonNull final CompletionArg<List<User>> completion) {
        Log.i(TAG, Trace.once("fetchMembers id:%s num:%s pos:%s", chatRoom.getId(), num, offset));
        HMRContext.work.async(() -> ServiceProvider.get(Channel.class)
                .run(new RPCFetchMemberList((int) chatRoom.getId(),
                        num, offset, completion)));
    }

    @Override
    public void fetchRoleMembers(@NonNull final ChatRoom chatRoom, final boolean online,
                                 @NonNull final CompletionArg<Map<String, List<User>>> completion) {
        Log.i(TAG,
                Trace.once("fetch all roler Members id:%s, online:%s", chatRoom.getId(), online));

        if (online) {
            ServiceProvider.get(Channel.class).run(new RPCFetchAdminList(
                    (int) chatRoom.getId(), "all",
                    completion));
        } else {
            ServiceProvider.get(Channel.class)
                    .run(new RPCPullAllAdminUser((int) chatRoom.getId(), "all",
                            completion));
        }
    }

    @Override
    public void fetchBasicInfo(@NonNull final ChatRoom chatRoom,
                               @NonNull final CompletionArg<ChatRoomInfo> completion) {

        Log.i(TAG, Trace.once("fetchBasicInfo id:%s", chatRoom.getId()));

        ServiceProvider.get(Channel.class)
                .run(new RPCFetchChatRoomInfo((int) chatRoom.getId(), completion));
    }

    @Override
    public void changeBasicInfo(@NonNull final ChatRoom chatRoom,
                                @NonNull final Map<ChatRoomInfo.BasicInfoType, String> propInfo,
                                @Nullable final Completion completion) {

        Log.i(TAG, Trace.once("changeBasicInfo id:%s", chatRoom.getId()));

        ServiceProvider.get(Channel.class).run(new RPCChangeChatRoomInfo(
                (int) chatRoom.getId(),
                propInfo,
                completion));
    }

    @Override
    public void addListener(@NonNull final ChatRoomListener listener) {
        listeners.add(listener);
        Log.d(TAG, Trace.once()
                .method("addListener")
                .info("name", listener.getClass().getSimpleName())
                .info("size", listeners.size()));
    }

    @Override
    public void removeListener(@NonNull final ChatRoomListener listener) {
        listeners.remove(listener);
        Log.d(TAG, Trace.once()
                .method("removeListener")
                .info("name", listener.getClass().getSimpleName())
                .info("size", listeners.size()));
    }

    @Override
    public void addMemberListener(@NonNull final ChatRoomMembterListener listener) {
        memberListeners.add(listener);
        Log.d(TAG, Trace.once()
                .method("addMemberListener")
                .info("name", listener.getClass().getSimpleName())
                .info("size", memberListeners.size()));
    }

    @Override
    public void removeMemberListener(
            @NonNull ChatRoomMembterListener listener) {
        memberListeners.remove(listener);
        Log.d(TAG, Trace.once()
                .method("removeMemberListener")
                .info("name", listener.getClass().getSimpleName())
                .info("size", memberListeners.size()));
    }

    @Override
    public boolean shouldReceiveMessage(HMR.Message message) {
        return message instanceof MemberCountChangeMsgParser.MemberCountChangeMsg
                || message instanceof UpdateChatRoomMsgParser.UpdateChatRoomMsg
                || message instanceof MemberUpdateRoleMsgParser.MemberUpdateRoleMsg
                || message instanceof KickOffMembersMsgParser.KickOffMembersMsg
                || message instanceof ChatRoomDismissMsgParser.ChatRoomDismissMsg
                || message instanceof MemberChangeMsgParser.MemberChangeMsg;
    }

    @Override
    public void onReceiveMessage(HMR.Message message) {
        DispatchQueue.main.async(() -> {
            if (message instanceof MemberCountChangeMsgParser.MemberCountChangeMsg) {
                MemberCountChangeMsgParser.MemberCountChangeMsg msg =
                        (MemberCountChangeMsgParser.MemberCountChangeMsg) message;
                for (ChatRoomMembterListener listener : memberListeners) {
                    listener.onMemberCount(msg.chatRoom, msg.count);
                }
            } else if (message instanceof UpdateChatRoomMsgParser.UpdateChatRoomMsg) {
                UpdateChatRoomMsgParser.UpdateChatRoomMsg msg =
                        (UpdateChatRoomMsgParser.UpdateChatRoomMsg) message;
                for (ChatRoomListener listener : listeners) {
                    listener.onBasicInfoChanged(msg.chatRoom, msg.changeBasicInfoMap);
                }
            } else if (message instanceof MemberUpdateRoleMsgParser.MemberUpdateRoleMsg) {
                MemberUpdateRoleMsgParser.MemberUpdateRoleMsg msg =
                        (MemberUpdateRoleMsgParser.MemberUpdateRoleMsg) message;
                for (ChatRoomMembterListener listener : memberListeners) {
                    if (msg.isAdd) {
                        listener.onRoleAdded(msg.chatRoom, msg.role,
                                msg.admin,
                                msg.user);
                    } else {
                        listener.onRoleRemoved(msg.chatRoom, msg.role,
                                msg.admin,
                                msg.user);
                    }
                }
            } else if (message instanceof KickOffMembersMsgParser.KickOffMembersMsg) {
                KickOffMembersMsgParser.KickOffMembersMsg msg =
                        (KickOffMembersMsgParser.KickOffMembersMsg) message;
                for (ChatRoomMembterListener listener : memberListeners) {
                    listener.onMemberKicked(msg.chatRoom, msg.admin, msg.kickedUsers, msg.reason);
                }
            } else if (message instanceof ChatRoomDismissMsgParser.ChatRoomDismissMsg) {
                ChatRoomDismissMsgParser.ChatRoomDismissMsg msg =
                        (ChatRoomDismissMsgParser.ChatRoomDismissMsg) message;
                for (ChatRoomListener listener : listeners) {
                    listener.onDeleteChatRoom(msg.chatRoom, msg.operator);
                }
            } else if (message instanceof MemberChangeMsgParser.MemberChangeMsg) {
                MemberChangeMsgParser.MemberChangeMsg msg =
                        (MemberChangeMsgParser.MemberChangeMsg) message;
                for (ChatRoomMembterListener listener : memberListeners) {
                    if (msg.isJoin) {
                        Log.i(TAG,
                                Trace.once().method("onReceiveMessage").msg("member join, size = " + msg.users.size()));
                        listener.onMemberJoin(msg.chatRoom, msg.users);
                    } else {
                        Log.i(TAG,
                                Trace.once().method("onReceiveMessage")
                                        .msg("member leave, size = " + msg.users.size()));
                        listener.onMemberLeave(msg.chatRoom, msg.users);
                    }
                }

            }
        });
    }

    @SuppressWarnings("WeakerAccess")
    public void addChatRoom(ChatRoom chatRoom) {
        chatRooms.add(chatRoom);
    }

    @SuppressWarnings("WeakerAccess")
    public void removeChatRoom(ChatRoom chatRoom) {
        chatRooms.remove(chatRoom);
    }

    private void _subscribeChatRoomBC(ChatRoom chatRoom, @NonNull Completion completion) {
        long roomId = chatRoom.getId();
        UserGroupType groupType = new UserGroupType(textChatBCGrpType, roomId);
        UserGroupType groupType2 = new UserGroupType(authBCGrpType, roomId);
        UserGroupType groupType3 = new UserGroupType(onlineBCGrpType, roomId);
        ArrayList<UserGroupType> lists = new ArrayList<>();
        lists.add(groupType);
        lists.add(groupType2);
        lists.add(groupType3);
        YYServiceCore.getInstance().subscribeBroadcast(lists,
                new IRPCChannel.RPCCallback<BroadSubOrUnSubTask.ResponseParam>() {
                    @Override
                    public void onSuccess(int var1, BroadSubOrUnSubTask.ResponseParam response) {
                        Log.i(TAG, Trace.once().method("subscribeBroadcast success")
                                .msg("resCode:%d, roomId:%d", response.mResCode, roomId));
                        CompletionUtils.dispatchSuccess(completion);
                    }

                    @Override
                    public void onFail(int requestId, int sdkResCode, int srvResCode, Exception e) {
                        Log.e(TAG, Trace.once().method("subscribeBroadcast failed")
                                .msg("sdkResCode:%d, srvResCode:%d, roomId:%d", sdkResCode,
                                        srvResCode,
                                        roomId));
                        CompletionUtils
                                .dispatchFailure(completion, new Error(srvResCode, "service err"));
                    }
                });
    }

    private void _unsubscribeChatRoomBC(ChatRoom chatRoom) {
        long roomId = chatRoom.getId();
        UserGroupType groupType = new UserGroupType(textChatBCGrpType, roomId);
        UserGroupType groupType2 = new UserGroupType(authBCGrpType, roomId);
        UserGroupType groupType3 = new UserGroupType(onlineBCGrpType, roomId);
        ArrayList<UserGroupType> lists = new ArrayList<>();
        lists.add(groupType);
        lists.add(groupType2);
        lists.add(groupType3);
        YYServiceCore.getInstance().unSubscribeBroadcast(lists,
                new IRPCChannel.RPCCallback<BroadSubOrUnSubTask.ResponseParam>() {
                    public void onSuccess(int var1, BroadSubOrUnSubTask.ResponseParam response) {
                        Log.i(TAG, Trace.once().method("unSubscribeBroadcast success")
                                .msg("resCode:%d, roomId:%d",
                                        response.mResCode, roomId));
                    }

                    public void onFail(int requestId, int sdkResCode, int srvResCode, Exception e) {
                        Log.i(TAG, Trace.once().method("unSubscribeBroadcast failed")
                                .msg("sdkResCode:%d, srvResCode:%d, roomId:%d",
                                        sdkResCode, srvResCode, roomId));
                    }
                });
    }

    private void _onChannelConnected() {
        _rejoinChatRoom();
    }

    private void _rejoinChatRoom() {
        if (chatRooms.size() > 0) {
            for (ChatRoom chatRoom : chatRooms) {
                Log.i(TAG, String.format(Locale.US,
                        "rejoin chatRoom after channel connected, roomId:%d",
                        chatRoom.getId()));

                ServiceProvider.get(Channel.class)
                        .run(new RPCJoinChatRoom((int) chatRoom.getId(),
                                new HashMap<>(), null));
            }
        }
    }

    private final Set<ChatRoomMembterListener> memberListeners = new HashSet<>();
    private final Set<ChatRoomListener> listeners = new HashSet<>();
    private final Set<ChatRoom> chatRooms = new HashSet<>();
    private Channel.StateChangedListener channelStateListener;
}
