package com.hummer.im.chatroom;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im.HMR;
import com.hummer.im.id.ChatRoom;
import com.hummer.im.id.User;
import com.hummer.im.shared.completion.Completion;
import com.hummer.im.shared.completion.CompletionArg;

import java.util.List;
import java.util.Map;

public interface ChatRoomService extends HMR.Service {

    /**
     * 创建聊天室
     *
     * @param chatRoomInfo 聊天室信息
     * @param completion 回调
     */
    public void createChatRoom(@NonNull ChatRoomInfo chatRoomInfo,
                               @NonNull  final CompletionArg<ChatRoom> completion);

    /**
     * 删除聊天室
     *
     * @param chatRoom   删除的聊天室
     * @param completion 回调
     */
    void deleteChatRoom(@NonNull ChatRoom chatRoom,
                        @Nullable CompletionArg<ChatRoom> completion);


    /**
     * 加入聊天室
     *
     * @param chatRoom   聊天室标识符
     * @param completion 回调
     */
    void join(@NonNull ChatRoom chatRoom, @NonNull Challenges.JoiningCompletion completion);

    /**
     * 离开聊天室
     *
     * @param chatRoom   聊天室标识符
     * @param completion 回调
     */
    void leave(@NonNull ChatRoom chatRoom, @Nullable CompletionArg<ChatRoom> completion);

    enum EKickInfo {
        Time,
        Reason,
    }

    /**
     * 踢用户出聊天室
     *
     * @param chatRoom   聊天室标识符
     * @param member     踢出的聊天室成员
     * @param extraInfo  踢人额外信息
     * @param completion 回调
     */
    void kick(@NonNull ChatRoom chatRoom,
              @NonNull User member,
              @Nullable Map<EKickInfo, String> extraInfo,
              @NonNull CompletionArg<ChatRoom> completion);


    final class DefaultRoleType {
        public static final String admin = "admin";
        public static final String owner = "owner";
    }

    /**
     * 添加聊天室权限角色
     *
     * @param chatRoom   聊天室标识符
     * @param member     成员
     * @param role       角色类型,具体值可参考{@link DefaultRoleType},业务也可以自己定义扩展角色
     * @param completion 回调
     */
    void addRole(@NonNull ChatRoom chatRoom,
                 @NonNull User member,
                 @NonNull String role,
                 @NonNull CompletionArg<ChatRoom> completion);

    /**
     * 移除聊天室权限角色
     *
     * @param chatRoom   聊天室标识符
     * @param member     成员
     * @param role       角色类型,具体值可参考{@link DefaultRoleType},业务也可以自己定义扩展角色
     * @param completion 回调
     */
    void removeRole(@NonNull ChatRoom chatRoom,
                    @NonNull User member,
                    @NonNull String role,
                    @NonNull CompletionArg<ChatRoom> completion);

    /**
     * 获取聊天室成员列表
     *
     * @param chatRoom   聊天室标识符
     * @param num        拉取的条数
     * @param offset     拉取的位置，第一页从0开始
     * @param completion 回调,List<Fellow> 聊天室成员列表
     */
    void fetchMembers(@NonNull ChatRoom chatRoom, int num, int offset,
                      @NonNull CompletionArg<List<User>> completion);

    /**
     * 获取聊天室的所有角色列表
     *
     * @param chatRoom   聊天室标识符
     * @param online     true获取在线的角色列表，false获取全部角色列表
     * @param completion 回调, Map<String, List<Fellow> 角色成员列表,key 对应的取值是DefaultRoleType类型，或者是业务自定义角色类型
     */
    void fetchRoleMembers(@NonNull ChatRoom chatRoom, boolean online,
                          @NonNull CompletionArg<Map<String, List<User>>> completion);

    /**
     * 获取聊天室基本属性信息
     *
     * @param chatRoom   聊天室标识符
     * @param completion 回调, RoomInfo 聊天室属性信息
     */
    void fetchBasicInfo(@NonNull ChatRoom chatRoom,
                        @NonNull CompletionArg<ChatRoomInfo> completion);


    /**
     * 修改聊天室基本属性信息
     *
     * @param chatRoom   聊天室标识符
     * @param propInfo   修改属性值
     * @param completion 回调
     */
    void changeBasicInfo(@NonNull ChatRoom chatRoom,
                         @NonNull Map<ChatRoomInfo.BasicInfoType, String> propInfo,
                         @NonNull Completion completion);

    /**
     * 添加聊天室回调监听器
     *
     * @param listener 监听器对象
     */
    void addListener(@NonNull final ChatRoomListener listener);

    /**
     * 移除聊天室回调监听器
     *
     * @param listener 监听器对象
     */
    void removeListener(@NonNull final ChatRoomListener listener);

    /**
     * 添加聊天室成员相关回调监听器
     *
     * @param listener 监听器对象
     */
    void addMemberListener(@NonNull final ChatRoomMembterListener listener);

    /**
     * 移除聊天室成员相关回调监听器
     *
     * @param listener 监听器对象
     */
    void removeMemberListener(@NonNull final ChatRoomMembterListener listener);

    interface ChatRoomListener {

        /**
         * 信令通道的广播
         */
        default void onBroadcast(@NonNull ChatRoom chatRoom, @NonNull User fellow, @NonNull String chat) { }

        /**
         * 聊天室基本信息变化的回调通知
         */
        default void onBasicInfoChanged(@NonNull ChatRoom chatRoom,
                                        @NonNull Map<ChatRoomInfo.BasicInfoType, String> propInfo) { }

        /**
         * 聊天室销毁的回调通知
         *
         * @param fellow   操作者
         * @param chatRoom 聊天室
         */
        default void onDeleteChatRoom(@NonNull ChatRoom chatRoom, @NonNull User fellow) { }
    }

    interface ChatRoomMembterListener {
        /**
         * 用户加入聊天室的回调通知
         */
        default void onMemberJoin(@NonNull ChatRoom chatRoom, @NonNull List<User> fellows) { }

        /**
         * 用户离开聊天室的回调通知
         */
        default void onMemberLeave(@NonNull ChatRoom chatRoom, @NonNull List<User> fellows) { }

        /**
         * 聊天室成员数变化的回调通知
         */
        default void onMemberCount(@NonNull ChatRoom chatRoom, int count) { }

        /**
         * 添加聊天室权限角色的回调通知
         */
        default void onRoleAdded(@NonNull ChatRoom chatRoom, @NonNull String role,
                                 @NonNull User admin,
                                 @NonNull User fellow) { }

        /**
         * 移除聊天室权限角色的回调通知
         */
        default void onRoleRemoved(@NonNull ChatRoom chatRoom,
                                   @NonNull String role,
                                   @NonNull User admin,
                                   @NonNull User fellow) { }

        /**
         * 用户被踢出聊天室的回调通知
         *
         * @param chatRoom 聊天室
         * @param admin    管理员
         * @param fellow   被踢用户
         * @param reason   原因
         */
        default void onMemberKicked(@NonNull ChatRoom chatRoom,
                                    @NonNull User admin,
                                    @NonNull List<User> fellow,
                                    @NonNull String reason) { }
    }
}




