package com.hummer.im.chatroom._internals.helper;

import com.hummer.im._internals.bridge.helper.HummerEvent;
import com.hummer.im._internals.log.Log;
import com.hummer.im.chatroom.ChatRoomInfo;
import com.hummer.im.chatroom.ChatRoomService;
import com.hummer.im.chatroom.model.attribute.ChatRoomAttributeOptions;
import com.hummer.im.chatroom.model.attribute.RoomBasicAttributesOptions;
import com.hummer.im.chatroom.model.kick.KickUserOptions;
import com.hummer.im.model.fetch.Direction;
import com.hummer.im.model.fetch.FetchingParams;
import com.hummer.im.model.fetch.MsgType;
import com.hummer.im.model.id.ChatRoom;
import com.hummer.im.model.id.User;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;


public class ChatRoomEvent {

    private static final String TAG = "ChatRoomEvent";

    private static final String KICK_REASON = "Reason";
    private static final String KICK_TIME = "Time";

    /********************************  Events Begin *****************************************/

    // ChatRoom
    public static final int EVENT_ROOM_SET_REGION = 300;
    public static final int EVENT_ROOM_CREATE_ROOM = 301;
    public static final int EVENT_ROOM_DISMISS_ROOM = 302;
    public static final int EVENT_ROOM_FETCH_BASIC_INFO             = 303;//"获取聊天室基本属性信息",
    public static final int EVENT_ROOM_CHANGE_BASIC_INFO            = 304;//"修改聊天室基本信息",
    public static final int EVENT_ROOM_FETCH_MEMBER_COUNT           = 306;//"获取聊天室人数",

    /** 查询聊天室的扩展属性 */
    public static final int EVENT_ROOM_FETCH_ROOM_EXTRA_ATTRIBUTES = 308;
    /** 设置聊天室扩展属性 */
    public static final int EVENT_ROOM_SET_ROOM_EXTRA_ATTRIBUTES = 309;
    /** 更新聊天室扩展属性 */
    public static final int EVENT_ROOM_UPDATE_ROOM_EXTRA_ATTRIBUTES = 310;
    /** 删除聊天室扩展属性 */
    public static final int EVENT_ROOM_DELETE_ROOM_EXTRA_ATTRIBUTES = 311;
    /** 清空聊天室扩展属性 */
    public static final int EVENT_ROOM_CLEAR_ROOM_EXTRA_ATTRIBUTES = 312;

    /** 更新聊天室基础信息 */
    public static final int EVENT_ROOM_UPDATE_ROOM_BASIC_ATTRIBUTES = 313;
    /** 获取聊天室基础信息 */
    public static final int EVENT_ROOM_FETCH_ROOM_BASIC_ATTRIBUTES = 314;

    // ChatRoom Member
    public static final int EVENT_ROOM_JOIN_ROOM = 401;
    public static final int EVENT_ROOM_EXIT_ROOM               = 402;//退出房间
    public static final int EVENT_ROOM_KICK                         = 403;//踢出房间
    public static final int EVENT_ROOM_ADD_ROLE                     = 404;//"添加权限角色",
    public static final int EVENT_ROOM_REMOVE_ROLE                  = 405;//"移除权限角色",
    public static final int EVENT_ROOM_FETCH_MEMBERS                = 406;//"获取成员列表",
    public static final int EVENT_ROOM_FETCH_ROLE_MEMBERS           = 407;//"获取带有角色的成员列表",
    public static final int EVENT_ROOM_MUTE_MEMBER                  = 408;//"禁言聊天室内的成员",
    public static final int EVENT_ROOM_UN_MUTE_MEMBER               = 409;//"解禁成员聊天室内的成员",
    public static final int EVENT_ROOM_FETCH_MUTED_USERS            = 410;//"获取禁言列表",
    public static final int EVENT_ROOM_IS_MUTED                     = 411;//"判别用户是否被禁言",
    public static final int EVENT_ROOM_SET_USER_INFO                = 412;//"设置自己的用户信息",
    public static final int EVENT_ROOM_DELETE_USER_INFO_BY_KEYS     = 413;//"删除自己的用户信息",
    public static final int EVENT_ROOM_ADD_OR_UPDATE_USER_INFO      = 414;//"添加或更新指定用户信息",
    public static final int EVENT_ROOM_FETCH_USER_INFO              = 415;//"获取指定用户的用户信息",
    public static final int EVENT_ROOM_BATCH_FETCH_USER_INFO        = 416;//"批量获取指定用户的用户信息",
    public static final int EVENT_ROOM_FETCH_ONLINE_USER_INFO_LIST  = 417;//"获取聊天室已设置用户信息的在线成员列表";

    /** 设置聊天室用户角色 */
    public static final int EVENT_ROOM_SET_USER_ROLE = 418;
    /** 删除聊天室用户角色 */
    public static final int EVENT_ROOM_DELETE_USER_ROLE = 419;
    /** 查询用户角色 */
    public static final int EVENT_ROOM_FETCH_USER_ROLE = 420;
    /** 查询在聊天室内带角色的用户列表 */
    public static final int EVENT_ROOM_FETCH_ROOM_USERS_BY_ROLE = 421;
    /** 踢人 */
    public static final int EVENT_ROOM_KICK_USER = 422;

    // ChatRoom Message

    /** 查询聊天室历史消息 */
    public static final int EVENT_ROOM_FETCH_HISTORY_MESSAGES = 501;

    public static class EventSetRegion extends HummerEvent.EventBase {
        private String region;

        EventSetRegion(String region) {
            event = EVENT_ROOM_SET_REGION;
            this.region = region;
        }

        @Override
        public byte[] marshall() {
            pushString16(region);
            return super.marshall();
        }
    }

    public static class EventChatRoom extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;

        EventChatRoom(long requestId, ChatRoom chatRoom) {
            this.requestId = requestId;
            this.chatRoom = chatRoom;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            return super.marshall();
        }
    }

    public static class EventCreateChatRoom extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoomInfo chatRoomInfo;

        EventCreateChatRoom(long requestId, ChatRoomInfo chatRoomInfo) {
            event = EVENT_ROOM_CREATE_ROOM;
            this.requestId = requestId;
            if (chatRoomInfo != null) {
                this.chatRoomInfo = chatRoomInfo;
            } else {
                this.chatRoomInfo = new ChatRoomInfo("", "", "", "");
            }
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            pushString16(chatRoomInfo.getName());
            pushString16(chatRoomInfo.getDescription());
            pushString16(chatRoomInfo.getBulletin());
            pushString16(chatRoomInfo.getAppExtra());
            return super.marshall();
        }
    }

    public static class EventJoinRoom extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom roomId;
        private Map<String, String> joinProps;

        EventJoinRoom(long requestId, ChatRoom roomId, Map<String, String> joinProps) {
            event = EVENT_ROOM_JOIN_ROOM;
            this.requestId = requestId;
            this.roomId = roomId;
            this.joinProps = joinProps;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(roomId == null ? 0 : roomId.getId());
            pushMap(joinProps, String.class);
            return super.marshall();
        }
    }

    public static class EventKickUser extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private User member;
        private long seconds;
        private String reason;

        EventKickUser(long requestId, ChatRoom chatRoom, User member,
                      Map<String, String> extraInfo, KickUserOptions options) {
            event = EVENT_ROOM_KICK_USER;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.member = member;
            this.seconds = 0;
            this.reason = "";
            if (extraInfo != null && extraInfo.size() > 0) {
                if (extraInfo.containsKey(KICK_REASON)) {
                    this.reason = extraInfo.get(KICK_REASON);
                }
                if (extraInfo.containsKey(KICK_TIME)) {
                    try {
                        String time = extraInfo.get(KICK_TIME);
                        if (time != null) {
                            this.seconds = Long.parseLong(time);
                        }
                    } catch (NumberFormatException e) {
                        Log.e(TAG, "NumberFormatException: " + e.getMessage());
                    }
                }
            }
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushInt64(member == null ? 0 : member.getId());
            pushInt64(seconds);
            pushString16(reason);
            return super.marshall();
        }
    }

    public static class EventKick extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private User member;
        private long seconds;
        private String reason;

        EventKick(long requestId, ChatRoom chatRoom, User member,
                      Map<ChatRoomService.EKickInfo, String> extraInfo) {
            event = EVENT_ROOM_KICK;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.member = member;
            this.seconds = 0;
            this.reason = "";
            if (extraInfo != null) {
                if (extraInfo.containsKey(ChatRoomService.EKickInfo.Reason)) {
                    this.reason = extraInfo.get(ChatRoomService.EKickInfo.Reason);
                }
                if (extraInfo.containsKey(ChatRoomService.EKickInfo.Time)) {
                    try {
                        this.seconds = Long.parseLong(extraInfo.get(ChatRoomService.EKickInfo.Time));
                    } catch (NumberFormatException e) {
                        Log.e(TAG, "NumberFormatException: " + e.getMessage());
                    }

                }
            }
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushInt64(member == null ? 0 : member.getId());
            pushInt64(seconds);
            pushString16(reason);
            return super.marshall();
        }
    }

    public static class EventRole extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private User member;
        private String role;

        EventRole(long requestId, ChatRoom chatRoom, User member, String role) {
            event = EVENT_ROOM_ADD_ROLE;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.member = member;
            this.role = role;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushInt64(member == null ? 0 : member.getId());
            pushString16(role);
            return super.marshall();
        }
    }

    public static class EventFetchMembers extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private int num;
        private int offset;

        EventFetchMembers(long requestId, ChatRoom chatRoom, int num, int offset) {
            event = EVENT_ROOM_FETCH_MEMBERS;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.num = num;
            this.offset = offset;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushInt(num);
            pushInt(offset);
            return super.marshall();
        }
    }

    public static class EventFetchUserRole extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private User user;

        EventFetchUserRole(long requestId, ChatRoom chatRoom, User user) {
            event = EVENT_ROOM_FETCH_USER_ROLE;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.user = user;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushInt64(user == null ? 0 : user.getId());
            return super.marshall();
        }
    }

    public static class EventFetchRoomUsersByRole extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private String role;

        EventFetchRoomUsersByRole(long requestId, ChatRoom chatRoom, String role) {
            event = EVENT_ROOM_FETCH_ROOM_USERS_BY_ROLE;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.role = role;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushString16(role == null ? "" : role);
            return super.marshall();
        }
    }

    public static class EventFetchRoleMembers extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private boolean online;

        EventFetchRoleMembers(long requestId, ChatRoom chatRoom, boolean online) {
            event = EVENT_ROOM_FETCH_ROLE_MEMBERS;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.online = online;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushBool(online);
            return super.marshall();
        }
    }

    public static class EventFetchRoomBasicAttributes extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private Set<String> keys;

        EventFetchRoomBasicAttributes(long requestId,
                                      ChatRoom chatRoom,
                                      Set<String> keys) {
            event = EVENT_ROOM_FETCH_ROOM_BASIC_ATTRIBUTES;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.keys = keys;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            if (keys == null) {
                keys = new HashSet<>();
            }
            pushInt(keys.size());
            for (String key : keys) {
                pushString16(key);
            }
            return super.marshall();
        }
    }

    public static class EventChangeBasicInfo extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private Map<String, String> propInfo;

        EventChangeBasicInfo(long requestId, ChatRoom chatRoom, Map<ChatRoomInfo.BasicInfoType, String> propInfo) {
            event = EVENT_ROOM_CHANGE_BASIC_INFO;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.propInfo = new HashMap<>();
            if (propInfo != null) {
                if (propInfo.containsKey(ChatRoomInfo.BasicInfoType.Name)) {
                    this.propInfo.put("Name", propInfo.get(ChatRoomInfo.BasicInfoType.Name));
                }
                if (propInfo.containsKey(ChatRoomInfo.BasicInfoType.Description)) {
                    this.propInfo.put("Description", propInfo.get(ChatRoomInfo.BasicInfoType.Description));
                }
                if (propInfo.containsKey(ChatRoomInfo.BasicInfoType.Bulletin)) {
                    this.propInfo.put("Bulletin", propInfo.get(ChatRoomInfo.BasicInfoType.Bulletin));
                }
                if (propInfo.containsKey(ChatRoomInfo.BasicInfoType.AppExtra)) {
                    this.propInfo.put("AppExtra", propInfo.get(ChatRoomInfo.BasicInfoType.AppExtra));
                }
            }
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushMap(propInfo, String.class);
            return super.marshall();
        }
    }

    public static class EventUpdateRoomBasicAttributes extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private Map<String, String> attributes;

        EventUpdateRoomBasicAttributes(long requestId,
                                       ChatRoom chatRoom,
                                       Map<String, String> attributes,
                                       RoomBasicAttributesOptions options) {
            event = EVENT_ROOM_UPDATE_ROOM_BASIC_ATTRIBUTES;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.attributes = attributes;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushMap(attributes, String.class);
            return super.marshall();
        }
    }

    public static class EventFetchRoomExtraAttributes extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private Set<String> keys;

        public EventFetchRoomExtraAttributes(long requestId, ChatRoom chatRoom, Set<String> keys) {
            event = EVENT_ROOM_FETCH_ROOM_EXTRA_ATTRIBUTES;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.keys = keys;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            if (keys == null) {
                keys = new HashSet<>();
            }
            pushInt(keys.size());
            for (String key : keys) {
                pushString16(key);
            }
            return super.marshall();
        }
    }

    public static class EventSetRoomExtraAttributes extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private Map<String, String> attributes;
        private ChatRoomAttributeOptions options;

        public EventSetRoomExtraAttributes(long requestId,
                                           ChatRoom chatRoom,
                                           Map<String, String> attributes,
                                           ChatRoomAttributeOptions options) {
            event = EVENT_ROOM_SET_ROOM_EXTRA_ATTRIBUTES;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.attributes = attributes;
            this.options = options;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushMap(attributes, String.class);

            return super.marshall();
        }
    }

    public static class EventUpdateRoomExtraAttributes extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private Map<String, String> attributes;
        private ChatRoomAttributeOptions options;

        public EventUpdateRoomExtraAttributes(long requestId,
                                              ChatRoom chatRoom,
                                              Map<String, String> attributes,
                                              ChatRoomAttributeOptions options) {
            event = EVENT_ROOM_UPDATE_ROOM_EXTRA_ATTRIBUTES;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.attributes = attributes;
            this.options = options;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushMap(attributes, String.class);

            return super.marshall();
        }
    }

    public static class EventDeleteRoomExtraAttributes extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private Set<String> keys;
        private ChatRoomAttributeOptions options;

        public EventDeleteRoomExtraAttributes(long requestId,
                                              ChatRoom chatRoom,
                                              Set<String> keys,
                                              ChatRoomAttributeOptions options) {
            event = EVENT_ROOM_DELETE_ROOM_EXTRA_ATTRIBUTES;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.keys = keys;
            this.options = options;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            if (keys == null) {
                keys = new HashSet<>();
            }
            pushInt(keys.size());
            for (String key : keys) {
                pushString16(key);
            }

            return super.marshall();
        }
    }

    public static class EventClearRoomExtraAttributes extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private ChatRoomAttributeOptions options;

        public EventClearRoomExtraAttributes(long requestId,
                                             ChatRoom chatRoom,
                                             ChatRoomAttributeOptions options) {
            event = EVENT_ROOM_CLEAR_ROOM_EXTRA_ATTRIBUTES;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.options = options;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            pushInt64(chatRoom == null ? 0 : chatRoom.getId());

            return super.marshall();
        }
    }

    public static class EventMuteMember extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private User member;
        private String reason;

        EventMuteMember(long requestId, ChatRoom chatRoom, User member, String reason) {
            event = EVENT_ROOM_MUTE_MEMBER;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.member = member;
            this.reason = reason;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushInt64(member == null ? 0 : member.getId());
            pushString16(reason);
            return super.marshall();
        }
    }

    public static class EventBatchFetchUserInfos extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private Set<User> users;

        EventBatchFetchUserInfos(long requestId, ChatRoom chatRoom, Set<User> users) {
            event = EVENT_ROOM_BATCH_FETCH_USER_INFO;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.users = users;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            if (users != null) {
                pushInt(users.size());
                for (User user : users) {
                    pushInt64(user.getId());
                }
            }
            return super.marshall();
        }
    }

    public static class EventUserInfo extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private Map<String, String> infoMap;

        EventUserInfo(long requestId, ChatRoom chatRoom, Map<String, String> infoMap) {
            event = EVENT_ROOM_BATCH_FETCH_USER_INFO;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.infoMap = infoMap;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushMap(infoMap, String.class);
            return super.marshall();
        }
    }

    public static class EventDeleteUserInfoByKeys extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private Set<String> keys;

        EventDeleteUserInfoByKeys(long requestId, ChatRoom chatRoom, Set<String> keys) {
            event = EVENT_ROOM_DELETE_USER_INFO_BY_KEYS;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.keys = keys;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            if (keys != null) {
                pushInt(keys.size());
                for (String key : keys) {
                    pushString16(key);
                }
            }
            return super.marshall();
        }
    }

    public static class EventFetchUserInfo extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private User member;

        EventFetchUserInfo(long requestId, ChatRoom chatRoom, User member) {
            event = EVENT_ROOM_FETCH_USER_INFO;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.member = member;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            //底层roomId是32位int
            pushInt(chatRoom == null ? 0 : chatRoom.getId());
            pushInt64(member == null ? 0 : member.getId());
            return super.marshall();
        }
    }

    public static class EventFetchHistoryMessages extends HummerEvent.EventBase {
        private long requestId;
        private ChatRoom chatRoom;
        private FetchingParams params;

        public EventFetchHistoryMessages(long requestId, ChatRoom chatRoom, FetchingParams params) {
            event = EVENT_ROOM_FETCH_HISTORY_MESSAGES;
            this.requestId = requestId;
            this.chatRoom = chatRoom;
            this.params = params;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            pushInt(chatRoom == null ? 0 : chatRoom.getId());

            Set<MsgType> msgTypes = params.getMsgTypes();
            Set<Integer> messageTypes = new HashSet<>();
            if (msgTypes != null) {
                for (MsgType msgType : msgTypes) {
                    messageTypes.add(msgType.getType());
                }
            }
            pushCollection(messageTypes, Integer.class);

            if (params.getAnchor() == null) {
                pushString16(null);
                pushInt64(0);
            } else {
                pushString16(params.getAnchor().getUuid());
                pushInt64(params.getAnchor().getTimestamp());
            }

            pushInt(params.getDirection() == null ? Direction.OLD.getCode() : params.getDirection().getCode());
            pushInt(params.getLimit());

            return super.marshall();
        }
    }

}
