package com.hummer.im.chatroom._internals;

import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.hummer.im.Error;
import com.hummer.im.ErrorEnum;
import com.hummer.im.HMR;
import com.hummer.im._internals.ChatServiceImpl;
import com.hummer.im._internals.bridge.helper.HummerDispatch;
import com.hummer.im._internals.bridge.helper.HummerNative;
import com.hummer.im._internals.bridge.marshall.Marshallable;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.Trace;
import com.hummer.im._internals.utility.CompletionUtils;
import com.hummer.im._internals.utility.HMRCompletion;
import com.hummer.im._internals.utility.HMRCompletionArg;
import com.hummer.im._internals.utility.HMRContext;
import com.hummer.im._internals.utility.ReportFunction;
import com.hummer.im._internals.utility.RequestIdBuilder;
import com.hummer.im._internals.utility.ServiceProvider;
import com.hummer.im.chatroom.Challenges;
import com.hummer.im.chatroom.ChatRoomInfo;
import com.hummer.im.chatroom.ChatRoomService;
import com.hummer.im.chatroom._internals.helper.ChatRoomNative;
import com.hummer.im.chatroom._internals.helper.ChatRoomNotification;
import com.hummer.im.chatroom.model.attribute.ChatRoomAttributeOptions;
import com.hummer.im.chatroom.model.attribute.ChatRoomExtraAttribute;
import com.hummer.im.chatroom.model.attribute.FetchRoomBasicAttributesResult;
import com.hummer.im.chatroom.model.attribute.RoomBasicAttributesOptions;
import com.hummer.im.chatroom.model.kick.KickOutType;
import com.hummer.im.chatroom.model.kick.KickUserOptions;
import com.hummer.im.chatroom.model.role.RoleOptions;
import com.hummer.im.model.RequestId;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.contents.Text;
import com.hummer.im.model.fetch.FetchingParams;
import com.hummer.im.model.fetch.FetchingResult;
import com.hummer.im.model.id.ChatRoom;
import com.hummer.im.model.id.User;
import com.hummer.im.model.kick.KickOff;
import com.hummer.im.model.kick.KickOffEnum;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;


public class ChatRoomServiceImpl implements ChatRoomService, ServiceProvider.Service, HummerNative.NotificationListener {

    private static final String INVALID_UID = "Invalid Uid";
    private static final String INVALID_OPERATED_UID = "Invalid Operated Uid";
    private static final String INVALID_ROOM_ID = "Invalid RoomId";
    private static final long MIN_ROOM_ID = 1;
    private static final long MAX_ROOM_ID = (1L << 32) - 2;

    /********************************  Notify Begin *****************************************/

    private class NotifyBase extends Marshallable {
        Object notification;

        public Object get() {
            return notification;
        }
    }

    private class NotifyBaseCallback extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.BaseCallbackInfo(popInt64(), popInt(), popString16UTF8());
        }
    }

    private class NotifyOnCreateChatRoom extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnCreateChatRoom(popInt64(), popInt(), popString16UTF8(), popInt());
        }
    }

    private class NotifyOnJoinChatRoom extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnJoinRoom(popInt64(), popInt(),
                    popString16UTF8(), popInt2Long(),
                    popMap(String.class, String.class));
        }
    }

    private class NotifyOnLeaveChatRoom extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnLeaveRoom(popInt64(), popInt(),
                    popString16UTF8(), popInt2Long());
        }
    }

    private class NotifyOnFetchUserRole extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long requestId = popInt64();
            int code = popInt();
            String desc = popString16UTF8();
            long roomId = popInt2Long();
            String role = popString16UTF8();
            notification = new ChatRoomNotification.OnFetchUserRole(requestId, code, desc, roomId, role);
        }
    }

    private class NotifyOnFetchMembers extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long requestId = popInt64();
            int code = popInt();
            String desc = popString16UTF8();
            long roomId = popInt2Long();
            int size = popInt();
            Set<Long> uids = new HashSet<>();
            for (int i = 0; i < size; i++) {
                uids.add(popInt64());
            }
            notification = new ChatRoomNotification.OnFetchMembers(requestId, code, desc, roomId, uids);
        }
    }

    private class NotifyOnFetchRoleMembers extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long requestId = popInt64();
            int code = popInt();
            String desc = popString16UTF8();
            long roomId = popInt2Long();
            Map<String, List<User>> data = new HashMap<>();
            int size = popInt();
            for (int i = 0; i < size; i++) {
                String key = popString16UTF8();
                List<User> users = new ArrayList<>();
                int count = popInt();
                for (int j = 0; j < count; j++) {
                    users.add(new User(popInt64()));
                }
                data.put(key, users);
            }
            notification = new ChatRoomNotification.OnFetchRoleMembers(requestId, code, desc, roomId, data);
        }
    }

    private class NotifyOnFetchMemberCount extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnFetchMemberCount(popInt64(), popInt(), popString16UTF8(), popInt2Long(), popInt());
        }
    }

    private class NotifyOnRoomProps extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnRoomProps(popInt64(), popInt(), popString16UTF8(), popInt2Long(), popMap(String.class, String.class));
        }
    }

    private class NotifyOnFetchMutedUsers extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long requestId = popInt64();
            int code = popInt();
            String desc = popString16UTF8();
            long roomId = popInt2Long();
            int size = popInt();
            Set<Long> uids = new HashSet<>();
            for (int i = 0; i < size; i++) {
                uids.add(popInt64());
            }
            notification = new ChatRoomNotification.OnFetchMutedUsers(requestId, code, desc, roomId, uids);
        }
    }

    private class NotifyOnIsMuted extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnIsMuted(popInt64(), popInt(), popString16UTF8(), popInt2Long(), popBool());
        }
    }

    private class NotifyOnUserInfoList extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long requestId = popInt64();
            int code = popInt();
            String desc = popString16UTF8();
            long roomId = popInt2Long();
            int size = popInt();
            Map<User, Map<String, String>> userInfoMap = new HashMap<>();
            for (int i = 0; i < size; i++) {
                long uid = popInt64();
                userInfoMap.put(new User(uid), popMap(String.class, String.class));
            }
            notification = new ChatRoomNotification.OnUserInfoList(requestId, code, desc, roomId, userInfoMap);
        }
    }

    private class NotifyOnFetchHistoryMessages extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long requestId = popInt64();
            int code = popInt();
            String desc = popString16UTF8();
            long roomId = popInt2Long();
            boolean hasMore = popBool();
            int size = popInt();
            List<Message> messages = new ArrayList<>();
            for (int i = 0; i < size; i++) {
                int stateType = popInt();
                int stateCode = popInt();
                String stateDesc = popString16UTF8();
                String uuid = popString16UTF8();
                long timestamp = popInt64();
                String sender = popString16UTF8();
                String receiver = popString16UTF8();
                String contentData = popString32("utf-8");
                String contentReserved = popString16();
                String extra = popString16UTF8();
                Map<String, String> kv = popMap(String.class, String.class);

                Message message = new Message();
                message.setState(Message.buildSate(stateType, stateCode, stateDesc));
                message.setSender(new User(extractUserId(sender)));
                message.setReceiver(new ChatRoom(roomId));
                message.setUuid(uuid);
                message.setTimestamp(timestamp);
                message.setContent(new Text(contentData));
                message.setAppExtra(extra);
                message.setKvExtra(kv);

                messages.add(message);
            }

            FetchingResult result = new FetchingResult().setHasMore(hasMore).setMessages(messages);

            notification = new ChatRoomNotification.OnFetchHistoryMessages(requestId, code, desc, result);
        }
    }

    private long extractUserId(String userId) {
        if (TextUtils.isEmpty(userId)) {
            return 0;
        }
        String idStr = userId.replace("user_", "");
        try {
            return Long.parseLong(idStr);
        } catch (NumberFormatException e) {
            return 0;
        }
    }

    private class NotifyOnFetchUserInfo extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnFetchUserInfo(popInt64(), popInt(), popString16UTF8(), popInt2Long(), popMap(String.class, String.class));
        }
    }

    private class NotifyOnBasicInfoChanged extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnBasicInfoChanged(popInt2Long(),
                    popInt64(),
                    popMap(String.class, String.class),
                    popInt64());
        }
    }

    private class NotifyOnFetchRoomBasicAttributesResult extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnFetchRoomBasicAttributesResult(popInt64(),
                    popInt(),
                    popString16UTF8(),
                    popInt2Long(),
                    popMap(String.class, String.class),
                    popInt64(),
                    popInt64());
        }
    }

    private class NotifyOnFetchRoomExtraAttributesResult extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnFetchRoomExtraAttributesResult(popInt64(),
                    popInt(),
                    popString16UTF8(),
                    popInt2Long(),
                    popMap(String.class, String.class),
                    popInt64(),
                    popInt64());
        }
    }

    private class NotifyOnSetRoomExtraAttributesResult extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnSetRoomExtraAttributesResult(popInt64(),
                    popInt(),
                    popString16UTF8(),
                    popInt2Long(),
                    popMap(String.class, String.class));
        }
    }

    private class NotifyOnUpdateRoomExtraAttributesResult extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnUpdateRoomExtraAttributesResult(popInt64(),
                    popInt(),
                    popString16UTF8(),
                    popInt2Long(),
                    popMap(String.class, String.class));
        }
    }

    private class NotifyOnDeleteRoomExtraAttributesResult extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);

            long requestId = popInt64();
            int code = popInt();
            String desc = popString16UTF8();
            long roomId = popInt2Long();
            int size = popInt();
            Set<String> keys = new HashSet<>();
            for (int i = 0; i < size; i++) {
                keys.add(popString16UTF8());
            }

            notification = new ChatRoomNotification.OnDeleteRoomExtraAttributesResult(requestId, code, desc, roomId, keys);
        }
    }

    private class NotifyOnClearRoomExtraAttributesResult extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnClearRoomExtraAttributesResult(popInt64(),
                    popInt(),
                    popString16UTF8(),
                    popInt2Long());
        }
    }

    private class NotifyOnRoomExtraAttributesChanged extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnRoomExtraAttributesChanged(popInt2Long(),
                    popMap(String.class, String.class),
                    popInt64(),
                    popInt64());
        }
    }

    private class NotifyOnMemberJoined extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long roomId = popInt2Long();
            int size = popInt();
            List<User> members = new ArrayList<>();
            for (int i = 0; i < size; i++) {
                members.add(new User(popInt64()));
            }
            notification = new ChatRoomNotification.OnMemberJoined(roomId, members);
        }
    }

    private class NotifyOnMemberCountChanged extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnMemberCountChanged(popInt2Long(), popInt());
        }
    }

    private class NotifyOnMemberKicked extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long roomId = popInt2Long();
            User user = new User(popInt64());
            int size = popInt();
            List<User> users = new ArrayList<>();
            for (int i = 0; i < size; i++) {
                users.add(new User(popInt64()));
            }
            String reason = popString16UTF8();
            int kickType = popInt();
            notification = new ChatRoomNotification.OnMemberKicked(roomId, user, users, reason, kickType);
        }
    }

    private class NotifyOnUserKicked extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long roomId = popInt2Long();
            User user = new User(popInt64());
            int kickOutType = popInt();
            Map<String, String> extraInfo = popMap(String.class, String.class);
            notification = new ChatRoomNotification.OnUserKicked(roomId, user, kickOutType, extraInfo);
        }
    }

    private class NotifyOnRoleAdded extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnRoleAdded(popInt2Long(), popString16UTF8(), popInt64(), popInt64());
        }
    }

    private class NotifyOnMemberMuted extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long roomId = popInt2Long();
            User operatorUser = new User(popInt64());
            int size = popInt();
            Set<User> members = new HashSet<>();
            for (int i = 0; i < size; i++) {
                members.add(new User(popInt64()));
            }
            String reason = popString16UTF8();
            notification = new ChatRoomNotification.OnMemberMuted(roomId, operatorUser, members, reason);
        }
    }

    private class NotifyOnUserInfoSet extends NotifyBase {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new ChatRoomNotification.OnUserInfoSet(popInt2Long(), popInt64(), popMap(String.class, String.class));
        }
    }


    @Override
    public void handleNotify(int type, byte[] data) {
        Log.i(TAG, "handleNotify | type: " + type);
        int notifyType = type / 100;
        if (notifyType == 4) {
            handleRoomEvent(type, data);
        } else if (notifyType == 5) {
            handleRoomMember(type, data);
        } else if (notifyType == 6) {
            handleRoomMessage(type, data);
        }
    }

    private void handleRoomMessage(int type, byte[] data) {
        try {
            if (type == ChatRoomNotification.NOTIFY_ON_FETCH_HISTORY_MESSAGES) {
                handleFetchHistoryMessagesCallback(data);
            }
        } catch (Exception e) {
            Log.e(TAG, "handleRoomMessage | type: " + type + ", err: " + e.getMessage());
        }
    }

    private void handleRoomEvent(int type, byte[] data) {
        try {
            switch (type) {
                case ChatRoomNotification.NOTIFY_ON_CREATE_CHAT_ROOM: {
                    handleCreateChatRoomCallback(data);
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_DISMISS_CHAT_ROOM: {
                    handleDismissChatRoomCallback(data);
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_FETCH_BASIC_INFO: {
                    handleFetchBasicInfoCallback(data);
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_CHANGE_BASIC_INFO: {
                    handleChangeBasicInfoCallback(data);
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_FETCH_MEMBER_COUNT: {
                    handleFetchMemberCountCallback(data);
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_CHAT_ROOM_DISMISSED: {
                    handleOnChatRoomDismissed(data);
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_BASIC_INFO_CHANGED: {
                    handleOnBasicInfoChanged(data);
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_FETCH_ROOM_BASIC_ATTRIBUTES_RESULT:
                    handleFetchRoomBasicAttributesResult(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_FETCH_ROOM_EXTRA_ATTRIBUTES_RESULT:
                    handleFetchRoomExtraAttributesResult(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_SET_ROOM_EXTRA_ATTRIBUTES_RESULT:
                    handleSetRoomExtraAttributesResult(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_UPDATE_ROOM_EXTRA_ATTRIBUTES_RESULT:
                    handleUpdateRoomExtraAttributesResult(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_DELETE_ROOM_EXTRA_ATTRIBUTES_RESULT:
                    handleDeleteRoomExtraAttributesResult(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_CLEAR_ROOM_EXTRA_ATTRIBUTES_RESULT:
                    handleClearRoomExtraAttributesResult(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_ROOM_EXTRA_ATTRIBUTES_SET:
                    handleOnRoomExtraAttributesSet(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_ROOM_EXTRA_ATTRIBUTES_UPDATED:
                    handleOnRoomExtraAttributesUpdate(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_ROOM_EXTRA_ATTRIBUTES_DELETED:
                    handleOnRoomExtraAttributesDelete(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_ROOM_EXTRA_ATTRIBUTES_CLEARED:
                    handleOnRoomExtraAttributesClear(data);
                    break;
                default:
                    break;
            }
        } catch (Exception e) {
            Log.e(TAG, "handleRoomEvent | type: " + type + ", err: " + e.getMessage());
        }
    }

    private void handleRoomMember(int type, byte[] data) {
        try {
            switch (type) {
                case ChatRoomNotification.NOTIFY_ON_JOIN_CHAT_ROOM:
                    handleJoinRoomCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_EXIT_CHAT_ROOM:
                    handleLeaveRoomCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_KICK:
                    handleKickUserCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_ADD_ROLE:
                    handleAddRoleCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_REMOVE_ROLE:
                    handleRemoveRoleCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_FETCH_MEMBERS:
                    handleFetchMembersCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_FETCH_ROLE_MEMBERS:
                    handleFetchRoleMembersCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_FETCH_USER_ROLE:
                    handleFetchUserRole(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_FETCH_ROOM_USERS_BY_ROLE:
                    handleFetchRoomUsersByRoleCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_MUTE_MEMBER:
                    handleMuteMemberCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_UN_MUTE_MEMBER:
                    handleUnMuteMemberCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_FETCH_MUTED_USERS:
                    handleFetchMutedUsersCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_IS_MUTED:
                    handleIsMutedCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_SET_USER_INFO:
                    handleSetUserInfoCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_DELETE_USER_INFO_BY_KEYS:
                    handleDeleteUserInfoByKeysCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_ADD_OR_UPDATE_USER_INFO:
                    handleAddOrUpdateUserInfoCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_FETCH_USER_INFO:
                    handleFetchUserInfoCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_BATCH_FETCH_USER_INFO:
                    handleBatchFetchUserInfoCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_FETCH_ONLINE_USER_INFO_LIST:
                    handleFetchOnlineUserInfoListCallback(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_MEMBER_JOINED: {
                    NotifyOnMemberJoined notify = new NotifyOnMemberJoined();
                    notify.unmarshall(data);
                    final ChatRoomNotification.OnMemberJoined info = (ChatRoomNotification.OnMemberJoined) notify.get();
                    HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                        @Override
                        public void visit() {
                            synchronized (mRoomMemberListeners) {
                                for (MemberListener listener : mRoomMemberListeners) {
                                    listener.onMemberJoined(info.getChatRoom(), info.getMembers());
                                }
                            }
                        }
                    });
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_ROOM_MEMBER_OFFLINE: {
                    NotifyOnMemberJoined notify = new NotifyOnMemberJoined();
                    notify.unmarshall(data);
                    final ChatRoomNotification.OnMemberJoined info = (ChatRoomNotification.OnMemberJoined) notify.get();
                    HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                        @Override
                        public void visit() {
                            synchronized (mRoomMemberListeners) {
                                for (MemberListener listener : mRoomMemberListeners) {
                                    listener.onMemberLeaved(info.getChatRoom(), info.getMembers(), 1,
                                            "断开连接");
                                    listener.onChatRoomMemberOffline(info.getChatRoom());
                                }
                            }
                        }
                    });
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_ROOM_MEMBER_LEAVED: {
                    NotifyOnMemberJoined notify = new NotifyOnMemberJoined();
                    notify.unmarshall(data);
                    final ChatRoomNotification.OnMemberJoined info = (ChatRoomNotification.OnMemberJoined) notify.get();
                    HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                        @Override
                        public void visit() {
                            synchronized (mRoomMemberListeners) {
                                for (MemberListener listener : mRoomMemberListeners) {
                                    listener.onMemberLeaved(info.getChatRoom(), info.getMembers(),
                                            0, "离开");
                                    listener.onChatRoomMemberLeave(info.getChatRoom(), info.getMembers());
                                }
                            }
                        }
                    });
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_MEMBER_COUNT_CHANGED: {
                    NotifyOnMemberCountChanged notify = new NotifyOnMemberCountChanged();
                    notify.unmarshall(data);
                    final ChatRoomNotification.OnMemberCountChanged info = (ChatRoomNotification.OnMemberCountChanged) notify.get();
                    HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                        @Override
                        public void visit() {
                            synchronized (mRoomMemberListeners) {
                                for (MemberListener listener : mRoomMemberListeners) {
                                    listener.onMemberCountChanged(info.getChatRoom(), info.getCount());
                                }
                            }
                        }
                    });
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_MEMBER_KICKED: {
                    NotifyOnMemberKicked notify = new NotifyOnMemberKicked();
                    notify.unmarshall(data);
                    final ChatRoomNotification.OnMemberKicked info = (ChatRoomNotification.OnMemberKicked) notify.get();
                    HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                        @Override
                        public void visit() {
                            synchronized (mRoomMemberListeners) {
                                for (MemberListener listener : mRoomMemberListeners) {
                                    KickOffEnum kickOffEnum = KickOffEnum.values()[info.getKickType()];
                                    listener.onMemberKicked(info.getChatRoom(), info.getUser(), info.getUsers(), new KickOff(info.getReason(), kickOffEnum));
                                    listener.onMemberKicked(info.getChatRoom(), info.getUser(),
                                            info.getUsers(), info.getReason());
                                }
                            }
                        }
                    });
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_USER_KICKED:
                    handleNotifyOnUserKicked(data);
                    break;
                case ChatRoomNotification.NOTIFY_ON_ROLE_ADDED: {
                    NotifyOnRoleAdded notify = new NotifyOnRoleAdded();
                    notify.unmarshall(data);
                    final ChatRoomNotification.OnRoleAdded info = (ChatRoomNotification.OnRoleAdded) notify.get();
                    HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                        @Override
                        public void visit() {
                            synchronized (mRoomMemberListeners) {
                                for (MemberListener listener : mRoomMemberListeners) {
                                    listener.onRoleAdded(info.getChatRoom(), info.getRole(), info.getAdmin(), info.getFellow());
                                    listener.onUserRoleSet(info.getChatRoom(), info.getRole(), info.getFellow(), info.getAdmin());
                                }
                            }
                        }
                    });
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_ROLE_REMOVED: {
                    NotifyOnRoleAdded notify = new NotifyOnRoleAdded();
                    notify.unmarshall(data);
                    final ChatRoomNotification.OnRoleAdded info = (ChatRoomNotification.OnRoleAdded) notify.get();
                    HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                        @Override
                        public void visit() {
                            synchronized (mRoomMemberListeners) {
                                for (MemberListener listener : mRoomMemberListeners) {
                                    listener.onRoleRemoved(info.getChatRoom(), info.getRole(), info.getAdmin(), info.getFellow());
                                    listener.onUserRoleDeleted(info.getChatRoom(), info.getRole(), info.getFellow(), info.getAdmin());
                                }
                            }
                        }
                    });
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_MEMBER_MUTED: {
                    NotifyOnMemberMuted notify = new NotifyOnMemberMuted();
                    notify.unmarshall(data);
                    final ChatRoomNotification.OnMemberMuted info = (ChatRoomNotification.OnMemberMuted) notify.get();
                    HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                        @Override
                        public void visit() {
                            synchronized (mRoomMemberListeners) {
                                for (MemberListener listener : mRoomMemberListeners) {
                                    listener.onMemberMuted(info.getChatRoom(), info.getOperatorUser(), info.getMembers(), info.getReason());
                                }
                            }
                        }
                    });
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_MEMBER_UN_MUTED: {
                    NotifyOnMemberMuted notify = new NotifyOnMemberMuted();
                    notify.unmarshall(data);
                    final ChatRoomNotification.OnMemberMuted info = (ChatRoomNotification.OnMemberMuted) notify.get();
                    HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                        @Override
                        public void visit() {
                            synchronized (mRoomMemberListeners) {
                                for (MemberListener listener : mRoomMemberListeners) {
                                    listener.onMemberUnmuted(info.getChatRoom(), info.getOperatorUser(), info.getMembers(), info.getReason());
                                }
                            }
                        }
                    });
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_USER_INFO_SET: {
                    NotifyOnUserInfoSet notify = new NotifyOnUserInfoSet();
                    notify.unmarshall(data);
                    final ChatRoomNotification.OnUserInfoSet info = (ChatRoomNotification.OnUserInfoSet) notify.get();
                    HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                        @Override
                        public void visit() {
                            synchronized (mRoomMemberListeners) {
                                for (MemberListener listener : mRoomMemberListeners) {
                                    listener.onUserInfoSet(info.getChatRoom(), info.getUser(), info.getInfoMap());
                                }
                            }
                        }
                    });
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_USER_INFO_DELETED: {
                    NotifyOnUserInfoSet notify = new NotifyOnUserInfoSet();
                    notify.unmarshall(data);
                    final ChatRoomNotification.OnUserInfoSet info = (ChatRoomNotification.OnUserInfoSet) notify.get();
                    HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                        @Override
                        public void visit() {
                            synchronized (mRoomMemberListeners) {
                                for (MemberListener listener : mRoomMemberListeners) {
                                    listener.onUserInfoDeleted(info.getChatRoom(), info.getUser(), info.getInfoMap());
                                }
                            }
                        }
                    });
                    break;
                }
                case ChatRoomNotification.NOTIFY_ON_USER_INFO_ADD_OR_UPDATED: {
                    NotifyOnUserInfoSet notify = new NotifyOnUserInfoSet();
                    notify.unmarshall(data);
                    final ChatRoomNotification.OnUserInfoSet info = (ChatRoomNotification.OnUserInfoSet) notify.get();
                    HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                        @Override
                        public void visit() {
                            synchronized (mRoomMemberListeners) {
                                for (MemberListener listener : mRoomMemberListeners) {
                                    listener.onUserInfoAddedOrUpdated(info.getChatRoom(), info.getUser(), info.getInfoMap());
                                }
                            }
                        }
                    });
                    break;
                }
                default:
                    break;
            }
        } catch (Exception e) {
            Log.e(TAG, "handleRoomMember | type: " + type + ", err: " + e.getMessage());
        }
    }

    private void reportReturnCode(String functionName, long requestId, int code) {
        RequestId request = mRequestIdMap.remove(requestId);
        if (request != null) {
            HMRContext.reportReturnCode(functionName, request.getCurrentTimeMillis(), code);
        }
    }

    private void handleCreateChatRoomCallback(byte[] data) {
        NotifyOnCreateChatRoom notify = new NotifyOnCreateChatRoom();
        notify.unmarshall(data);
        final ChatRoomNotification.OnCreateChatRoom callbackInfo = (ChatRoomNotification.OnCreateChatRoom) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletionArg<ChatRoom> completion = mCreateChatRoomCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getChatRoom(), callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.createChatRoom, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleDismissChatRoomCallback(byte[] data) {
        NotifyBaseCallback notify = new NotifyBaseCallback();
        notify.unmarshall(data);
        final ChatRoomNotification.BaseCallbackInfo callbackInfo = (ChatRoomNotification.BaseCallbackInfo) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = mDismissChatRoomCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.dismissChatRoom, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleJoinRoomCallback(byte[] data) {
        NotifyOnJoinChatRoom notify = new NotifyOnJoinChatRoom();
        notify.unmarshall(data);
        final ChatRoomNotification.OnJoinRoom callbackInfo =
                (ChatRoomNotification.OnJoinRoom) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = mJoinRoomCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.join, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleLeaveRoomCallback(byte[] data) {
        NotifyOnLeaveChatRoom notify = new NotifyOnLeaveChatRoom();
        notify.unmarshall(data);
        final ChatRoomNotification.OnLeaveRoom callbackInfo =
                (ChatRoomNotification.OnLeaveRoom) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = mLeaveRoomCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.leave, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleKickUserCallback(byte[] data) {
        NotifyBaseCallback notify = new NotifyBaseCallback();
        notify.unmarshall(data);
        final ChatRoomNotification.BaseCallbackInfo callbackInfo = (ChatRoomNotification.BaseCallbackInfo) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = mKickCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());
    }

    private void handleAddRoleCallback(byte[] data) {
        NotifyBaseCallback notify = new NotifyBaseCallback();
        notify.unmarshall(data);
        final ChatRoomNotification.BaseCallbackInfo callbackInfo = (ChatRoomNotification.BaseCallbackInfo) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = CHAT_ROOM_COMPLETIONS.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());
    }

    private void handleRemoveRoleCallback(byte[] data) {
        NotifyBaseCallback notify = new NotifyBaseCallback();
        notify.unmarshall(data);
        final ChatRoomNotification.BaseCallbackInfo callbackInfo = (ChatRoomNotification.BaseCallbackInfo) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = CHAT_ROOM_COMPLETIONS.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());
    }

    private void handleFetchMembersCallback(byte[] data) {
        NotifyOnFetchMembers notify = new NotifyOnFetchMembers();
        notify.unmarshall(data);
        final ChatRoomNotification.OnFetchMembers callbackInfo = (ChatRoomNotification.OnFetchMembers) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletionArg<List<User>> completion = mFetchMembersCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getUsers(), callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.fetchMembers, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleFetchUserRole(byte[] data) {
        NotifyOnFetchUserRole notify = new NotifyOnFetchUserRole();
        notify.unmarshall(data);
        final ChatRoomNotification.OnFetchUserRole callback = (ChatRoomNotification.OnFetchUserRole) notify.get();
        RequestId requestId = new RequestId(callback.getRequestId());
        HMRCompletionArg<String> completion = FETCH_USER_ROLE_COMPLETIONS.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callback.getRole(), callback.getCode(), callback.getDesc());

        /// report上报
        reportReturnCode(ReportFunction.FETCH_USER_ROLE, callback.requestId, callback.getCode());
    }

    private void handleFetchRoleMembersCallback(byte[] data) {
        handleFetchRoleUsers(data, ReportFunction.fetchRoleMembers);
    }

    private void handleFetchRoomUsersByRoleCallback(byte[] data) {
        handleFetchRoleUsers(data, ReportFunction.FETCH_ROOM_USERS_BY_ROLE);
    }

    private void handleFetchRoleUsers(byte[] data, String functionName) {
        NotifyOnFetchRoleMembers notify = new NotifyOnFetchRoleMembers();
        notify.unmarshall(data);
        final ChatRoomNotification.OnFetchRoleMembers callbackInfo = (ChatRoomNotification.OnFetchRoleMembers) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletionArg<Map<String, List<User>>> completion = mFetchRoleMembersCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getUsers(), callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(functionName, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleFetchMemberCountCallback(byte[] data) {
        NotifyOnFetchMemberCount notify = new NotifyOnFetchMemberCount();
        notify.unmarshall(data);
        final ChatRoomNotification.OnFetchMemberCount callbackInfo = (ChatRoomNotification.OnFetchMemberCount) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletionArg<Integer> completion = mFetchMemberCountCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCount(), callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.fetchMemberCount, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleFetchBasicInfoCallback(byte[] data) {
        NotifyOnRoomProps notify = new NotifyOnRoomProps();
        notify.unmarshall(data);
        final ChatRoomNotification.OnRoomProps callbackInfo = (ChatRoomNotification.OnRoomProps) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletionArg<ChatRoomInfo> completion = mFetchBasicInfoCompletions.remove(requestId);
        ChatRoomInfo chatRoomInfo = new ChatRoomInfo("", "", "", "");
        if (callbackInfo.getProps() != null) {
            chatRoomInfo = new ChatRoomInfo(callbackInfo.getProps().get("Name"),
                    callbackInfo.getProps().get("Description"),
                    callbackInfo.getProps().get("Bulletin"),
                    callbackInfo.getProps().get("AppExtra"));
        }
        HummerDispatch.dispatchCompletion(completion, chatRoomInfo, callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.fetchBasicInfo, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleChangeBasicInfoCallback(byte[] data) {
        NotifyOnRoomProps notify = new NotifyOnRoomProps();
        notify.unmarshall(data);
        final ChatRoomNotification.OnRoomProps callbackInfo = (ChatRoomNotification.OnRoomProps) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = mChangeBasicInfoCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());
    }

    private void handleMuteMemberCallback(byte[] data) {
        NotifyBaseCallback notify = new NotifyBaseCallback();
        notify.unmarshall(data);
        final ChatRoomNotification.BaseCallbackInfo callbackInfo = (ChatRoomNotification.BaseCallbackInfo) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = mMuteMemberCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.muteMember, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleUnMuteMemberCallback(byte[] data) {
        NotifyBaseCallback notify = new NotifyBaseCallback();
        notify.unmarshall(data);
        final ChatRoomNotification.BaseCallbackInfo callbackInfo = (ChatRoomNotification.BaseCallbackInfo) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = mUnMuteMemberCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.unmuteMember, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleFetchMutedUsersCallback(byte[] data) {
        NotifyOnFetchMutedUsers notify = new NotifyOnFetchMutedUsers();
        notify.unmarshall(data);
        final ChatRoomNotification.OnFetchMutedUsers callbackInfo = (ChatRoomNotification.OnFetchMutedUsers) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletionArg<Set<User>> completion = mFetchMuteUsersCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getUsers(), callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.fetchMutedUsers, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleIsMutedCallback(byte[] data) {
        NotifyOnIsMuted notify = new NotifyOnIsMuted();
        notify.unmarshall(data);
        final ChatRoomNotification.OnIsMuted callbackInfo = (ChatRoomNotification.OnIsMuted) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletionArg<Boolean> completion = mIsMutedCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getMuted(), callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.isMuted, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleSetUserInfoCallback(byte[] data) {
        NotifyBaseCallback notify = new NotifyBaseCallback();
        notify.unmarshall(data);
        final ChatRoomNotification.BaseCallbackInfo callbackInfo = (ChatRoomNotification.BaseCallbackInfo) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = mSetUserInfoCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.setUserInfo, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleDeleteUserInfoByKeysCallback(byte[] data) {
        NotifyBaseCallback notify = new NotifyBaseCallback();
        notify.unmarshall(data);
        final ChatRoomNotification.BaseCallbackInfo callbackInfo = (ChatRoomNotification.BaseCallbackInfo) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = mDeleteUserInfoByKeysCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.deleteUserInfoByKeys, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleAddOrUpdateUserInfoCallback(byte[] data) {
        NotifyBaseCallback notify = new NotifyBaseCallback();
        notify.unmarshall(data);
        final ChatRoomNotification.BaseCallbackInfo callbackInfo = (ChatRoomNotification.BaseCallbackInfo) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = mAddOrUpdateUserInfoCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.addOrUpdateUserInfo, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleFetchUserInfoCallback(byte[] data) {
        NotifyOnFetchUserInfo notify = new NotifyOnFetchUserInfo();
        notify.unmarshall(data);
        final ChatRoomNotification.OnFetchUserInfo callbackInfo = (ChatRoomNotification.OnFetchUserInfo) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletionArg<Map<String, String>> completion = mFetchUserInfoCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getUserInfoMap(), callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.fetchUserInfo, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleBatchFetchUserInfoCallback(byte[] data) {
        NotifyOnUserInfoList notify = new NotifyOnUserInfoList();
        notify.unmarshall(data);
        final ChatRoomNotification.OnUserInfoList callbackInfo = (ChatRoomNotification.OnUserInfoList) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletionArg<Map<User, Map<String, String>>> completion = mBathFetchUserInfosCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getUserInfoMap(), callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.batchFetchUserInfos, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleFetchOnlineUserInfoListCallback(byte[] data) {
        NotifyOnUserInfoList notify = new NotifyOnUserInfoList();
        notify.unmarshall(data);
        final ChatRoomNotification.OnUserInfoList callbackInfo = (ChatRoomNotification.OnUserInfoList) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletionArg<Map<User, Map<String, String>>> completion = mFetchOnlineUserInfoListCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getUserInfoMap(), callbackInfo.getCode(), callbackInfo.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.fetchOnlineUserInfoList, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleFetchHistoryMessagesCallback(byte[] data) {
        NotifyOnFetchHistoryMessages notify = new NotifyOnFetchHistoryMessages();
        notify.unmarshall(data);
        final ChatRoomNotification.OnFetchHistoryMessages callbackInfo = (ChatRoomNotification.OnFetchHistoryMessages) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletionArg<FetchingResult> completion = mFetchHistoryMessagesCompletions.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getFetchingResult(), callbackInfo.getCode(), callbackInfo.getDesc());
        // report 上报
        reportReturnCode(ReportFunction.FETCH_HISTORY_MESSAGES, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleOnBasicInfoChanged(byte[] data) {
        NotifyOnBasicInfoChanged notify = new NotifyOnBasicInfoChanged();
        notify.unmarshall(data);
        final ChatRoomNotification.OnBasicInfoChanged info = (ChatRoomNotification.OnBasicInfoChanged) notify.get();
        final Map<ChatRoomInfo.BasicInfoType, String> propInfo = new HashMap<>();
        Map<String, String> props = info.getProps();
        if (props != null) {
            if (props.containsKey("Name")) {
                propInfo.put(ChatRoomInfo.BasicInfoType.Name, props.get("Name"));
            }
            if (props.containsKey("Description")) {
                propInfo.put(ChatRoomInfo.BasicInfoType.Description, props.get("Description"));
            }
            if (props.containsKey("Bulletin")) {
                propInfo.put(ChatRoomInfo.BasicInfoType.Bulletin, props.get("Bulletin"));
            }
            if (props.containsKey("AppExtra")) {
                propInfo.put(ChatRoomInfo.BasicInfoType.AppExtra, props.get("AppExtra"));
            }
        }
        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (mRoomListeners) {
                    for (ChatRoomListener listener : mRoomListeners) {
                        listener.onBasicInfoChanged(info.getChatRoom(), propInfo);
                        listener.onRoomBasicAttributesUpdated(info.getChatRoom(),
                                info.getProps(),
                                info.getUser(),
                                info.getLatestUpdateTs());
                    }
                }
            }
        });
    }

    private void handleOnChatRoomDismissed(byte[] data) {
        NotifyOnBasicInfoChanged notify = new NotifyOnBasicInfoChanged();
        notify.unmarshall(data);
        final ChatRoomNotification.OnBasicInfoChanged info = (ChatRoomNotification.OnBasicInfoChanged) notify.get();
        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (mRoomListeners) {
                    for (ChatRoomService.ChatRoomListener listener : mRoomListeners) {
                        listener.onChatRoomDismissed(info.getChatRoom(), info.getUser());
                    }
                }
            }
        });
    }

    private void handleFetchRoomBasicAttributesResult(byte[] data) {
        NotifyOnFetchRoomBasicAttributesResult notify = new NotifyOnFetchRoomBasicAttributesResult();
        notify.unmarshall(data);
        final ChatRoomNotification.OnFetchRoomBasicAttributesResult callbackInfo = (ChatRoomNotification.OnFetchRoomBasicAttributesResult) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletionArg<FetchRoomBasicAttributesResult> completion = FETCH_ROOM_BASIC_ATTRIBUTES_COMPLETIONS.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getResult(), callbackInfo.getCode(), callbackInfo.getDesc());

        // report 上报
        reportReturnCode(ReportFunction.FETCH_ROOM_BASIC_ATTRIBUTES, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleFetchRoomExtraAttributesResult(byte[] data) {
        NotifyOnFetchRoomExtraAttributesResult notify = new NotifyOnFetchRoomExtraAttributesResult();
        notify.unmarshall(data);
        final ChatRoomNotification.OnFetchRoomExtraAttributesResult callbackInfo = (ChatRoomNotification.OnFetchRoomExtraAttributesResult) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletionArg<ChatRoomExtraAttribute> completion = FETCH_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getChatRoomExtraAttribute(), callbackInfo.getCode(), callbackInfo.getDesc());

        // report 上报
        reportReturnCode(ReportFunction.FETCH_ROOM_EXTRA_ATTRIBUTES, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleSetRoomExtraAttributesResult(byte[] data) {
        NotifyOnSetRoomExtraAttributesResult notify = new NotifyOnSetRoomExtraAttributesResult();
        notify.unmarshall(data);
        final ChatRoomNotification.OnSetRoomExtraAttributesResult callbackInfo = (ChatRoomNotification.OnSetRoomExtraAttributesResult) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = SET_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());

        // report 上报
        reportReturnCode(ReportFunction.SET_ROOM_EXTRA_ATTRIBUTES, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleUpdateRoomExtraAttributesResult(byte[] data) {
        NotifyOnUpdateRoomExtraAttributesResult notify = new NotifyOnUpdateRoomExtraAttributesResult();
        notify.unmarshall(data);
        final ChatRoomNotification.OnUpdateRoomExtraAttributesResult callbackInfo = (ChatRoomNotification.OnUpdateRoomExtraAttributesResult) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = UPDATE_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());

        // report 上报
        reportReturnCode(ReportFunction.UPDATE_ROOM_EXTRA_ATTRIBUTES, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleDeleteRoomExtraAttributesResult(byte[] data) {
        NotifyOnDeleteRoomExtraAttributesResult notify = new NotifyOnDeleteRoomExtraAttributesResult();
        notify.unmarshall(data);
        final ChatRoomNotification.OnDeleteRoomExtraAttributesResult callbackInfo = (ChatRoomNotification.OnDeleteRoomExtraAttributesResult) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = DELETE_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());

        // report 上报
        reportReturnCode(ReportFunction.DELETE_ROOM_EXTRA_ATTRIBUTES, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleClearRoomExtraAttributesResult(byte[] data) {
        NotifyOnClearRoomExtraAttributesResult notify = new NotifyOnClearRoomExtraAttributesResult();
        notify.unmarshall(data);
        final ChatRoomNotification.OnClearRoomExtraAttributesResult callbackInfo = (ChatRoomNotification.OnClearRoomExtraAttributesResult) notify.get();
        RequestId requestId = new RequestId(callbackInfo.getRequestId());
        HMRCompletion completion = CLEAR_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callbackInfo.getCode(), callbackInfo.getDesc());

        // report 上报
        reportReturnCode(ReportFunction.CLEAR_ROOM_EXTRA_ATTRIBUTES, callbackInfo.requestId, callbackInfo.getCode());
    }

    private void handleOnRoomExtraAttributesSet(byte[] data) {
        NotifyOnRoomExtraAttributesChanged notify = new NotifyOnRoomExtraAttributesChanged();
        notify.unmarshall(data);
        final ChatRoomNotification.OnRoomExtraAttributesChanged changed = (ChatRoomNotification.OnRoomExtraAttributesChanged) notify.get();
        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (mRoomListeners) {
                    for (ChatRoomService.ChatRoomListener listener : mRoomListeners) {
                        listener.onRoomExtraAttributesSet(changed.getChatRoom(), changed.getAttributes(), changed.getUser(), changed.getLatestUpdateTs());
                    }
                }
            }
        });
    }

    private void handleOnRoomExtraAttributesUpdate(byte[] data) {
        NotifyOnRoomExtraAttributesChanged notify = new NotifyOnRoomExtraAttributesChanged();
        notify.unmarshall(data);
        final ChatRoomNotification.OnRoomExtraAttributesChanged changed = (ChatRoomNotification.OnRoomExtraAttributesChanged) notify.get();
        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (mRoomListeners) {
                    for (ChatRoomService.ChatRoomListener listener : mRoomListeners) {
                        listener.onRoomExtraAttributesUpdated(changed.getChatRoom(), changed.getAttributes(), changed.getUser(), changed.getLatestUpdateTs());
                    }
                }
            }
        });
    }

    private void handleOnRoomExtraAttributesDelete(byte[] data) {
        NotifyOnRoomExtraAttributesChanged notify = new NotifyOnRoomExtraAttributesChanged();
        notify.unmarshall(data);
        final ChatRoomNotification.OnRoomExtraAttributesChanged changed = (ChatRoomNotification.OnRoomExtraAttributesChanged) notify.get();
        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (mRoomListeners) {
                    for (ChatRoomService.ChatRoomListener listener : mRoomListeners) {
                        listener.onRoomExtraAttributesDeleted(changed.getChatRoom(), changed.getAttributes(), changed.getUser(), changed.getLatestUpdateTs());
                    }
                }
            }
        });
    }

    private void handleOnRoomExtraAttributesClear(byte[] data) {
        NotifyOnRoomExtraAttributesChanged notify = new NotifyOnRoomExtraAttributesChanged();
        notify.unmarshall(data);
        final ChatRoomNotification.OnRoomExtraAttributesChanged changed = (ChatRoomNotification.OnRoomExtraAttributesChanged) notify.get();
        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (mRoomListeners) {
                    for (ChatRoomService.ChatRoomListener listener : mRoomListeners) {
                        listener.onRoomExtraAttributesCleared(changed.getChatRoom(), changed.getAttributes(), changed.getUser(), changed.getLatestUpdateTs());
                    }
                }
            }
        });
    }

    private void handleNotifyOnUserKicked(byte[] data) {
        NotifyOnUserKicked notify = new NotifyOnUserKicked();
        notify.unmarshall(data);
        final ChatRoomNotification.OnUserKicked info = (ChatRoomNotification.OnUserKicked) notify.get();
        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (mRoomMemberListeners) {
                    for (MemberListener listener : mRoomMemberListeners) {
                        listener.onUserKicked(info.getChatRoom(), info.getUser(), info.getKickOutType(), info.getExtraInfo());
                    }
                }
            }
        });
    }

    /********************************  Notify End *****************************************/

    private RequestId newRequestId() {
        RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        mRequestIdMap.put(requestId.getId(), requestId);
        return requestId;
    }

    @Override
    public void initService() {
        HummerNative.registerNotificationListener(this);
        ChatServiceImpl.registerSendingExtension(new ChatServiceImpl.SendingExtension() {
            @Override
            public ChatServiceImpl.MessagePacket makeMessagePacket(Message message) {
                ChatServiceImpl.MessagePacket packet = null;
                if (message.getContent() instanceof Signal) {
                    Signal signal = (Signal) message.getContent();
                    packet = new ChatServiceImpl.MessagePacket();
                    packet.setUuid(message.getUuid());
                    packet.setData(signal.content);
                    packet.setExtra(message.getAppExtra());
                    packet.setKvExtra(message.getKvExtra());
                    if (signal.user != null) {
                        packet.setMsgType(ChatServiceImpl.MessagePacket.MSG_TYPE_UNICAST);
                        packet.setUserId(signal.user.getId());
                    } else {
                        packet.setMsgType(ChatServiceImpl.MessagePacket.MSG_TYPE_BROADCAST);
                    }
                    packet.setRoomId(message.getReceiver().getId());
                } else if (message.getContent() instanceof Text) {
                    Text text = (Text) message.getContent();
                    packet = new ChatServiceImpl.MessagePacket();
                    packet.setUuid(message.getUuid());
                    packet.setData(text.getText());
                    packet.setExtra(message.getAppExtra());
                    packet.setKvExtra(message.getKvExtra());
                    packet.setRoomId(message.getReceiver().getId());
                    packet.setMsgType(ChatServiceImpl.MessagePacket.MSG_TYPE_TEXT);
                }
                return packet;
            }

            @Override
            public boolean parseMessage(Message message, int msgType, long userId, long roomId, String data) {
                if (msgType == ChatServiceImpl.MessagePacket.MSG_TYPE_TEXT) {
                    message.setContent(new Text(data));
                    message.setReceiver(new ChatRoom(roomId));
                    return true;
                } else if (msgType == ChatServiceImpl.MessagePacket.MSG_TYPE_BROADCAST) {
                    message.setContent(Signal.broadcast(data));
                    message.setReceiver(new ChatRoom(roomId));
                    return true;
                } else if (msgType == ChatServiceImpl.MessagePacket.MSG_TYPE_UNICAST) {
                    message.setContent(Signal.unicast(new User(userId), data));
                    message.setReceiver(new ChatRoom(roomId));
                    return true;
                }
                return false;
            }
        });
    }

    @Override
    public void openService(@NonNull HMRCompletion completion) {
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public int serviceSort() {
        return 0;
    }

    @Override
    public void closeService() {

    }

    @Override
    public void setRegion(String region) {
        ChatRoomNative.setRegion(region);
        HMRContext.reportReturnCode(ReportFunction.setRegion, HMRContext.getCurrentTime());
    }

    @Override
    public void createChatRoom(@NonNull ChatRoomInfo chatRoomInfo, @NonNull HMR.CompletionArg<ChatRoom> completion) {
        final RequestId requestId = newRequestId();
        if (completion != null) {
            HMRCompletionArg<ChatRoom> c = new HMRCompletionArg<>(requestId, completion);
            mCreateChatRoomCompletions.put(requestId, c);
        }
        ChatRoomNative.createRoom(requestId.getId(), chatRoomInfo);
    }

    @Override
    public void dismissChatRoom(@NonNull ChatRoom chatRoom, @Nullable HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = new HMRCompletion(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mDismissChatRoomCompletions.put(requestId, c);
        }
        ChatRoomNative.dismissRoom(requestId.getId(), chatRoom);
    }

    @Override
    public void join(@NonNull ChatRoom chatRoom, @NonNull Map<String, String> joinProps, @NonNull final Challenges.JoiningCompletion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = new HMRCompletion(requestId, new HMR.Completion() {
            @Override
            public void onSuccess() {
                completion.onSucceed();
            }

            @Override
            public void onFailed(Error err) {
                completion.onFailure(err);
            }
        });

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mJoinRoomCompletions.put(requestId, c);
        }
        ChatRoomNative.joinRoom(requestId.getId(), chatRoom, joinProps);
    }

    @Override
    public void leave(@NonNull ChatRoom chatRoom, @Nullable HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = new HMRCompletion(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mLeaveRoomCompletions.put(requestId, c);
        }
        ChatRoomNative.exitRoom(requestId.getId(), chatRoom);
    }

    @Override
    public void kick(@NonNull ChatRoom chatRoom, @NonNull User member,
                     @Nullable Map<EKickInfo, String> extraInfo,
                     @NonNull HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = createHMRCompletion(requestId, completion, ReportFunction.kick);

        // uid 不能小于0
        if (member == null || member.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, INVALID_UID));
            return;
        }
        /*
         * room id 的类型是 uint32(无符号32位整型, [0, 4294967295])，而java使用基础类型long来表示roomId
         * 显然是超出uint32的范围的，所以需要保证业务输入的roomId在uint32的范围内
         * sdk 判断范围，不在范围则报错
         */
        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mKickCompletions.put(requestId, c);
        }
        ChatRoomNative.kick(requestId.getId(), chatRoom, member, extraInfo);
    }

    @Override
    public void kickUser(@NonNull ChatRoom chatRoom,
                         @NonNull User user,
                         Map<String, String> extraInfo,
                         KickUserOptions options,
                         HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = createHMRCompletion(requestId, completion, ReportFunction.KICK_USER);

        /*
         * room id 的类型是 uint32(无符号32位整型, [1, 4294967294])，而java使用基础类型long来表示roomId
         * 显然是超出uint32的范围的，所以需要保证业务输入的roomId在uint32的范围内
         * sdk 判断范围，不在范围则报错
         */
        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        // uid 不能小于0
        if (user == null || user.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, INVALID_OPERATED_UID));
            return;
        }

        if (completion != null) {
            mKickCompletions.put(requestId, c);
        }
        ChatRoomNative.kickUser(requestId.getId(), chatRoom, user, extraInfo, options);
    }

    @Override
    public void setUserRole(@NonNull ChatRoom chatRoom,
                            @NonNull User user,
                            @NonNull String role,
                            RoleOptions options,
                            final HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = createHMRCompletion(requestId, completion, ReportFunction.SET_USER_ROLE);

        /*
         * room id 的类型是 uint32(无符号32位整型, [1, 4294967294])，而java使用基础类型long来表示roomId
         * 显然是超出uint32的范围的，所以需要保证业务输入的roomId在uint32的范围内
         * sdk 判断范围，不在范围则报错
         */
        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        // uid 不能小于0
        if (user == null || user.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, INVALID_OPERATED_UID));
            return;
        }

        if (completion != null) {
            CHAT_ROOM_COMPLETIONS.put(requestId, c);
        }
        ChatRoomNative.setUserRole(requestId.getId(), chatRoom, user, role);
    }

    @Override
    public void deleteUserRole(@NonNull ChatRoom chatRoom,
                               @NonNull User user,
                               RoleOptions options,
                               HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = createHMRCompletion(requestId, completion, ReportFunction.DELETE_USER_ROLE);

        /*
         * room id 的类型是 uint32(无符号32位整型, [1, 4294967294])，而java使用基础类型long来表示roomId
         * 显然是超出uint32的范围的，所以需要保证业务输入的roomId在uint32的范围内
         * sdk 判断范围，不在范围则报错
         */
        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        // uid 不能小于0
        if (user == null || user.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, INVALID_OPERATED_UID));
            return;
        }

        if (completion != null) {
            CHAT_ROOM_COMPLETIONS.put(requestId, c);
        }

        ChatRoomNative.deleteUserRole(requestId.getId(), chatRoom, user);
    }

    @Override
    public void fetchUserRole(@NonNull ChatRoom chatRoom,
                              @NonNull User user,
                              HMR.CompletionArg<String> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<String> c = new HMRCompletionArg<>(requestId, completion);

        /*
         * room id 的类型是 uint32(无符号32位整型, [1, 4294967294])，而java使用基础类型long来表示roomId
         * 显然是超出uint32的范围的，所以需要保证业务输入的roomId在uint32的范围内
         * sdk 判断范围，不在范围则报错
         */
        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        // uid 不能小于0
        if (user == null || user.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, INVALID_OPERATED_UID));
            return;
        }

        if (completion != null) {
            FETCH_USER_ROLE_COMPLETIONS.put(requestId, c);
        }
        ChatRoomNative.fetchUserRole(requestId.getId(), chatRoom, user);
    }

    @Override
    public void fetchRoomUsersByRole(@NonNull ChatRoom chatRoom,
                                     String role,
                                     HMR.CompletionArg<Map<String, List<User>>> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<Map<String, List<User>>> c = new HMRCompletionArg<>(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mFetchRoleMembersCompletions.put(requestId, c);
        }
        ChatRoomNative.fetchRoomUsersByRole(requestId.getId(), chatRoom, role);
    }

    @Override
    public void addRole(@NonNull ChatRoom chatRoom, @NonNull User member, @NonNull String role,
                        @NonNull HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = createHMRCompletion(requestId, completion, ReportFunction.addRole);

        // uid 不能小于0
        if (member == null || member.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, INVALID_OPERATED_UID));
            return;
        }
        /*
         * room id 的类型是 uint32(无符号32位整型, [0, 4294967295])，而java使用基础类型long来表示roomId
         * 显然是超出uint32的范围的，所以需要保证业务输入的roomId在uint32的范围内
         * sdk 判断范围，不在范围则报错
         */
        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            CHAT_ROOM_COMPLETIONS.put(requestId, c);
        }
        ChatRoomNative.addRole(requestId.getId(), chatRoom, member, role);
    }

    @Override
    public void removeRole(@NonNull ChatRoom chatRoom, @NonNull User member, @NonNull String role,
                           @NonNull HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = createHMRCompletion(requestId, completion, ReportFunction.removeRole);

        // uid 不能小于0
        if (member == null || member.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, INVALID_OPERATED_UID));
            return;
        }
        /*
         * room id 的类型是 uint32(无符号32位整型, [0, 4294967295])，而java使用基础类型long来表示roomId
         * 显然是超出uint32的范围的，所以需要保证业务输入的roomId在uint32的范围内
         * sdk 判断范围，不在范围则报错
         */
        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            CHAT_ROOM_COMPLETIONS.put(requestId, c);
        }
        ChatRoomNative.removeRole(requestId.getId(), chatRoom, member, role);
    }

    @Override
    public void fetchMembers(@NonNull ChatRoom chatRoom, int num, int offset, @NonNull HMR.CompletionArg<List<User>> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<List<User>> c = new HMRCompletionArg<>(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mFetchMembersCompletions.put(requestId, c);
        }
        ChatRoomNative.fetchMembers(requestId.getId(), chatRoom, num, offset);
    }

    @Override
    public void fetchRoleMembers(@NonNull ChatRoom chatRoom, boolean online, @NonNull HMR.CompletionArg<Map<String, List<User>>> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<Map<String, List<User>>> c = new HMRCompletionArg<>(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mFetchRoleMembersCompletions.put(requestId, c);
        }
        ChatRoomNative.fetchRoleMembers(requestId.getId(), chatRoom, online);
    }

    @Override
    public void fetchMemberCount(@NonNull ChatRoom chatRoom, @NonNull HMR.CompletionArg<Integer> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<Integer> c = new HMRCompletionArg<>(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mFetchMemberCountCompletions.put(requestId, c);
        }
        ChatRoomNative.fetchMemberCount(requestId.getId(), chatRoom);
    }

    @Override
    public void fetchBasicInfo(@NonNull ChatRoom chatRoom, @NonNull HMR.CompletionArg<ChatRoomInfo> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<ChatRoomInfo> c = new HMRCompletionArg<>(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mFetchBasicInfoCompletions.put(requestId, c);
        }
        ChatRoomNative.fetchBasicInfo(requestId.getId(), chatRoom);
    }

    @Override
    public void fetchRoomBasicAttributes(@NonNull ChatRoom chatRoom,
                                         Set<String> keys,
                                         HMR.CompletionArg<FetchRoomBasicAttributesResult> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<FetchRoomBasicAttributesResult> c = new HMRCompletionArg<>(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            FETCH_ROOM_BASIC_ATTRIBUTES_COMPLETIONS.put(requestId, c);
        }
        ChatRoomNative.fetchRoomBasicAttributes(requestId.getId(), chatRoom, keys);
    }

    @Override
    public void changeBasicInfo(@NonNull ChatRoom chatRoom, @NonNull Map<ChatRoomInfo.BasicInfoType, String> propInfo, @NonNull HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = createHMRCompletion(requestId, completion, ReportFunction.changeBasicInfo);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mChangeBasicInfoCompletions.put(requestId, c);
        }
        ChatRoomNative.changeBasicInfo(requestId.getId(), chatRoom, propInfo);
    }

    @Override
    public void updateRoomBasicAttributes(@NonNull ChatRoom chatRoom,
                                          @NonNull Map<String, String> attributes,
                                          RoomBasicAttributesOptions options,
                                          HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = createHMRCompletion(requestId, completion, ReportFunction.UPDATE_ROOM_BASIC_ATTRIBUTES);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mChangeBasicInfoCompletions.put(requestId, c);
        }
        ChatRoomNative.updateRoomBasicAttributes(requestId.getId(), chatRoom, attributes, options);
    }

    @Override
    public void fetchRoomExtraAttributes(@NonNull ChatRoom chatRoom,
                                         @Nullable Set<String> keys,
                                         @Nullable HMR.CompletionArg<ChatRoomExtraAttribute> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<ChatRoomExtraAttribute> c = new HMRCompletionArg<>(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            FETCH_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS.put(requestId, c);
        }
        ChatRoomNative.fetchRoomExtraAttributes(requestId.getId(), chatRoom, keys);
    }

    @Override
    public void setRoomExtraAttributes(@NonNull ChatRoom chatRoom,
                                       @NonNull Map<String, String> attributes,
                                       @Nullable ChatRoomAttributeOptions options,
                                       @Nullable HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = new HMRCompletion(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            SET_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS.put(requestId, c);
        }
        ChatRoomNative.setRoomExtraAttributes(requestId.getId(), chatRoom, attributes, options);
    }

    @Override
    public void updateRoomExtraAttributes(@NonNull ChatRoom chatRoom,
                                          @NonNull Map<String, String> attributes,
                                          @Nullable ChatRoomAttributeOptions options,
                                          @Nullable HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = new HMRCompletion(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            UPDATE_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS.put(requestId, c);
        }
        ChatRoomNative.updateRoomExtraAttributes(requestId.getId(), chatRoom, attributes, options);
    }

    @Override
    public void deleteRoomExtraAttributes(@NonNull ChatRoom chatRoom,
                                          @NonNull Set<String> keys,
                                          @Nullable ChatRoomAttributeOptions options,
                                          @Nullable HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = new HMRCompletion(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            DELETE_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS.put(requestId, c);
        }
        ChatRoomNative.deleteRoomExtraAttributes(requestId.getId(), chatRoom, keys, options);
    }

    @Override
    public void clearRoomExtraAttributes(@NonNull ChatRoom chatRoom,
                                         @Nullable ChatRoomAttributeOptions options,
                                         @Nullable HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = new HMRCompletion(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            CLEAR_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS.put(requestId, c);
        }
        ChatRoomNative.clearRoomExtraAttributes(requestId.getId(), chatRoom, options);
    }

    @Override
    public void muteMember(@NonNull ChatRoom chatRoom, @NonNull User member, @Nullable String reason, @NonNull HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = new HMRCompletion(requestId, completion);

        if (member == null || member.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, INVALID_OPERATED_UID));
            return;
        }

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mMuteMemberCompletions.put(requestId, c);
        }
        ChatRoomNative.muteMember(requestId.getId(), chatRoom, member, reason);
    }

    @Override
    public void unmuteMember(@NonNull ChatRoom chatRoom, @NonNull User member, @Nullable String reason, @NonNull HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = new HMRCompletion(requestId, completion);

        if (member == null || member.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, INVALID_OPERATED_UID));
            return;
        }
        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mUnMuteMemberCompletions.put(requestId, c);
        }
        ChatRoomNative.unMuteMember(requestId.getId(), chatRoom, member, reason);
    }

    @Override
    public void fetchMutedUsers(@NonNull ChatRoom chatRoom, @NonNull HMR.CompletionArg<Set<User>> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<Set<User>> c = new HMRCompletionArg<>(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mFetchMuteUsersCompletions.put(requestId, c);
        }
        ChatRoomNative.fetchMutedUsers(requestId.getId(), chatRoom);
    }

    @Override
    public void isMuted(@NonNull ChatRoom chatRoom, @NonNull User member, @NonNull HMR.CompletionArg<Boolean> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<Boolean> c = new HMRCompletionArg<>(requestId, completion);

        if (member == null || member.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, INVALID_UID));
            return;
        }
        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mIsMutedCompletions.put(requestId, c);
        }
        ChatRoomNative.isMuted(requestId.getId(), chatRoom, member);
    }

    @Override
    public void fetchOnlineUserInfoList(@NonNull ChatRoom chatRoom, @NonNull HMR.CompletionArg<Map<User, Map<String, String>>> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<Map<User, Map<String, String>>> c = new HMRCompletionArg<>(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mFetchOnlineUserInfoListCompletions.put(requestId, c);
        }
        ChatRoomNative.fetchOnlineUserInfoList(requestId.getId(), chatRoom);
    }

    @Override
    public void batchFetchUserInfos(@NonNull ChatRoom chatRoom, @NonNull Set<User> users, @Nullable HMR.CompletionArg<Map<User, Map<String, String>>> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<Map<User, Map<String, String>>> c = new HMRCompletionArg<>(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        for (User user : users) {
            if (user == null || user.getId() < 0) {
                CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, INVALID_UID));
                return;
            }
        }

        if (completion != null) {
            mBathFetchUserInfosCompletions.put(requestId, c);
        }
        ChatRoomNative.batchFetchUserInfos(requestId.getId(), chatRoom, users);
    }

    @Override
    public void fetchUserInfo(@NonNull ChatRoom chatRoom, @NonNull User member, @NonNull HMR.CompletionArg<Map<String, String>> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<Map<String, String>> c = new HMRCompletionArg<>(requestId, completion);

        if (member == null || member.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, INVALID_UID));
            return;
        }
        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mFetchUserInfoCompletions.put(requestId, c);
        }
        ChatRoomNative.fetchUserInfo(requestId.getId(), chatRoom, member);
    }

    @Override
    public void setUserInfo(@NonNull ChatRoom chatRoom, @NonNull Map<String, String> infoMap, @Nullable HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = new HMRCompletion(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mSetUserInfoCompletions.put(requestId, c);
        }
        ChatRoomNative.setUserInfo(requestId.getId(), chatRoom, infoMap);
    }

    @Override
    public void deleteUserInfoByKeys(@NonNull ChatRoom chatRoom, @NonNull Set<String> keys, @Nullable HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = new HMRCompletion(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mDeleteUserInfoByKeysCompletions.put(requestId, c);
        }
        ChatRoomNative.deleteUserInfoByKeys(requestId.getId(), chatRoom, keys);
    }

    @Override
    public void addOrUpdateUserInfo(@NonNull ChatRoom chatRoom, @NonNull Map<String, String> infoMap, @Nullable HMR.Completion completion) {
        final RequestId requestId = newRequestId();
        HMRCompletion c = new HMRCompletion(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (error != null) {
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mAddOrUpdateUserInfoCompletions.put(requestId, c);
        }
        ChatRoomNative.addOrUpdateUserInfo(requestId.getId(), chatRoom, infoMap);
    }

    @Override
    public void fetchHistoryMessages(@NonNull ChatRoom chatRoom,
                                     @NonNull FetchingParams params,
                                     HMR.CompletionArg<FetchingResult> completion) {
        final RequestId requestId = newRequestId();
        HMRCompletionArg<FetchingResult> c = new HMRCompletionArg<>(requestId, completion);

        Error error = validRoomId(chatRoom);
        if (params == null) {
            error = new Error(ErrorEnum.INVALID_PARAMETER, "<params> should not be null");
        }
        if (error != null) {
            Log.e(TAG, Trace.method("fetchHistoryMessages").msg(error.toString()));
            CompletionUtils.dispatchFailure(c, error);
            return;
        }

        if (completion != null) {
            mFetchHistoryMessagesCompletions.put(requestId, c);
        }

        ChatRoomNative.fetchHistoryMessages(requestId.getId(), chatRoom, params);
    }

    private static Error validRoomId(ChatRoom chatRoom) {
        if (chatRoom == null || chatRoom.getId() < MIN_ROOM_ID || chatRoom.getId() > MAX_ROOM_ID) {
            return new Error(ErrorEnum.INVALID_PARAMETER, INVALID_ROOM_ID);
        }
        return null;
    }

    @Override
    public void addListener(@NonNull ChatRoomListener listener) {
        if (listener != null) {
            synchronized (mRoomListeners) {
                mRoomListeners.add(listener);
            }
        }
        HMRContext.reportReturnCode(ReportFunction.addListener, HMRContext.getCurrentTime());
    }

    @Override
    public void removeListener(@NonNull ChatRoomListener listener) {
        if (listener != null) {
            synchronized (mRoomListeners) {
                mRoomListeners.remove(listener);
            }
        }
        HMRContext.reportReturnCode(ReportFunction.removeListener, HMRContext.getCurrentTime());
    }

    @Override
    public void addMemberListener(@NonNull MemberListener listener) {
        if (listener != null) {
            synchronized (mRoomMemberListeners) {
                mRoomMemberListeners.add(listener);
            }
        }
        HMRContext.reportReturnCode(ReportFunction.addMemberListener, HMRContext.getCurrentTime());
    }

    @Override
    public void removeMemberListener(@NonNull MemberListener listener) {
        if (listener != null) {
            synchronized (mRoomMemberListeners) {
                mRoomMemberListeners.remove(listener);
            }
        }
        HMRContext.reportReturnCode(ReportFunction.removeMemberListener, HMRContext.getCurrentTime());
    }

    private HMRCompletion createHMRCompletion(@NonNull final RequestId requestId,
                                              final HMR.Completion completion,
                                              final String functionName) {
        return new HMRCompletion(requestId, new HMR.Completion() {
            @Override
            public void onSuccess() {
                if (completion != null) {
                    completion.onSuccess();
                }
                // 上报
                reportReturnCode(functionName, requestId.getId(), ErrorEnum.OK.getCode());
            }

            @Override
            public void onFailed(Error err) {
                if (completion != null) {
                    completion.onFailed(err);
                }
                reportReturnCode(functionName, requestId.getId(), err.code);
            }
        });
    }

    private static final String TAG = "ChatRoomService";
    private static final Map<Long, RequestId> mRequestIdMap = new ConcurrentHashMap<>();

    private static final Set<ChatRoomService.ChatRoomListener> mRoomListeners = new CopyOnWriteArraySet<>();
    private static final Set<ChatRoomService.MemberListener> mRoomMemberListeners = new CopyOnWriteArraySet<>();


    private static final Map<RequestId, HMRCompletionArg<ChatRoom>> mCreateChatRoomCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<List<User>>> mFetchMembersCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<Map<String, List<User>>>> mFetchRoleMembersCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<Integer>> mFetchMemberCountCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<ChatRoomInfo>> mFetchBasicInfoCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<Set<User>>> mFetchMuteUsersCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<Boolean>> mIsMutedCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<Map<User, Map<String, String>>>> mFetchOnlineUserInfoListCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<Map<User, Map<String, String>>>> mBathFetchUserInfosCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<Map<String, String>>> mFetchUserInfoCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> mDismissChatRoomCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> mJoinRoomCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> mLeaveRoomCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> mKickCompletions = new ConcurrentHashMap<>();

    private static final Map<RequestId, HMRCompletionArg<ChatRoomExtraAttribute>> FETCH_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> SET_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> UPDATE_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> DELETE_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> CLEAR_ROOM_EXTRA_ATTRIBUTE_COMPLETIONS = new ConcurrentHashMap<>();

    private static final Map<RequestId, HMRCompletion> mChangeBasicInfoCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> mMuteMemberCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> mUnMuteMemberCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> mSetUserInfoCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> mDeleteUserInfoByKeysCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> mAddOrUpdateUserInfoCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<FetchingResult>> mFetchHistoryMessagesCompletions = new ConcurrentHashMap<>();

    /**
     * 相同类型的使用同一个就好了
     */
    private static final Map<RequestId, HMRCompletion> CHAT_ROOM_COMPLETIONS = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<String>> FETCH_USER_ROLE_COMPLETIONS = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<FetchRoomBasicAttributesResult>> FETCH_ROOM_BASIC_ATTRIBUTES_COMPLETIONS = new ConcurrentHashMap<>();
}
