package com.hummer.im.chatroom;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.hummer.im.HMR;
import com.hummer.im.chatroom.model.attribute.ChatRoomExtraAttribute;
import com.hummer.im.chatroom.model.attribute.FetchRoomBasicAttributesResult;
import com.hummer.im.chatroom.model.attribute.RoomBasicAttributesOptions;
import com.hummer.im.chatroom.model.kick.KickOutType;
import com.hummer.im.chatroom.model.kick.KickUserOptions;
import com.hummer.im.chatroom.model.role.RoleOptions;
import com.hummer.im.model.chat.Content;
import com.hummer.im.model.fetch.FetchingParams;
import com.hummer.im.model.fetch.FetchingResult;
import com.hummer.im.model.id.ChatRoom;
import com.hummer.im.model.id.User;
import com.hummer.im.model.kick.KickOff;
import com.hummer.im.chatroom.model.attribute.ChatRoomAttributeOptions;

import java.util.List;
import java.util.Map;
import java.util.Set;

public interface ChatRoomService {

    /**
     * chatroom 切换区域
     *
     * @param region 国家区域
     */
    void setRegion(String region);

    /**
     * 创建聊天室
     *
     * @param chatRoomInfo 聊天室信息
     * @param completion   回调
     */
    void createChatRoom(@NonNull ChatRoomInfo chatRoomInfo,
                        @NonNull HMR.CompletionArg<ChatRoom> completion);

    /**
     * 解散聊天室
     *
     * @param chatRoom   聊天室标识符
     * @param completion 回调
     */
    void dismissChatRoom(@NonNull ChatRoom chatRoom,
                         @Nullable HMR.Completion completion);

    /**
     * 加入聊天室
     *
     * @param chatRoom   聊天室标识符
     * @param joinProps  预留属性，传入约定数据。没有则直接new HashMap()
     * @param completion 回调
     */
    void join(@NonNull ChatRoom chatRoom,
              @NonNull Map<String, String> joinProps,
              @NonNull Challenges.JoiningCompletion completion);

    /**
     * 离开聊天室
     *
     * @param chatRoom   聊天室标识符
     * @param completion 回调
     */
    void leave(@NonNull ChatRoom chatRoom, @Nullable HMR.Completion completion);

    enum EKickInfo {
        Time,
        Reason,
    }

    /**
     * 踢用户出聊天室
     *
     * @param chatRoom   聊天室标识符
     * @param member     踢出的聊天室成员
     * @param extraInfo  踢人额外信息
     * @param completion 回调
     *
     * @deprecated 于2.15.0废弃，由{@link ChatRoomService#kickUser}代替
     */
    void kick(@NonNull ChatRoom chatRoom,
              @NonNull User member,
              @Nullable Map<EKickInfo, String> extraInfo,
              @NonNull HMR.Completion completion);

    /**
     * 把聊天室内用户踢出聊天室
     *
     * @param chatRoom   聊天室
     * @param user       被踢出聊天室的成员
     * @param extraInfo  踢人的额外信息。目前支持的Key主要是两个，
     *                   一个是 {@link EKickInfo#Reason}, 用于携带踢人原因的 Key， 其 Key 对应的 Value 只需填写对应的踢人原因的字符串即可；
     *                   另一个是 {@link EKickInfo#Time} 用于携带限制时长的 Key， 其 Key 对应的 Value 只需要填写踢人限制时长的秒数转换成字符串即可, 这些字段都是透传字段，服务器不做处理
     * @param options    预留参数
     * @param completion 回调
     *
     * @since 2.15.0
     */
    void kickUser(@NonNull ChatRoom chatRoom,
                  @NonNull User user,
                  Map<String, String> extraInfo,
                  KickUserOptions options,
                  HMR.Completion completion);

    final class Roles {
        public static final String Admin = "admin";
        public static final String Owner = "owner";
    }

    /**
     * 设置聊天室用户角色
     *
     * <br> 单个聊天室内，最多可以指定10个Admin
     * <p> 只有Owner才可以指定Admin
     * <br> Owner指定Admin时，可以不在聊天室内，被操作者必须在聊天室内
     * <br> Owner不能设置自己为Admin
     * <br> 一个用户只能被指定为一个角色，不能同时存在多个角色
     * <br> 目前只能设置admin，其他都失败
     *
     * @param chatRoom   聊天室
     * @param user       需要被设置角色的用户
     * @param role       具体角色
     * @param options    可选参数，预留
     * @param completion 回调
     *
     * @since 2.15.0
     */
    void setUserRole(@NonNull ChatRoom chatRoom,
                     @NonNull User user,
                     @NonNull String role,
                     RoleOptions options,
                     HMR.Completion completion);

    /**
     * 删除聊天室用户角色
     *
     * <br> 只有Owner才可以删除/取消特定用户被赋予的该聊天室管理员[Admin]角色
     * <br> 操作者和被操作者都可以不在聊天室内
     *
     * @param chatRoom   聊天室
     * @param user       需要被删除角色的用户
     * @param options    可选参数，预留
     * @param completion 回调
     *
     * @since 2.15.0
     */
    void deleteUserRole(@NonNull ChatRoom chatRoom,
                        @NonNull User user,
                        RoleOptions options,
                        HMR.Completion completion);

    /**
     * 查询用户角色
     *
     * @param chatRoom   聊天室
     * @param user       用户
     * @param completion 回调
     *
     * @since 2.15.0
     */
    void fetchUserRole(@NonNull ChatRoom chatRoom,
                       @NonNull User user,
                       HMR.CompletionArg<String> completion);

    /**
     * 查询在聊天室内带角色的用户列表
     *
     * <br> 无角色限制，无是否在聊天室内限制，所有用户均可以调用该接口查询
     * <br> 返回的只是创建者[owner]、管理员[admin]角色列表，不支持返回普通用户列表
     *
     * @param chatRoom   聊天室
     * @param role       角色. null 表示查询所有当前在聊天室内的所有带角色的用户列表; owner和admin分别查询对应角色的用户列表
     * @param completion 回调
     *
     * @since 2.15.0
     */
    void fetchRoomUsersByRole(@NonNull ChatRoom chatRoom,
                              String role,
                              HMR.CompletionArg<Map<String, List<User>>> completion);

    /**
     * 添加聊天室权限角色
     *
     * @param chatRoom   聊天室标识符
     * @param member     成员
     * @param role       角色类型,具体值可参考{@link Roles},业务也可以自己定义扩展角色
     * @param completion 回调
     *
     * @deprecated 2.15.0 版本废弃，由{@link ChatRoomService#setUserRole}替代
     */
    @Deprecated
    void addRole(@NonNull ChatRoom chatRoom,
                 @NonNull User member,
                 @NonNull String role,
                 @NonNull HMR.Completion completion);

    /**
     * 移除聊天室权限角色
     *
     * @param chatRoom   聊天室标识符
     * @param member     成员
     * @param role       角色类型,具体值可参考{@link Roles},业务也可以自己定义扩展角色
     * @param completion 回调
     *
     * @deprecated 2.15.0 版本废弃，由{@link ChatRoomService#deleteUserRole}替代
     */
    @Deprecated
    void removeRole(@NonNull ChatRoom chatRoom,
                    @NonNull User member,
                    @NonNull String role,
                    @NonNull HMR.Completion completion);

    /**
     * 获取聊天室成员列表
     *
     * @param chatRoom   聊天室标识符
     * @param num        拉取的条数
     * @param offset     拉取的位置，第一页从0开始
     * @param completion 回调,List<Fellow> 聊天室成员列表
     */
    void fetchMembers(@NonNull ChatRoom chatRoom, int num, int offset,
                      @NonNull HMR.CompletionArg<List<User>> completion);

    /**
     * 获取聊天室的所有角色列表
     *
     * @param chatRoom   聊天室标识符
     * @param online     true获取在线的角色列表，false获取全部角色列表
     * @param completion 回调, Map<String, List<Fellow> 角色成员列表,key 对应的取值是DefaultRoleType类型，或者是业务自定义角色类型
     *
     * @deprecated 于2.15.0版本废弃，由 {@link ChatRoomService#fetchRoomUsersByRole} 代替
     */
    @Deprecated
    void fetchRoleMembers(@NonNull ChatRoom chatRoom, boolean online,
                          @NonNull HMR.CompletionArg<Map<String, List<User>>> completion);

    /**
     * 获取聊天室的成员总数
     *
     * @param chatRoom   聊天室
     * @param completion 回调
     */
    void fetchMemberCount(@NonNull ChatRoom chatRoom,
                          @NonNull HMR.CompletionArg<Integer> completion);

    /**
     * 获取聊天室基本属性信息
     *
     * @param chatRoom   聊天室标识符
     * @param completion 回调, RoomInfo 聊天室属性信息
     *
     * @deprecated
     */
    @Deprecated
    void fetchBasicInfo(@NonNull ChatRoom chatRoom,
                        @NonNull HMR.CompletionArg<ChatRoomInfo> completion);

    /**
     * 查询房间基础属性
     *
     * @param chatRoom   聊天室标识符
     * @param keys       属性key集合
     * @param completion 回调, RoomInfo 聊天室属性信息
     *
     * @since 2.15.0
     */
    void fetchRoomBasicAttributes(@NonNull ChatRoom chatRoom,
                                  Set<String> keys,
                                  HMR.CompletionArg<FetchRoomBasicAttributesResult> completion);

    /**
     * 修改聊天室基本属性信息
     *
     * @param chatRoom   聊天室标识符
     * @param propInfo   修改属性值
     * @param completion 回调
     *
     * @deprecated
     */
    @Deprecated
    void changeBasicInfo(@NonNull ChatRoom chatRoom,
                         @NonNull Map<ChatRoomInfo.BasicInfoType, String> propInfo,
                         @NonNull HMR.Completion completion);

    /**
     * 修改聊天室基础属性
     *
     * @param chatRoom   聊天室标识符
     * @param attributes 修改的属性值
     * @param options    預留參數
     * @param completion 回调
     *
     * @since 2.15.0
     */
    void updateRoomBasicAttributes(@NonNull ChatRoom chatRoom,
                                   @NonNull Map<String, String> attributes,
                                   RoomBasicAttributesOptions options,
                                   HMR.Completion completion);

    /**
     * 设置聊天室扩展属性
     *
     * <p> 每个属性Key最大32字节
     * <p> 每个属性值最大8KB
     * <p> 每个聊天室最多有32条属性
     * <p> 设置聊天室属性的时候Key不能为空（返回错误），Value可以为空
     * <p> Owner 和Admin 权限的用户才可以执行设置聊天室扩展属性操作
     * <p> 操作者需要在该聊天室聊天室内才可以执行设置聊天室扩展属性操作
     *
     * @param chatRoom   聊天室
     * @param attributes 属性
     * @param options    预留字段
     * @param completion 回调
     *
     * @since 2.14.0
     */
    void setRoomExtraAttributes(@NonNull ChatRoom chatRoom,
                                @NonNull Map<String, String> attributes,
                                @Nullable ChatRoomAttributeOptions options,
                                @Nullable HMR.Completion completion);

    /**
     * 更新聊天室扩展属性：属性存在则更新；属性不存在则添加；
     *
     * <p> 每个属性Key最大32字节
     * <p> 每个属性值最大8KB
     * <p> 每个聊天室最多有32条属性
     * <p> 设置聊天室属性的时候Key不能为空（返回错误），Value可以为空
     * <p> Owner 和Admin 权限的用户才可以执行设置聊天室扩展属性操作
     * <p> 操作者需要在该聊天室聊天室内才可以执行设置聊天室扩展属性操作
     *
     * @param chatRoom   聊天室
     * @param attributes 属性
     * @param options    预留参数
     * @param completion 回调
     *
     * @since 2.14.0
     */
    void updateRoomExtraAttributes(@NonNull ChatRoom chatRoom,
                                   @NonNull Map<String, String> attributes,
                                   @Nullable ChatRoomAttributeOptions options,
                                   @Nullable HMR.Completion completion);

    /**
     * 删除聊天室扩展属性
     *
     * <p> Owner 和Admin 权限的用户才可以执行设置聊天室扩展属性操作
     * <p> 操作者需要在该聊天室聊天室内才可以执行设置聊天室扩展属性操作
     *
     * @param chatRoom   聊天室
     * @param keys       要删除的属性key
     * @param options    预留参数
     * @param completion 回调
     *
     * @since 2.14.0
     */
    void deleteRoomExtraAttributes(@NonNull ChatRoom chatRoom,
                                   @NonNull Set<String> keys,
                                   @Nullable ChatRoomAttributeOptions options,
                                   @Nullable HMR.Completion completion);

    /**
     * 清空聊天室扩展属性
     *
     * <p> Owner 和Admin 权限的用户才可以执行设置聊天室扩展属性操作
     * <p> 操作者需要在该聊天室聊天室内才可以执行设置聊天室扩展属性操作
     *
     * @param chatRoom   聊天室
     * @param options    预留参数
     * @param completion 回调
     *
     * @since 2.14.0
     */
    void clearRoomExtraAttributes(@NonNull ChatRoom chatRoom,
                                  @Nullable ChatRoomAttributeOptions options,
                                  @Nullable HMR.Completion completion);

    /**
     * 查询聊天室的扩展属性
     *
     * <p> 无角色限制，任何用户均可以执行查询聊天室扩展属性操作
     * <p> 无在聊天室内限制，任何用户均可以执行查询聊天室扩展属性操作
     *
     * @param chatRoom   聊天室
     * @param keys       属性keys，默认为空，即表示查询全量属性，最多不超过32个key
     * @param completion 回调
     *
     * @since 2.14.0
     */
    void fetchRoomExtraAttributes(@NonNull ChatRoom chatRoom,
                                  @Nullable Set<String> keys,
                                  @Nullable HMR.CompletionArg<ChatRoomExtraAttribute> completion);

    /**
     * 聊天室公屏禁言
     *
     * <pre>
     * // 禁言 mChatRoom 下的 member 成员，无原因
     * HMR.getService(ChatRoomService.class).muteMember(mChatRoom, fellow, null,
     *          new RichCompletion()
     *                  .onSuccess(() -> {
     *                      // doSomething
     *                  })
     *                  .onFailure(err -> {
     *                      // doSomething
     *                  }));
     * </pre>
     *
     * @param chatRoom   聊天室
     * @param member     被禁言的成员, 成员必须在聊天室内
     * @param reason     禁言原因
     * @param completion 回调
     */
    void muteMember(@NonNull ChatRoom chatRoom,
                    @NonNull User member,
                    @Nullable String reason,
                    @NonNull HMR.Completion completion);

    /**
     * 聊天室解除禁言
     *
     * <pre>
     * HMR.getService(ChatRoomService.class).unmuteMember(mChatRoom, fellow, null,
     *          new RichCompletion()
     *                  .onSuccess(() -> {
     *                      // doSomething
     *                  })
     *                  .onFailure(err -> {
     *                      // doSomething
     *                  }));
     * </pre>
     *
     * @param chatRoom   聊天室标识
     * @param member     被解禁成员，必须在聊天室内
     * @param reason     解禁原因
     * @param completion 回调
     */
    void unmuteMember(@NonNull ChatRoom chatRoom,
                      @NonNull User member,
                      @Nullable String reason,
                      @NonNull HMR.Completion completion);

    /**
     * 获取禁言用户列表：所有在chatroom中被禁言过的用户
     *
     * <pre>
     * HMR.getService(ChatRoomService.class).fetchMutedUsers(mChatRoom, new RichCompletionArg<Set<User>>()
     *                 .onSuccess((users) -> {
     *                     // doSomething
     *                 })
     *                 .onFailure(error -> {
     *                     // doSomething
     *                 }));
     * </pre>
     *
     * @param chatRoom   聊天室
     * @param completion 回调
     */
    void fetchMutedUsers(@NonNull ChatRoom chatRoom, @NonNull HMR.CompletionArg<Set<User>> completion);

    /**
     * 用户是否被禁言：true 表示是; false 表示否
     *
     * <pre>
     * HMR.getService(ChatRoomService.class).isMuted(mChatRoom, user,
     *             new RichCompletionArg<Boolean>()
     *                     .onSuccess(isMuted -> {
     *                          doSomething
     *                     })
     *                     .onFailure(error -> {
     *                         doSomething
     *                     }));
     * </pre>
     *
     * @param chatRoom   聊天室
     * @param member     被查询是否禁言的用户
     * @param completion 回调
     */
    void isMuted(@NonNull ChatRoom chatRoom,
                 @NonNull User member,
                 @NonNull HMR.CompletionArg<Boolean> completion);

    /**
     * 获取聊天室下在线用户信息
     *
     * @param chatRoom   聊天室
     * @param completion 回调
     */
    void fetchOnlineUserInfoList(@NonNull ChatRoom chatRoom,
                                 @NonNull HMR.CompletionArg<Map<User, Map<String, String>>> completion);

    /**
     * 批量获取指定用户的用户信息
     *
     * @param chatRoom   聊天室
     * @param users      需要查询的用户集合
     * @param completion 回调
     */
    void batchFetchUserInfos(@NonNull ChatRoom chatRoom,
                             @NonNull Set<User> users,
                             @Nullable HMR.CompletionArg<Map<User, Map<String, String>>> completion);

    /**
     * 获取指定用户的全部属性
     *
     * @param chatRoom   聊天室
     * @param member
     * @param completion 回调
     */
    void fetchUserInfo(@NonNull ChatRoom chatRoom,
                       @NonNull User member,
                       @NonNull HMR.CompletionArg<Map<String, String>> completion);

    /**
     * 设置用户在当前频道的信息
     *
     * @param chatRoom   聊天室
     * @param infoMap    信息
     * @param completion 回调
     */
    void setUserInfo(@NonNull final ChatRoom chatRoom,
                     @NonNull final Map<String, String> infoMap,
                     @Nullable final HMR.Completion completion);

    /**
     * 删除用户指定属性
     *
     * @param chatRoom   聊天室
     * @param keys       属性key
     * @param completion 回调
     */
    void deleteUserInfoByKeys(@NonNull ChatRoom chatRoom,
                              @NonNull Set<String> keys,
                              @Nullable HMR.Completion completion);

    /**
     * 添加或更新用户属性
     *
     * @param chatRoom   聊天室
     * @param infoMap    信息
     * @param completion 回调
     */
    void addOrUpdateUserInfo(@NonNull ChatRoom chatRoom,
                             @NonNull Map<String, String> infoMap,
                             @Nullable HMR.Completion completion);

    /**
     * 查询历史消息
     *
     * @param chatRoom   聊天室
     * @param params     获取历史消息需要的条件, {@link FetchingParams}
     * @param completion 回调
     */
    void fetchHistoryMessages(@NonNull ChatRoom chatRoom,
                              @NonNull FetchingParams params,
                              HMR.CompletionArg<FetchingResult> completion);

    /**
     * 添加聊天室回调监听器
     *
     * @param listener 监听器对象
     */
    void addListener(@NonNull final ChatRoomListener listener);

    /**
     * 移除聊天室回调监听器
     *
     * @param listener 监听器对象
     */
    void removeListener(@NonNull final ChatRoomListener listener);

    /**
     * 添加聊天室成员相关回调监听器
     *
     * @param listener 监听器对象
     */
    void addMemberListener(@NonNull final MemberListener listener);

    /**
     * 移除聊天室成员相关回调监听器
     *
     * @param listener 监听器对象
     */
    void removeMemberListener(@NonNull final MemberListener listener);

    interface ChatRoomListener {

        /**
         * 聊天室基本信息变化的回调通知
         *
         * @deprecated 于2.15.0废弃，由{@link ChatRoomListener#onRoomBasicAttributesUpdated} 代替
         */
        @Deprecated
        void onBasicInfoChanged(@NonNull ChatRoom chatRoom,
                                @NonNull Map<ChatRoomInfo.BasicInfoType, String> propInfo);

        /**
         * 聊天室基础属性更新回调
         *
         * @param chatRoom       聊天室
         * @param attributes     修改的基础属性
         * @param operator       操作者
         * @param latestUpdateTs 最后一次更新时间戳
         *
         * @since 2.15.0
         */
        void onRoomBasicAttributesUpdated(@NonNull ChatRoom chatRoom,
                                          @NonNull Map<String, String> attributes,
                                          @NonNull User operator,
                                          long latestUpdateTs);

        /**
         * 聊天室被解散的通知
         *
         * @param member   操作者
         * @param chatRoom 聊天室
         */
        void onChatRoomDismissed(@NonNull ChatRoom chatRoom, @NonNull User member);

        /**
         * 聊天室扩展属性设置通知
         *
         * @param chatRoom       聊天室
         * @param attributes     对应属性值
         * @param operator       操作者
         * @param latestUpdateTs 最后一次更新时间
         *
         * @since 2.14.0
         */
        void onRoomExtraAttributesSet(@NonNull ChatRoom chatRoom,
                                      @NonNull Map<String, String> attributes,
                                      @NonNull User operator,
                                      long latestUpdateTs);

        /**
         * 聊天室扩展属性设置通知
         *
         * @param chatRoom       聊天室
         * @param attributes     对应属性值
         * @param operator       操作者
         * @param latestUpdateTs 最后一次更新时间
         *
         * @since 2.14.0
         */
        void onRoomExtraAttributesUpdated(@NonNull ChatRoom chatRoom,
                                          @NonNull Map<String, String> attributes,
                                          @NonNull User operator,
                                          long latestUpdateTs);

        /**
         * 聊天室扩展属性设置通知
         *
         * @param chatRoom       聊天室
         * @param attributes     对应属性值
         * @param operator       操作者
         * @param latestUpdateTs 最后一次更新时间
         *
         * @since 2.14.0
         */
        void onRoomExtraAttributesDeleted(@NonNull ChatRoom chatRoom,
                                          @NonNull Map<String, String> attributes,
                                          @NonNull User operator,
                                          long latestUpdateTs);

        /**
         * 聊天室扩展属性设置通知
         *
         * @param chatRoom       聊天室
         * @param attributes     对应属性值
         * @param operator       操作者
         * @param latestUpdateTs 最后一次更新时间
         *
         * @since 2.14.0
         */
        void onRoomExtraAttributesCleared(@NonNull ChatRoom chatRoom,
                                          @NonNull Map<String, String> attributes,
                                          @NonNull User operator,
                                          long latestUpdateTs);
    }

    interface MemberListener {
        /**
         * 用户加入聊天室的回调通知
         */
        void onMemberJoined(@NonNull ChatRoom chatRoom, @NonNull List<User> members);

        /**
         * 用户离开聊天室的回调通知
         *
         * @param chatRoom 聊天室
         * @param members  离开的成员列表
         * @param type     离开原因类型：0 离开; 1 断开连接
         * @param reason   离开原因
         *
         * @since 2.9.6, 后合并到2.10分支，2.9.1XX相应版本无此功能
         * @deprecated As of Hummer version 2.9.6, replaced by
         * <code>onChatRoomMemberLeave(ChatRoom chatRoom, List members)</code>
         * and <code>onChatRoomMemberOffline(ChatRoom chatRoom)</code>.
         */
        @Deprecated
        void onMemberLeaved(@NonNull ChatRoom chatRoom,
                            @NonNull List<User> members,
                            int type,
                            @NonNull String reason);

        /**
         * 用户离开聊天室的回调通知
         *
         * @param chatRoom 聊天室
         * @param members  离开的成员列表
         *
         * @since 2.9.6
         */
        void onChatRoomMemberLeave(@NonNull ChatRoom chatRoom, @NonNull List<User> members);

        /**
         * 当前登录用户断网超时回调通知
         *
         * @param chatRoom 聊天室
         *
         * @since 2.9.6
         */
        void onChatRoomMemberOffline(@NonNull ChatRoom chatRoom);

        /**
         * 聊天室成员数变化的回调通知
         */
        void onMemberCountChanged(@NonNull ChatRoom chatRoom, int count);

        /**
         * 添加聊天室权限角色的回调通知
         *
         * @deprecated 2.15.0 版本废弃，由{@link MemberListener#onUserRoleSet}代替
         */
        @Deprecated
        void onRoleAdded(@NonNull ChatRoom chatRoom,
                         @NonNull String role,
                         @NonNull User admin,
                         @NonNull User fellow);

        /**
         * 移除聊天室权限角色的回调通知
         *
         * @deprecated 2.15.0 版本废弃，由{@link MemberListener#onUserRoleDeleted}代替
         */
        @Deprecated
        void onRoleRemoved(@NonNull ChatRoom chatRoom,
                           @NonNull String role,
                           @NonNull User admin,
                           @NonNull User fellow);

        /**
         * 设置用户角色回调通知
         *
         * @param chatRoom 聊天室
         * @param role     被赋予的具体角色
         * @param user     被赋予角色的用户
         * @param operator 操作者
         *
         * @since 2.15.0
         */
        void onUserRoleSet(@NonNull ChatRoom chatRoom,
                           @NonNull String role,
                           @NonNull User user,
                           @NonNull User operator);

        /**
         * 删除用户角色回调通知
         *
         * @param chatRoom 聊天室
         * @param role     被删除的具体角色
         * @param user     被删除角色的用户
         * @param operator 操作者
         *
         * @since 2.15.0
         */
        void onUserRoleDeleted(@NonNull ChatRoom chatRoom,
                               @NonNull String role,
                               @NonNull User user,
                               @NonNull User operator);

        /**
         * 用户被踢出聊天室的回调通知
         *
         * @param chatRoom 聊天室
         * @param operator 执行踢出操作的管理员
         * @param type     被踢类型
         * @param infos    扩展信息字段，包含Time,Reason
         *
         * @since 2.15.0
         */
        void onUserKicked(@NonNull ChatRoom chatRoom,
                          @NonNull User operator,
                          @NonNull KickOutType type,
                          @NonNull Map<String, String> infos);

        /**
         * 用户被踢出聊天室的回调通知
         *
         * @param chatRoom 聊天室
         * @param admin    管理员
         * @param member   被踢用户
         * @param kickOff  踢人类型和原因等
         *
         * @deprecated 于2.15.0版本废弃，由{@link MemberListener#onUserKicked}替代
         */
        @Deprecated
        void onMemberKicked(@NonNull ChatRoom chatRoom,
                            @NonNull User admin,
                            @NonNull List<User> member,
                            @NonNull KickOff kickOff);

        /**
         * 用户被踢出聊天室的回调通知
         *
         * @param chatRoom 聊天室
         * @param admin    管理员
         * @param member   被踢用户
         * @param reason   原因
         *
         * @since 2.10.0
         * @deprecated
         */
        @Deprecated
        void onMemberKicked(@NonNull ChatRoom chatRoom,
                            @NonNull User admin,
                            @NonNull List<User> member,
                            @NonNull String reason);

        /**
         * 用户被禁言通知
         *
         * @param chatRoom 聊天室
         * @param operator 禁言者
         * @param members  被禁言者们
         * @param reason   原因
         */
        void onMemberMuted(@NonNull ChatRoom chatRoom,
                           @NonNull User operator,
                           @NonNull Set<User> members,
                           @Nullable String reason);

        /**
         * 用户解禁通知
         *
         * @param chatRoom 聊天室
         * @param operator 解禁操作人
         * @param members  被解禁者们
         * @param reason   原因
         */
        void onMemberUnmuted(@NonNull ChatRoom chatRoom,
                             @NonNull User operator,
                             @NonNull Set<User> members,
                             @Nullable String reason);

        /**
         * 聊天室成员信息设置回调通知
         *
         * @param chatRoom 聊天室
         * @param user     成员
         * @param infoMap  信息
         */
        void onUserInfoSet(@NonNull ChatRoom chatRoom,
                           @NonNull User user,
                           @NonNull Map<String, String> infoMap);

        /**
         * 聊天室成员信息新增/更新回调通知
         *
         * @param chatRoom 聊天室
         * @param user     属性变更用户
         * @param infoMap  变更属性
         */
        void onUserInfoAddedOrUpdated(@NonNull ChatRoom chatRoom,
                                      @NonNull User user,
                                      @NonNull Map<String, String> infoMap);

        /**
         * 聊天室成员信息删除回调通知
         *
         * @param chatRoom 聊天室
         * @param user     属性变更用户
         * @param infoMap  删除的属性
         */
        void onUserInfoDeleted(@NonNull ChatRoom chatRoom,
                               @NonNull User user,
                               @NonNull Map<String, String> infoMap);
    }

    /**
     * 聊天室信令消息
     */
    final class Signal extends Content {
        public final User user;
        public final String content;

        public static Signal unicast(@NonNull User user, @NonNull String content) {
            return new Signal(user, content);
        }

        public static Signal broadcast(@NonNull String content) {
            return new Signal(null, content);
        }

        private Signal(User user, @NonNull String content) {
            this.user = user;
            this.content = content;
        }

        @Override
        public String toString() {
            return "ChatRoom.Signal{" + content + "}";
        }
    }

}




