package com.hummer.im.channel._internals;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Base64;

import com.google.protobuf.ByteString;
import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.Objects;
import com.hummer.im._internals.chatsvc.ChatServiceImpl;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.mq.Source;
import com.hummer.im._internals.proto.ChannelProto;
import com.hummer.im._internals.proto.Im;
import com.hummer.im._internals.proto.Push;
import com.hummer.im._internals.proto.Signal;
import com.hummer.im._internals.shared.DispatchQueue;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im._internals.shared.statis.TextUtils;
import com.hummer.im.channel.ChannelService;
import com.hummer.im.channel._internals.rpc.RPCDeleteUserInfo;
import com.hummer.im.channel._internals.rpc.RPCFetchMembers;
import com.hummer.im.channel._internals.rpc.RPCFetchMembersByAttr;
import com.hummer.im.channel._internals.rpc.RPCJoinChannel;
import com.hummer.im.channel._internals.rpc.RPCLeaveChannel;
import com.hummer.im.channel._internals.rpc.RPCSendBroadcast;
import com.hummer.im.channel._internals.rpc.RPCSetUserInfo;
import com.hummer.im.model.signal.Message;
import com.hummer.im.model.signal.SendingOptions;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.completion.RichCompletionArg;
import com.hummer.im.model.id.User;
import com.hummer.im.service.Channel;
import com.hummer.im.service.MQService;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import static com.hummer.im._internals.proto.Signal.NotifyUinfoChange.OpType.kOpUpdate;
import static com.hummer.im._internals.proto.Signal.NotifyUinfoChange.OpType.kOpDel;
import static com.hummer.im._internals.proto.Signal.NotifyUinfoChange.OpType.kOpSet;

public class ChannelServiceImpl implements ChannelService, ServiceProvider.Service {

    public static final String TAG = "ChannelServiceImpl";
    private static final String OFFLINE_DURATION_KEY = "props_offline_check_duration";
    private static long kRejoinConditionGap = 55 * 1000;
    private static final long kReferenceValue = 946656000;

    private final HashSet<Listener> mListeners = new HashSet<>();
    private final HashSet<com.hummer.im.model.Channel> mJoinedChannel = new HashSet<>();
    private Channel.StateChangedListener mChannelStateListener;
    private long mInitialTimeInterval = 0L;

    private static final String GROUP_URI
            = "service_api_gateway/cim.proto.PushService.UnreliableIMPushGroupSysMsg";

    private void registerChatServiceExtensions() {
        ChatServiceImpl.registerParserExtension(new ChatServiceImpl.ParserExtension() {
            @Override
            public String toString() {
                return "ChannelServiceImpl";
            }

            @Override
            public com.hummer.im.model.chat.Message parseMessage(String service, String function, byte[] data) {
                String actualUri = service + '/' + function;

                if (Objects.equals(actualUri, GROUP_URI)) {
                    if (isChannelTimeOut()) {
                        Log.i(TAG, Trace.once()
                                .method("parseMessage")
                                .msg("Ignore the join msg for heartbeat timeout"));
                        return null;
                    }

                    try {
                        Push.UnreliableIMPushGroupSysMsgRequest req = Push.UnreliableIMPushGroupSysMsgRequest
                                .newBuilder()
                                .mergeFrom(data)
                                .build();

                        if (req.getEnvName() != null && !req.getEnvName().isEmpty()) {
                            HMRContext.Region remoteRegion = HMRContext.Region.make(req.getRegion()
                                    + "/" + req.getEnvType()
                                    + "/" + req.getEnvName());

                            String localEnv = HMRContext.region.toString();
                            String remoteEnv = remoteRegion.toString();

                            if (!Objects.equals(remoteEnv, localEnv)) {
                                // 收到不属于当前接入区域的消息，直接忽略

                                Log.i(TAG, Trace.once().method("parseMessage")
                                        .msg("Ignored parse of different env")
                                        .info("localEnv", localEnv)
                                        .info("remoteEnv", remoteEnv));

                                return null;
                            }
                        }

                        parseChannelMessage(req.getMsg().getContent(),
                                req.getMsg().getTimestamp(),
                                channelNameFromTopic(req.getTopic()));
                    } catch (Throwable e) {
                        Log.e(TAG,
                                Trace.once("Failed parsing UnreliableIMPushGroupSysMsgRequest")
                                        .info("Exception", e));
                    }
                }

                return null;
            }

            @Override
            public com.hummer.im.model.chat.Message parseMessage(Im.Msg msg, Source fromSource) {
                try {
                    if (msg.getAction() == Im.Action.kNotifyUinfoChange_VALUE) { // 成员信息变更通知
                        parseMemberInfoChangeNotify(msg.getContent());
                    } else if (msg.getAction() // 用户进出频道通知
                            == ChannelProto.ChannelAction.kNotifyUserOnlineStatusChange_VALUE) {
                        parseUserChangeNotify(msg.getContent());
                    } else if (msg.getAction() == Im.Action.kNotifySignalMessage_VALUE) { // 信令消息
                        parseChannelMessage(msg.getContent(), msg.getTimestamp(), fromSource);
                    }
                } catch (Throwable t) {
                    Log.e(TAG, Trace.once().method("parse")
                            .info("action", msg.getAction())
                            .info("msg", msg.getContent())
                            .info("exception", t.getMessage()));
                }

                return null;
            }
        });
    }

    private void parseChannelMessage(final ByteString data, long timestamp, Source fromSource) throws Throwable {
        if (fromSource.getMode() instanceof Source.Shared) {
            if (isChannelTimeOut()) {
                Log.i(TAG, Trace.once()
                        .method("parseChannelMessage")
                        .msg("Ignore the join msg for heartbeat timeout"));
                return;
            }

            String channelName = channelNameFromTopic(fromSource.getMode().topicName());
            parseChannelMessage(data, timestamp, channelName);
        }
    }

    private void parseChannelMessage(final ByteString data, long timestamp, String channelName) throws Throwable {

        com.hummer.im.model.Channel channel = new com.hummer.im.model.Channel(channelName);
        if (TextUtils.isBlank(channelName) || !mJoinedChannel.contains(channel)) {
            Log.i(TAG, Trace.once().method("parseChannelMessage")
                    .msg("user[%d] already leave the channel[%s]", HMR.getMe().getId(), channel.getName()));
            return;
        }

        Signal.SignalMessage req = Signal.SignalMessage
                .newBuilder()
                .mergeFrom(data)
                .build();

        Message message = new Message(req.getType(), req.getContent().toByteArray());
        message.setServerAcceptedTs(timestamp);
        message.setAppExtras(req.getExtensionsMap());

        notifyReceivedMessage(channel, message, new User(req.getFromUid()));
    }

    private void notifyReceivedMessage(@NonNull final com.hummer.im.model.Channel channel,
                                       @NonNull final Message message,
                                       @NonNull final User fromUser) {
        Log.i(TAG, Trace.once()
                .method("notifyReceivedMessage")
                .info("message.type", message.getType())
                .info("message.ts", message.getServerAcceptedTs())
                .info("fromUser", fromUser.getId())
                .info("channel", channel.getName()));

        DispatchQueue.main.async("ChannelService::notifyReceivedMessage", new Runnable() {
            @Override
            public void run() {

                synchronized (mListeners) {
                    HashSet<Listener> copyListener = (HashSet<Listener>) mListeners.clone();
                    for (Listener l : copyListener) {
                        l.onReceivedMessage(channel, message, fromUser);
                    }
                }
            }
        });
    }

    private void parseUserChangeNotify(ByteString data) throws Throwable {
        if (isChannelTimeOut()) {
            Log.i(TAG, Trace.once()
                    .method("parseUserChangeNotify")
                    .msg("Ignore the join msg for heartbeat timeout"));
            return;
        }

        ChannelProto.NotifyUserStatusChange req = ChannelProto.NotifyUserStatusChange
                .newBuilder()
                .mergeFrom(data)
                .build();
        com.hummer.im.model.Channel channel = new com.hummer.im.model.Channel(req.getGroupId());
        if (!mJoinedChannel.contains(channel)) {
            Log.i(TAG, Trace.once().method("parseUserChangeNotify")
                    .info("logId", req.getLogId())
                    .msg("user[%d] already leave the channel[%s]", HMR.getMe().getId(), channel.getName()));
            return;
        }

        final List<User> joinUsers = new ArrayList<>();
        final List<User> leaveUsers = new ArrayList<>();
        if (req.getJoinuidsCount() > 0) {
            for (Long uid : req.getJoinuidsList()) {
                joinUsers.add(new User(uid));
            }
            notifyUsersJoined(channel, joinUsers);
        }
        if (req.getLeaveuidsCount() > 0) {
            for (Long uid : req.getLeaveuidsList()) {
                leaveUsers.add(new User(uid));
            }
            notifyUsersLeaved(channel, leaveUsers);
        }
    }

    private void parseMemberInfoChangeNotify(ByteString data) throws Throwable {
        if (isChannelTimeOut()) {
            Log.i(TAG, Trace.once()
                    .method("parseMemberInfoChangeNotify")
                    .msg("Ignore the join msg for heartbeat timeout"));
            return;
        }

        Signal.NotifyUinfoChange req = Signal.NotifyUinfoChange
                .newBuilder()
                .mergeFrom(data)
                .build();

        com.hummer.im.model.Channel channel = new com.hummer.im.model.Channel(req.getGroupId());
        if (!mJoinedChannel.contains(channel)) {
            Log.i(TAG, Trace.once()
                    .method("Ignore the join msg for heartbeat timeout")
                    .info("groupId", req.getGroupId()));
            return;
        }

        if (Objects.equals(req.getOpType(), kOpSet)) {
            notifyUserInfoUpdate(channel, new User(req.getUid()), req.getInfosMap());
        } else if (Objects.equals(req.getOpType(), kOpDel)) {
            notifyUserInfoDelete(channel, new User(req.getUid()), req.getInfosMap().keySet());
        } else if (req.getOpType().equals(kOpUpdate)) {
            Log.w(TAG, Trace.once().method("parseMemberInfoChangeNotify")
                    .msg("the operator type[%s] is not parse", req.getOpType()));
        } else {
            Log.w(TAG, Trace.once().method("parseMemberInfoChangeNotify")
                    .msg("the operator type[%s] is not exist", req.getOpType()));
        }
    }

    // Mark - notify message

    private void notifyUsersLeaved(@NonNull final com.hummer.im.model.Channel channel,
                                   @NonNull final List<User> users) {
        Log.i(TAG, Trace.once()
                .method("notifyUsersLeaved")
                .info("channel", channel.getName())
                .info("users", users));

        DispatchQueue.main.async("ChannelServiceImpl::notifyUserLeaved", new Runnable() {
            @Override
            public void run() {

                synchronized (mListeners) {
                    HashSet<Listener> copyListener = (HashSet<Listener>) mListeners.clone();
                    for (Listener l : copyListener) {
                        l.onUsersLeaved(channel, users);
                    }
                }
            }
        });
    }

    private void notifyUsersJoined(@NonNull final com.hummer.im.model.Channel channel,
                                   @NonNull final List<User> users) {
        Log.i(TAG, Trace.once()
                .method("notifyUsersJoined")
                .info("channel", channel.getName())
                .info("users", users));

        DispatchQueue.main.async("ChannelService::notifyUserJoined", new Runnable() {
            @Override
            public void run() {

                synchronized (mListeners) {
                    HashSet<Listener> copyListener = (HashSet<Listener>) mListeners.clone();
                    for (Listener l : copyListener) {
                        l.onUsersJoined(channel, users);
                    }
                }
            }
        });
    }

    private void notifyUserInfoUpdate(@NonNull final com.hummer.im.model.Channel channel,
                                      @NonNull final User user,
                                      @Nullable final Map<String, String> infoMap) {
        Log.i(TAG, Trace.once()
                .method("notifyUserInfoUpdate")
                .info("channel", channel.getName())
                .info("user", user.getId())
                .info("info", infoMap));

        DispatchQueue.main.async("ChannelService::notifyMemberInfoUpdate", new Runnable() {
            @Override
            public void run() {
                synchronized (mListeners) {
                    HashSet<Listener> copyListener = (HashSet<Listener>) mListeners.clone();
                    for (Listener l : copyListener) {
                        l.onUserInfoSet(channel, user, infoMap);
                    }
                }
            }
        });
    }

    private void notifyUserInfoDelete(@NonNull final com.hummer.im.model.Channel channel,
                                      @NonNull final User user,
                                      @Nullable final Set<String> keys) {
        Log.i(TAG, Trace.once()
                .method("notifyUserInfoDelete")
                .info("user", user.getId())
                .info("channel", channel.getName()));

        DispatchQueue.main.async("ChannelService::notifyUserInfoDelete", new Runnable() {
            @Override
            public void run() {

                synchronized (mListeners) {
                    HashSet<Listener> copyListener = (HashSet<Listener>) mListeners.clone();
                    for (Listener l : copyListener) {
                        l.onUserInfoRemove(channel, user, keys);
                    }
                }
            }
        });
    }

    // Mark - internal logic

    private String getTopic(String channelName) {
        return String.format(Locale.US, "SignalService/%s", channelName);
    }

    private static long djbHash(String topic) {
        String topicBase64 = Base64.encodeToString(topic.getBytes(), Base64.NO_WRAP);

        long hash = 5381;
        for (int i = 0; i < topicBase64.length(); i++) {
            long hash1 = (hash << 5) & 0x00000000ffffffffL;

            hash = ((hash1 + hash) & 0x00000000ffffffffL) + topicBase64.charAt(i);

            hash = hash & 0x00000000ffffffffL;
        }

        return hash;
    }

    private void leaveChannelIfNeed() {
        if (!isChannelTimeOut()) {
            Log.i(TAG, "leaveChannelIfNeed return");
            return;
        }

        if (mJoinedChannel.size() > 0) {
            Log.i(TAG, "Leave all channel for offline");
            synchronized (mJoinedChannel) {
                HashSet<com.hummer.im.model.Channel> cloneChannel
                        = (HashSet<com.hummer.im.model.Channel>) mJoinedChannel.clone();

                List<User> users = new ArrayList<>();
                users.add(HMR.getMe());
                for (com.hummer.im.model.Channel channel : cloneChannel) {
                    mJoinedChannel.remove(channel);
                    notifyUsersLeaved(channel, users);
                    leaveChannel(channel, null);
                }
            }
        }
    }

    private boolean isChannelTimeOut() {
        // YYServiceSDK 的在线是通过 YYServiceSDK 的心跳来进行维持的
        // 退到后台后或者断网的时候, Service 那边的心跳停止发送
        // 频道的后台会读取这个在线状态，当超过75秒（跟后台确认的时间）没有发送心跳，则会认为这个用户离线
        // 频道台会踢这个用户出频道，所以当失效时间段小于60秒的时候，则无需处理
        long now = System.currentTimeMillis();

        return mInitialTimeInterval > kReferenceValue && (now - mInitialTimeInterval) >= kRejoinConditionGap;
    }

    private void recordInitialTimeIntervalIfNeeded() {
        if (mInitialTimeInterval < kReferenceValue) {
            mInitialTimeInterval = System.currentTimeMillis();
        }
    }

    private void resetInitialTimeInterval() {
        mInitialTimeInterval = 0L;
    }

    private void prepareChannel() {
        if (mChannelStateListener != null) {
            Log.e(TAG, Trace.once().method("prepareChannel")
                    .info("mChannelStateListener exception", null));
            return;
        }

        mChannelStateListener = new Channel.StateChangedListener() {
            @Override
            public void onPreChannelConnected() {

            }

            @Override
            public void onChannelConnected() {
                HMRContext.work.async("ChannelService::onChannelConnected", new Runnable() {
                    @Override
                    public void run() {
                        Log.w(TAG, Trace.once().method("onChannelConnected"));
                        leaveChannelIfNeed();
                        resetInitialTimeInterval();
                    }
                });
            }

            @Override
            public void onChannelDisconnected() {
                HMRContext.work.async("ChannelService::onChannelDisconnected", new Runnable() {
                    @Override
                    public void run() {
                        Log.w(TAG, Trace.once().method("onChannelDisconnected"));
                        recordInitialTimeIntervalIfNeeded();
                    }
                });
            }
        };
        HMR.getService(Channel.class).addStateListener(mChannelStateListener);
    }

    private void teardownChannel() {
        HMR.getService(Channel.class).removeStateListener(mChannelStateListener);

        if (mJoinedChannel.size() > 0) {
            synchronized (mJoinedChannel) {
                for (com.hummer.im.model.Channel channel : mJoinedChannel) {
                    HMR.getService(Channel.class)
                            .run(new RPCLeaveChannel(channel.getName(), null, null));

                    final String topic = getTopic(channel.getName());
                    HMR.getService(MQService.class)
                            .removeSource(new Source(new Source.Shared(djbHash(topic), topic)));
                }
            }
            mJoinedChannel.clear();
        }
        mChannelStateListener = null;
    }

    // Mark - ChannelService interface

    @Override
    public void addListener(Listener listener) {
        synchronized (mListeners) {
            mListeners.add(listener);
        }

        Log.i(TAG, Trace.once()
                .method("addListener")
                .info("name", listener.getClass().getSimpleName())
                .info("size", mListeners.size()));
    }

    @Override
    public void removeListener(Listener listener) {
        synchronized (mListeners) {
            mListeners.remove(listener);
        }

        Log.i(TAG, Trace.once()
                .method("removeListener")
                .info("name", listener.getClass().getSimpleName())
                .info("size", mListeners.size()));
    }

    @Override
    public void joinChannel(@NonNull final com.hummer.im.model.Channel channel,
                            @Nullable final Map<String, String> appExtras,
                            @Nullable final HMR.Completion completion) {

        Log.i(TAG, Trace.once().method("joinChannel").info("channel", channel));

        RichCompletion rCompletion = new RichCompletion(completion, "ChannelService::joinChannel");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode"));
            return;
        }

        HMR.getService(Channel.class)
                .run(new RPCJoinChannel(channel.getName(),
                        appExtras,
                        new RichCompletionArg<>(new HMR.CompletionArg<Map<String, String>>() {
                            @Override
                            public void onSuccess(Map<String, String> appExtras) {
                                Log.i(TAG, Trace.once()
                                        .method("joinChannel Done")
                                        .info("channel", channel.getName())
                                        .info("extras", appExtras));

                                String duration = appExtras.get(OFFLINE_DURATION_KEY);
                                if (TextUtils.isNotBlank(duration)) {
                                    kRejoinConditionGap = (Long.valueOf(duration) - 5) * 1000;
                                }

                                HMRContext.work.async("", new Runnable() {
                                    @Override
                                    public void run() {
                                        final String topic = getTopic(channel.getName());

                                        HMR.getService(MQService.class).addSource(
                                                new Source(
                                                        new Source.Shared(
                                                                djbHash(topic),
                                                                null,
                                                                topic,
                                                                MQService.FetchStrategy.IgnoreBefore)));
                                        mJoinedChannel.add(channel);
                                    }
                                });
                                if (completion != null) {
                                    completion.onSuccess();
                                }
                            }

                            @Override
                            public void onFailed(Error err) {
                                Log.i(TAG, Trace.once()
                                        .method("joinChannel fail")
                                        .info("channel", channel));
                                if (completion != null) {
                                    completion.onFailed(err);
                                }
                            }
                        }, "")));
    }

    @Override
    public void leaveChannel(@NonNull final com.hummer.im.model.Channel channel,
                             @Nullable final HMR.Completion completion) {
        Log.i(TAG, Trace.once().method("leaveChannel").info("channel", channel));

        RichCompletion rCompletion = new RichCompletion(completion, "ChannelService::leaveChannel");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode"));
            return;
        }

        HMR.getService(Channel.class).run(new RPCLeaveChannel(channel.getName(),
                null,
                new RichCompletion(new HMR.Completion() {
                    @Override
                    public void onSuccess() {
                        Log.i(TAG, Trace.once()
                                .method("leaveChannel Done")
                                .info("channel", channel.getName()));

                        HMRContext.work.async("", new Runnable() {
                            @Override
                            public void run() {
                                final String topic = getTopic(channel.getName());

                                HMR.getService(MQService.class)
                                        .removeSource(new Source(new Source.Shared(djbHash(topic), topic)));
                                mJoinedChannel.remove(channel);
                            }
                        });

                        if (completion != null) {
                            completion.onSuccess();
                        }
                    }

                    @Override
                    public void onFailed(Error err) {
                        Log.i(TAG, Trace.once()
                                .method("leaveChannel fail")
                                .info("channel", channel.getName())
                                .info("error", err.toString()));

                        if (completion != null) {
                            completion.onFailed(err);
                        }
                    }
                }, "")));
    }

    @Override
    public void fetchMembers(@NonNull com.hummer.im.model.Channel channel,
                             @Nullable HMR.CompletionArg<List<User>> completion) {
        RichCompletionArg<List<User>> rCompletion
                = new RichCompletionArg<>(completion, "ChannelService::fetchMembers");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode"));
            return;
        }

        Log.i(TAG, Trace.once().method("fetchMembers")
                .info("channel", channel));

        HMR.getService(Channel.class).run(new RPCFetchMembers(channel.getName(), rCompletion));
    }

    @Override
    public void fetchMembers(@NonNull com.hummer.im.model.Channel channel,
                             @NonNull String key,
                             @NonNull String value,
                             @Nullable HMR.CompletionArg<List<User>> completion) {

        RichCompletionArg<List<User>> rCompletion
                = new RichCompletionArg<>(completion, "ChannelService::fetchMembersByAttr");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode"));
            return;
        }

        Log.i(TAG, Trace.once().method("fetchMembersByAttr")
                .info("channel", channel)
                .info("key", key)
                .info("value", value));

        HMR.getService(Channel.class).run(
                new RPCFetchMembersByAttr(channel.getName(), key, value, rCompletion));

    }

    @Override
    public void setUserInfo(@NonNull com.hummer.im.model.Channel channel,
                            @NonNull Map<String, String> infoMap,
                            @Nullable HMR.Completion completion) {
        RichCompletion rCompletion
                = new RichCompletion(completion, "ChannelService::setUserInfo");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode"));
            return;
        }

        Log.i(TAG, Trace.once().method("setUserInfo")
                .info("channel", channel)
                .info("infoMap", infoMap));

        HMR.getService(Channel.class).run(new RPCSetUserInfo(channel.getName(), infoMap, rCompletion));
    }

    @Override
    public void removeUserInfoByKeys(@NonNull com.hummer.im.model.Channel channel,
                                     @NonNull Set<String> keys,
                                     @Nullable HMR.Completion completion) {
        RichCompletion rCompletion
                = new RichCompletion(completion, "ChannelService::removeUserInfoByKeys");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode"));
            return;
        }

        Log.i(TAG, Trace.once().method("deleteUserAttrByKeys")
                .info("channel", channel)
                .info("keys", keys));

        HMR.getService(Channel.class).run(new RPCDeleteUserInfo(channel.getName(), keys, rCompletion));
    }

    @Override
    public void send(@NonNull com.hummer.im.model.Channel channel,
                     @NonNull Message message,
                     @NonNull SendingOptions sendingOptions,
                     @NonNull HMR.Completion completion) {

        RichCompletion rCompletion = new RichCompletion(completion, "");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode"));
            return;
        }

        HMR.getService(Channel.class).run(new RPCSendBroadcast(channel.getName(),
                message.getType(),
                new String(message.getData()),
                !sendingOptions.isUnreliable(),
                rCompletion));
    }

    private String channelNameFromTopic(String topic) {
        if (topic == null) {
            // 防止异常，传入null字符串
            Log.e(TAG, Trace.once()
                    .method("channelNameFromTopic")
                    .info("topic", "null"));
            return null;
        }

        String[] parts = topic.split("/");
        if (parts.length < 2) {
            Log.e(TAG, Trace.once()
                    .method("channelNameFromTopic")
                    .info("topic", "length exception"));
            return null;
        }

        return parts[1];
    }

    // Mark - Service interface

    @Override
    public Class[] staticDependencies() {
        return new Class[0];
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{Channel.class};
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return new Class[0];
    }

    @Override
    public void initService() {
        registerChatServiceExtensions();
    }

    @Override
    public void openService(@NonNull RichCompletion completion) {
        prepareChannel();
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void closeService() {
        teardownChannel();
        resetInitialTimeInterval();
    }
}
