package com.hummer.im.channel;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im.HMR;
import com.hummer.im.model.Channel;
import com.hummer.im.model.signal.Message;
import com.hummer.im.model.signal.SendingOptions;
import com.hummer.im.model.id.User;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * chanel service
 */
public interface ChannelService {

    /**
     * 进入频道的接口
     *
     * @param channel    频道: 频道名称只能由 '[A,Z],[a,z],[0,9],-,_' 字符组成
     * @param appExtras  加入频道时的扩展信息
     * @param completion 进入完成的异步回调
     */
    void joinChannel(@NonNull Channel channel,
                     @Nullable Map<String, String> appExtras,
                     @Nullable HMR.Completion completion);

    /**
     * 退出频道的接口
     *
     * @param channel    频道: 频道名称只能由 '[A,Z],[a,z],[0,9],-,_' 字符组成
     * @param completion 退出完成的异步回调
     */
    void leaveChannel(@NonNull Channel channel,
                      @Nullable HMR.Completion completion);

    /**
     * 获取频道成员列表
     *
     * @param channel    频道
     * @param completion 回调
     */
    void fetchMembers(@NonNull Channel channel,
                      @Nullable HMR.CompletionArg<List<User>> completion);

    /**
     * 获取具有相应属性值的频道成员列表
     *
     * @param channel    频道
     * @param key        属性键
     * @param value      属性值
     * @param completion 回调
     */
    void fetchMembers(@NonNull Channel channel,
                      @NonNull String key,
                      @NonNull String value,
                      @Nullable final HMR.CompletionArg<List<User>> completion);

    /**
     * 设置用户在当前频道的信息
     *
     * @param channel    频道
     * @param infoMap    信息
     * @param completion 回调
     */
    void setUserInfo(@NonNull final Channel channel,
                     @NonNull final Map<String, String> infoMap,
                     @Nullable final HMR.Completion completion);

    /**
     * 删除用户在当前频道的某些信息
     *
     * @param channel    频道
     * @param keys       属性的key集合
     * @param completion 回调
     */
    void removeUserInfoByKeys(@NonNull final Channel channel,
                              @NonNull final Set<String> keys,
                              @Nullable final HMR.Completion completion);

    /**
     * 发送组消息
     *
     * @param channel    频道
     * @param message    消息
     * @param options    发送消息所需要的配置信息
     * @param completion 回调
     */
    void send(@NonNull Channel channel,
              @NonNull Message message,
              @NonNull SendingOptions options,
              @NonNull HMR.Completion completion);

    interface Listener {
        /**
         * 当收到某个成员进入频道时， 会收到该事件的通知回调
         *
         * @param channel 频道的名称
         * @param users   进入频道的用户列表
         */
        void onUsersJoined(@NonNull Channel channel, @NonNull List<User> users);

        /**
         * 当收到某个成员退出频道时， 会收到该事件的通知回调
         *
         * @param channel 频道
         * @param users   退出频道的用户列表
         */
        void onUsersLeaved(@NonNull Channel channel, @NonNull List<User> users);

        /**
         * 频道成员信息设置回调通知
         *
         * @param channel 频道
         * @param user    成员
         * @param infoMap 信息
         */
        void onUserInfoSet(@NonNull Channel channel,
                           @NonNull User user,
                           @NonNull Map<String, String> infoMap);

        /**
         * 频道成员信息删除回调通知
         *
         * @param channel 频道
         * @param user    成员
         * @param keys    信息
         */
        void onUserInfoRemove(@NonNull Channel channel,
                              @NonNull User user,
                              @NonNull Set<String> keys);

        /**
         * 当有一个用户往频道内发一条信令消息时，频道内所有的成员都会受到该事件的通知回调
         *
         * @param channel  频道的名称
         * @param message  收到的信令消息
         * @param fromUser 发送者的标识
         */
        void onReceivedMessage(Channel channel, Message message, User fromUser);
    }

    /**
     * 添加Channel成员的监听器
     *
     * @param listener 需要添加的监听器
     */
    void addListener(Listener listener);

    /**
     * 移除Channel成员监听器
     *
     * @param listener 需要移除的监听器
     */
    void removeListener(Listener listener);
}
